﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Threading;

using NWCore.DataModel;
using NWCore.Utility;

namespace NWCore
{
    #region Log level enumerator and message logging handler delegation

    /// <summary>
    /// Enumerator for log levels.
    /// </summary>
    public enum LogLevels
    {
        Debug         = 0x00000001,
        Information   = 0x00000002,
        Warning       = 0x00000003,
        Error         = 0x00000004,
        Fatal         = 0x00000005,

        OpenLogFile   = 0x00000006,
        Console       = 0x00000007,

        /// <summary>
        /// This flag indicates the log message is for mapping ( to a string in the string table...etc. ),
        /// instead of being used directly as a format string.
        /// </summary>
        MsgForMapping = 0x0F000000
    }


    /// <summary>
    /// Delegation for outputting log messages.
    /// </summary>
    /// <param name="level">Log level.</param>
    /// <param name="msg">The message to output.</param>
    /// <param name="args">Message formatting arguments.</param>
    public delegate void LogHandler( LogLevels level,
                                     string msg,
                                     params object[] args );

    #endregion

    /// <summary>
    /// Static class for the kernel for NWCore.
    /// </summary>
    public static class NWKernel
    {
        #region Initialize

        /// <summary>
        /// Initialize the kernel.
        /// </summary>
        public static bool Initialize()
        {
            s_syncContext = SynchronizationContext.Current;

            // CRC32 helper
            s_CRC32Helper = new CRC32();
            s_CRC32Helper.Initialize();

            NWKernel.ShouldOverrideTextureCopyFlags = false;
            NWKernel.OverriddenTextureCopyFlag      = true;

            return true;
        }

        #endregion

        #region Properties : Managers

        /// <summary>Get the CRC32 helper.</summary>
        public static CRC32 CRC32Helper
        {
            get { return s_CRC32Helper; }
        }

        #endregion

        #region Other properties

        /// <summary>
        /// Get or set the flag indicating if should use
        /// NWKernel.OverriddenTextureCopyFlag to determine
        /// if should copy the texture resources.
        /// </summary>
        /// <remarks>
        /// It's dangerous to modify this flag directly.
        /// Anyone wants to modify this flag should use
        /// NWCoreTextureCopyFlagStack instead.
        /// (see after class NWKernel's definition)
        /// </remarks>
        public static bool ShouldOverrideTextureCopyFlags
        {
            get;
            set;
        }


        /// <summary>
        /// Get or set the flag indicating if should copy the texture resources
        /// or simply copy the reference to the resources.
        /// </summary>
        /// <remarks>
        /// It's dangerous to modify this flag directly.
        /// Anyone wants to modify this flag should use
        /// NWCoreTextureCopyFlagStack instead.
        /// (see after class NWKernel's definition)
        /// </remarks>
        public static bool OverriddenTextureCopyFlag
        {
            get;
            set;
        }


        /// <summary>Get the synchronization context for the main thread.</summary>
        public static SynchronizationContext SyncContext
        {
            get { return s_syncContext; }
        }

        #endregion

        #region Default values

        /// <summary>The default value for EmitterData.DrawPath.</summary>
        public static string DefaultDrawPath { get; set; }


        /// <summary>The default value for EmitterData.DrawPathID.</summary>
        public static int DefaultDrawPathID { get; set; }


        /// <summary>The default value for EmitterData.UserShaderCompileDef1.</summary>
        public static string DefaultUserShaderCompileDef1 { get; set; }


        /// <summary>The default value for EmitterData.UserShaderCompileDef2.</summary>
        public static string DefaultUserShaderCompileDef2 { get; set; }

        #endregion

        #region Access default values

        /// <summary>
        /// Set default user shader parameters.
        /// </summary>
        /// <param name="defaultParams">The default user shader parameters.</param>
        public static void SetDefaultUserShaderParams( DefaultUserShaderParameter[] defaultParams )
        {
            s_defaultUserShaderParams = defaultParams;
        }


        /// <summary>
        /// Get the default parameters for the specified user shader.
        /// </summary>
        /// <param name="iIndex">The index to the user shader.</param>
        /// <returns>The default parameters.</returns>
        public static DefaultUserShaderParameter GetDefaultUserShaderParams( int iIndex )
        {
            if ( s_defaultUserShaderParams==null )
                return new DefaultUserShaderParameter();

            if ( iIndex<0 || iIndex>=s_defaultUserShaderParams.Length )
                return new DefaultUserShaderParameter();

            return s_defaultUserShaderParams[iIndex];
        }


        /// <summary>
        /// Set default emitter user data.
        /// </summary>
        /// <param name="defaultData">The default emitter user data.</param>
        public static void SetDefaultEmitterUserData( DefaultEmitterUserData[] defaultData )
        {
            s_defaultEmitterUserData = defaultData;
        }


        /// <summary>
        /// Get the default values for the specified emitter user data.
        /// </summary>
        /// <param name="iIndex">The index to the user data.</param>
        /// <returns>The default values.</returns>
        public static DefaultEmitterUserData GetDefaultEmitterUserData( int iIndex )
        {
            if ( s_defaultEmitterUserData==null )
                return new DefaultEmitterUserData();

            iIndex += 1; // The index is supposed to be from -1 ~ 7
            if ( iIndex<0 || iIndex>=s_defaultEmitterUserData.Length )
                return new DefaultEmitterUserData();

            return s_defaultEmitterUserData[iIndex];
        }

        #endregion

        #region User shader codes

        /// <summary>
        /// Set user vertex shader paths for the specified custom shader definition.
        /// </summary>
        /// <remarks>Set iIndex=-1 to setup general user vertex shader paths.</remarks>
        /// <param name="iIndex">The index to the custom shader definition.</param>
        /// <param name="shaderPaths">The shader paths.</param>
        public static void SetUserVertexShaderPaths( int iIndex,
                                                     List<string> shaderPaths )
        {
            System.Diagnostics.Debug.Assert( iIndex>=-1 && iIndex<s_userVtxShaderCodes.Length );

            if ( iIndex==-1 )
                s_generalUserVtxShaderCode = string.Empty;
            else
                s_userVtxShaderCodes[iIndex] = string.Empty;
            if ( shaderPaths==null )
                return;

            bool bAppendNewLine = false;
            foreach ( string path in shaderPaths )
            {
                try
                {
                    if ( iIndex==-1 )
                    {
                        s_generalUserVtxShaderCode +=
                            (bAppendNewLine==true ? "\n" : string.Empty) + System.IO.File.ReadAllText( path );
                    }
                    else
                    {
                        s_userVtxShaderCodes[iIndex] +=
                            (bAppendNewLine==true ? "\n" : string.Empty) + System.IO.File.ReadAllText( path );
                    }

                    bAppendNewLine = true;
                }
                catch
                {
                    // Failed opening the shader, just skip it.
                }
            }
        }


        /// <summary>
        /// Set user fragment shader paths for the specified custom shader definition.
        /// </summary>
        /// <remarks>Set iIndex=-1 to setup general user vertex shader paths.</remarks>
        /// <param name="iIndex">The index to the custom shader definition.</param>
        /// <param name="shaderPaths">The shader paths.</param>
        public static void SetUserFragmentShaderPaths( int iIndex,
                                                     List<string> shaderPaths )
        {
            System.Diagnostics.Debug.Assert( iIndex>=-1 && iIndex<s_userFrgShaderCodes.Length );

            if ( iIndex==-1 )
                s_generalUserFrgShaderCode = string.Empty;
            else
                s_userFrgShaderCodes[iIndex] = string.Empty;
            if ( shaderPaths==null )
                return;

            bool bAppendNewLine = false;
            foreach ( string path in shaderPaths )
            {
                try
                {
                    if ( iIndex==-1 )
                    {
                        s_generalUserFrgShaderCode +=
                            (bAppendNewLine==true ? "\n" : string.Empty) + System.IO.File.ReadAllText( path );
                    }
                    else
                    {
                        s_userFrgShaderCodes[iIndex] +=
                            (bAppendNewLine==true ? "\n" : string.Empty) + System.IO.File.ReadAllText( path );
                    }

                    bAppendNewLine = true;
                }
                catch
                {
                    // Failed opening the shader, just skip it.
                }
            }
        }


        /// <summary>
        /// Get the user vertex shader code of the specified custom shader definition.
        /// </summary>
        /// <param name="iIndex">The index to the custom shader definition.</param>
        /// <returns>The user vertex shader code.</returns>
        public static string GetUserVertexShaderCode( int iIndex )
        {
            if ( iIndex==-1 )
                return s_generalUserVtxShaderCode;
            else if ( iIndex<-1 || iIndex>=s_userVtxShaderCodes.Length )
                return null;

            if ( string.IsNullOrEmpty(s_generalUserVtxShaderCode)==false &&
                 string.IsNullOrEmpty(s_userVtxShaderCodes[iIndex])==false )
            {
                return s_generalUserVtxShaderCode + "\n" + s_userVtxShaderCodes[iIndex];
            }
            else
            {
                return s_userVtxShaderCodes[iIndex];
            }
        }

        /// <summary>
        /// Get the user fragment shader code of the specified custom shader definition.
        /// </summary>
        /// <param name="iIndex">The index to the custom shader definition.</param>
        /// <returns>The user fragment shader code.</returns>
        public static string GetUserFragmentShaderCode( int iIndex )
        {
            if ( iIndex==-1 )
                return s_generalUserFrgShaderCode;
            else if ( iIndex<-1 || iIndex>=s_userFrgShaderCodes.Length )
                return null;

            if ( string.IsNullOrEmpty(s_generalUserFrgShaderCode)==false &&
                 string.IsNullOrEmpty(s_userFrgShaderCodes[iIndex])==false )
            {
                return s_generalUserFrgShaderCode + "\n" + s_userFrgShaderCodes[iIndex];
            }
            else
            {
                return s_userFrgShaderCodes[iIndex];
            }
        }

        #endregion

        #region Overwritten shader codes

        /// <summary>
        /// Add an shader code that overwrites the specified shader project.
        /// </summary>
        /// <param name="shaderProjPath">The shader project file path.</param>
        /// <param name="shaderCode">The shader code.</param>
        public static void AddOverwrittenShaderCode( string shaderProjPath,
                                                     string shaderCode )
        {
            if ( string.IsNullOrEmpty(shaderProjPath)==true )
                return;

            uint iPathCRC = CRC32Helper.ComputeCRC32Str(shaderProjPath.ToLower());
            if ( s_overwrittenShaderCodes.ContainsKey(iPathCRC)==true )
            {
                s_overwrittenShaderCodes[iPathCRC] = shaderCode;
            }
            else
            {
                s_overwrittenShaderCodes.Add( iPathCRC, shaderCode );
            }
        }


        /// <summary>
        /// Get the overwritten shader code of the specified shader project.
        /// </summary>
        /// <param name="shaderProjPath">The shader project file path.</param>
        /// <returns>The shader code or null if the shader project has no overwritten code added.</returns>
        public static string GetOverwrittenShaderCode( string shaderProjPath )
        {
            if ( string.IsNullOrEmpty(shaderProjPath)==true )
                return null;

            uint iPathCRC = CRC32Helper.ComputeCRC32Str(shaderProjPath.ToLower());
            if ( s_overwrittenShaderCodes.ContainsKey(iPathCRC)==false )
                return null;

            return s_overwrittenShaderCodes[iPathCRC];
        }

        #endregion

        #region Log message related

        /// <summary>
        /// Output log message.
        /// </summary>
        /// <param name="level">The log level.</param>
        /// <param name="msg">The message to log out.</param>
        public static void OutputLog( LogLevels level,
                                      string msg )
        {
            if ( s_logHandler!=null )
            {
                s_logHandler( level, msg );
            }
        }


        /// <summary>
        /// Output log message with formatting string.
        /// </summary>
        /// <param name="level">The log level.</param>
        /// <param name="format">The message formatting string.</param>
        /// <param name="args">The arguments for the formatting string.</param>
        public static void OutputLog( LogLevels level,
                                      string format,
                                      params object[] args )
        {
            if ( s_logHandler!=null )
            {
                s_logHandler( level, string.Format( format, args ) );
            }
        }


        /// <summary>
        /// Register a handler when log messages need to be outputted.
        /// </summary>
        /// <param name="handler">The handler.</param>
        public static void RegisterLogHandler( LogHandler handler )
        {
            s_logHandler = handler;
        }


        /// <summary>
        /// Output memory usage and return as a string.
        /// </summary>
        /// <returns>The formatted memory usage text.</returns>
        public static string OutputMemoryUsage()
        {
            ulong iPhysicalMemSize;
            ulong iVirtualMemSize;
            ulong iAvailPhysicalMemSize;
            ulong iAvailVirtualMemSize;
            ulong iProcPhyMemSize;
            ulong iProcVirMemSize;

            NWKernel.GetMemoryUsageInfo( out iPhysicalMemSize,
                                         out iVirtualMemSize,
                                         out iAvailPhysicalMemSize,
                                         out iAvailVirtualMemSize,
                                         out iProcPhyMemSize,
                                         out iProcVirMemSize );

            string msg = "Memory usage:\r\n";
            msg += "    Total physical memory      : " + NWKernel.FormatSizeStr(iPhysicalMemSize) + "\r\n";
            msg += "    Available physical memory  : " + NWKernel.FormatSizeStr(iAvailPhysicalMemSize) + "\r\n";
            msg += "    Process physical mem usage : " + NWKernel.FormatSizeStr(iProcPhyMemSize) + "\r\n";
            msg += "    Total virtual memory       : " + NWKernel.FormatSizeStr(iVirtualMemSize) + "\r\n";
            msg += "    Available virtual memory   : " + NWKernel.FormatSizeStr(iAvailVirtualMemSize) + "\r\n";
            msg += "    Process virtual mem usage  : " + NWKernel.FormatSizeStr(iProcVirMemSize) + "\r\n";

            return msg;
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Native Win32 method for getting memory status.
        /// </summary>
        /// <param name="lpBuffer">The result.</param>
        /// <returns>True on success.</returns>
        [return: System.Runtime.InteropServices.MarshalAs( System.Runtime.InteropServices.UnmanagedType.Bool )]
        [System.Runtime.InteropServices.DllImport( "kernel32.dll" )]
        static extern bool GlobalMemoryStatusEx( ref NWCore.Win32.MEMORYSTATUSEX lpBuffer );

        /// <summary>
        /// Get current system memory usage information.
        /// </summary>
        /// <param name="iPhysicalMemSize">Total system physical memory size.</param>
        /// <param name="iVirtualMemSize">Total system virtual memory size.</param>
        /// <param name="iAvailPhysicalMemSize">Available physical memory size.</param>
        /// <param name="iAvailVirtualMemSize">Available virtual memory size.</param>
        /// <param name="iProcPhyMemSize">Total physical memory used by this process.</param>
        /// <param name="iProcVirMemSize">Total virtual memory used by this process.</param>
        /// <returns>True on success.</returns>
        public static bool GetMemoryUsageInfo( out ulong iPhysicalMemSize,
                                               out ulong iVirtualMemSize,
                                               out ulong iAvailPhysicalMemSize,
                                               out ulong iAvailVirtualMemSize,
                                               out ulong iProcPhyMemSize,
                                               out ulong iProcVirMemSize )
        {
            iPhysicalMemSize      = 0;
            iAvailPhysicalMemSize = 0;
            iVirtualMemSize       = 0;
            iAvailVirtualMemSize  = 0;
            iProcPhyMemSize       = 0;
            iProcVirMemSize       = 0;

            // Get the current process for its memory usage.
            System.Diagnostics.Process proc =
                System.Diagnostics.Process.GetCurrentProcess();
            if ( proc==null )
                return false;

            // Process memory usage.
            iProcPhyMemSize = (ulong)proc.WorkingSet64;
            iProcVirMemSize = (ulong)proc.VirtualMemorySize64;

            // Create the structure for getting the system memory status.
            NWCore.Win32.MEMORYSTATUSEX result = new NWCore.Win32.MEMORYSTATUSEX();
            result.Init();

            // Get system memory status.
            GlobalMemoryStatusEx( ref result );

            iPhysicalMemSize      = result.ullTotalPhys;
            iAvailPhysicalMemSize = result.ullAvailPhys;
            iVirtualMemSize       = result.ullTotalVirtual;
            iAvailVirtualMemSize  = result.ullAvailVirtual;

            return true;
        }


        /// <summary>
        /// Format the given size in bytes into a string.
        /// </summary>
        /// <param name="size">The size in bytes.</param>
        /// <returns>The formatted string.</returns>
        public static string FormatSizeStr( ulong size )
        {
            double fFileSize = (double)size;

            const double fKB = 1024.0;
            const double fMB = 1048576.0;
            const double fGB = 1073741824.0;
            const double fTB = 1099511627776.0;

            if ( fFileSize<fKB )
            {
                return fFileSize.ToString( "N0" ) + " Bytes";
            }
            else if ( fFileSize<fGB )
            {
                return (fFileSize / fKB).ToString( "N1" )  + " KB";
            }
            else if ( fFileSize<fTB )
            {
                return (fFileSize / fMB).ToString( "N1" )  + " MB";
            }

            return (fFileSize / fGB).ToString( "N1" )  + " GB";
        }

        #endregion

        #region Static variables

        private static CRC32      s_CRC32Helper = null;
        private static LogHandler s_logHandler  = null;

        private static DefaultUserShaderParameter[] s_defaultUserShaderParams  = null;
        private static DefaultEmitterUserData[]     s_defaultEmitterUserData   = null;

        private static SynchronizationContext       s_syncContext              = null;

        private static string[]                     s_userVtxShaderCodes       = new string[8];
        private static string[]                     s_userFrgShaderCodes       = new string[8];
        private static string                       s_generalUserVtxShaderCode = string.Empty;
        private static string                       s_generalUserFrgShaderCode = string.Empty;

        private static Dictionary<uint, string>     s_overwrittenShaderCodes   = new Dictionary<uint, string>();

        #endregion
    }

    #region Texture copy flag stack

    /// <summary>
    /// Texture copy flag stack.
    /// Usage :
    ///     using ( NWCoreTextureCopyFlagStack stack = new NWCoreTextureCopyFlagStack(bCopy) )
    ///     {
    ///         // Do anything, eq. someEmitterData.Set( sourceEmitterData );
    ///         // The texture copy flags will be overridden and reset after
    ///         // leave this "using" block.
    ///     }
    /// </summary>
    public class NWCoreTextureCopyFlagStack : IDisposable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="bCopy">
        /// Flag indicating whether should copy the texture resources or not.
        /// </param>
        public NWCoreTextureCopyFlagStack( bool bCopy )
        {
            bShouldOverrideTextureCopyFlags = NWKernel.ShouldOverrideTextureCopyFlags;
            bSavedTextureCopyFlag           = NWKernel.OverriddenTextureCopyFlag;

            NWKernel.ShouldOverrideTextureCopyFlags = true;
            NWKernel.OverriddenTextureCopyFlag      = bCopy;
        }


        /// <summary>
        /// Dispose, restore the flags.
        /// </summary>
        public void Dispose()
        {
            NWKernel.ShouldOverrideTextureCopyFlags = bShouldOverrideTextureCopyFlags;
            NWKernel.OverriddenTextureCopyFlag      = bSavedTextureCopyFlag;
        }


        private bool bShouldOverrideTextureCopyFlags = false;
        private bool bSavedTextureCopyFlag           = false;
    }

    #endregion
}
