﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

namespace App.Utility
{
    /// <summary>
    /// 範囲クラス。
    /// </summary>
    public sealed class Range
    {
        // 最小値
        private int _minimum = 0;
        // 最大値
        private int _maximum = 0;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Range() {}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Range(int minimum, int maximum)
        {
            _minimum = minimum;
            _maximum = maximum;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Range(Range range)
        {
            _minimum = range._minimum;
            _maximum = range._maximum;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        public int Minimum
        {
            get { return _minimum;  }
            set { _minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        public int Maximum
        {
            get { return _maximum;  }
            set { _maximum = value; }
        }

        /// <summary>
        /// 範囲内かどうか。
        /// </summary>
        public bool Contains(int value)
        {
            return Contains(value, _minimum, _maximum);
        }

        /// <summary>
        /// 範囲内かどうか（静的メソッド版）。
        /// </summary>
        public static bool Contains(int value, int minimum, int maximum)
        {
            return minimum <= value && value <= maximum;
        }

        /// <summary>
        /// 範囲内に丸め込む。
        /// </summary>
        public void Truncate(ref int value)
        {
            if      (value < _minimum) { value = _minimum; }
            else if (value > _maximum) { value = _maximum; }
        }

        /// <summary>
        /// 範囲内に丸め込む。
        /// </summary>
        public int Truncate(int value)
        {
            if      (value < _minimum) { return _minimum; }
            else if (value > _maximum) { return _maximum; }
            return value;
        }

        /// <summary>
        /// 範囲の拡張。
        /// </summary>
        public void Inflate(int value)
        {
            if (value < _minimum) _minimum = value;
            else if (value > _maximum) _maximum = value;
        }

        /// <summary>
        /// 範囲の長さ。
        /// </summary>
        public int Length
        {
            get { return _maximum - _minimum; }
        }
    }

    /// <summary>
    /// 小数値範囲クラス。
    /// </summary>
    public sealed class RangeF
    {
        // 最小値
        private float _minimum = 0;
        // 最大値
        private float _maximum = 0;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public RangeF() {}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public RangeF(float minimum, float maximum)
        {
            _minimum = minimum;
            _maximum = maximum;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public RangeF(RangeF range)
        {
            _minimum = range._minimum;
            _maximum = range._maximum;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        public float Minimum
        {
            get { return _minimum;  }
            set { _minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        public float Maximum
        {
            get { return _maximum;  }
            set { _maximum = value; }
        }

        /// <summary>
        /// 範囲内かどうか。
        /// </summary>
        public bool Contains(float value)
        {
            return Contains(value, _minimum, _maximum);
        }

        /// <summary>
        /// 範囲内かどうか（静的メソッド版）。
        /// </summary>
        public static bool Contains(float value, float minimum, float maximum)
        {
            return minimum <= value && value <= maximum;
        }

        /// <summary>
        /// 範囲内に丸め込む。
        /// </summary>
        public void Truncate(ref float value)
        {
            if      (value < _minimum) { value = _minimum; }
            else if (value > _maximum) { value = _maximum; }
        }

        /// <summary>
        /// 範囲内に丸め込む。
        /// </summary>
        public float Truncate(float value)
        {
            if      (value < _minimum) { return _minimum; }
            else if (value > _maximum) { return _maximum; }
            return value;
        }

        /// <summary>
        /// 範囲の拡張。
        /// </summary>
        public void Inflate(float value)
        {
            if (value < _minimum) _minimum = value;
            else if (value > _maximum) _maximum = value;
        }

        /// <summary>
        /// 範囲の長さ。
        /// </summary>
        public float Length
        {
            get { return _maximum - _minimum; }
        }
    }
}
