﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using App.Controls;
using App.Data;
using NW4F.DataModel;
using NW4F.Viewer;

namespace App.IO
{
    /// <summary>
    /// テクスチャオープナ。
    /// </summary>
    public class TextureOpener : DocumentOpener
    {
        /// <summary>
        /// ファイル検索
        /// </summary>
        public static string SearchFile(string basePath, string fileName)
        {
            // RTEXを探す
            {
                // テクスチャフォルダ
                string filePath = basePath + DocumentConstants.TextureDirectory + fileName;
                if (File.Exists(filePath)) { return filePath; }

                // カレントフォルダ
                filePath = basePath + fileName;
                if (File.Exists(filePath)) { return filePath; }
            }
//TODO:未対応
#if false
            // TGAを探す
            {
                string fileName = name + DocumentConstants.DotTga;

                // テクスチャフォルダ
                string filePath = basePath + DocumentConstants.TextureDirectory + fileName;
                if (File.Exists(filePath))
                {
                    if (TgaReader.IsAppendInformation(filePath)) { return filePath; }
                }

                // カレントフォルダ
                filePath = basePath + fileName;
                if (File.Exists(filePath))
                {
                    if (TgaReader.IsAppendInformation(filePath)) { return filePath; }
                }
            }
#endif
            return null;
        }

        /// <summary>
        /// テクスチャ名前リストがロードされているか
        /// </summary>
        public static List<string> LoadsNameList(List<string> nameList)
        {
            List<string> notFoundNames = new List<string>();
            foreach (string name in nameList)
            {
                if (TextureManager.Search(name) == null) { notFoundNames.Add(name); }
            }
            return notFoundNames;
        }

        /// <summary>
        /// テクスチャ名前リストを探す
        /// </summary>
        public static List<string> FindNameList(string basePath, List<string> nameList)
        {
            List<string> result = new List<string>();
            foreach (string name in nameList)
            {
                // すでにロード済みなら無視
                if (TextureManager.Search(name) != null) { continue; }
                if (SearchFile(basePath, name) == null) { result.Add(name); }
            }
            return result;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// テクスチャリストを開く。
        /// </summary>
        public bool OpenNameList(string basePath, List<string> nameList)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            bool result = OpenNameList(basePath, nameList, changed);
            Document.NotifyComponentChanged(this, changed);
            return result;
        }

        /// <summary>
        /// テクスチャリストを開く（TemplateMethod）。
        /// </summary>
        public virtual bool OpenNameList(string basePath, List<string> nameList,
            DocumentComponentChangedEventArgs changed)
        {
            bool result = true;
            foreach (string name in nameList)
            {
                if (TextureManager.Search(name) != null) { continue; }
                if (Open(basePath, name, changed) == null) { result = false; }
            }
            return result;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// テクスチャを開く。
        /// </summary>
        public ITextureDocument Open(string basePath, string name)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            ITextureDocument result = Open(basePath, name, changed);
            Document.NotifyComponentChanged(this, changed);
            return result;
        }

        /// <summary>
        /// テクスチャを開く（TemplateMethod）。
        /// </summary>
        public virtual ITextureDocument Open(
            string basePath, string name, DocumentComponentChangedEventArgs changed)
        {
            string filePath = SearchFile(basePath, name);
            if (filePath == null)
            {
                ErrorFileNotFound(name);
                return null;
            }
            return Open(filePath, changed);
        }

        /// <summary>
        /// テクスチャを開く。
        /// </summary>
        public ITextureDocument Open(string filePath)
        {
            DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
            ITextureDocument result = Open(filePath, changed);
            Document.NotifyComponentChanged(this, changed);
            return result;
        }

        /// <summary>
        /// テクスチャを開く。
        /// </summary>
        public ITextureDocument Open(string filePath, DocumentComponentChangedEventArgs changed)
        {
            if (!CheckFileExists(filePath)) { return null; }
            return OpenDocument(filePath, changed) as ITextureDocument;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// ドキュメントを開く（TemplateMethod）。
        /// </summary>
        protected override IDocument OpenDocumentImplement(
            string filePath, DocumentComponentChangedEventArgs changed)
        {
//TODO:未対応
#if false
            Debug.Assert(DocumentConstants.IsCtexPath(filePath) ||
                DocumentConstants.IsTgaPath(filePath));
#endif
#if false ////TODO:未対応
            using (MCSDisableBlock disableBlock = new MCS.MCSDisableBlock())
            using (Document.PropertyChangedSuppressBlock block =
                    new Document.PropertyChangedSuppressBlock())
            {
                // コンテントの読み込み
                IList<FileIoContentObjects> contents = IO.DocumentIOUtility.ReadContents(filePath);

                foreach (FileIoContentObjects content in contents)
                {
                    foreach (IContentObject contentItem in content.Objects)
                    {
                        ITexture texture = contentItem as ITexture;
                        if (texture != null)
                        {
                            // 同名データの解決
                            string name = Path.GetFileNameWithoutExtension(filePath);
                            TextureDocument sameNamed = TextureManager.Search(name);
                            if (sameNamed != null)
                            {
                                return ResolveSameNamed(sameNamed, null, filePath, name);
                            }

                            // 選択しているプロジェクトへ追加する
                            if (ProjectManager.Selected == null)
                            {
                                if (ProjectManager.Projects.Count > 0)
                                {
                                    // プロジェクトが未選択なら初めの１つ目を選択する
                                    ProjectManager.Selected = ProjectManager.Projects[0];
                                }
                                else
                                {
                                    // まだ１つもプロジェクトが無い場合は追加する
                                    DocumentIO.Command_FileNewProject(
                                        new MenuCommandArgs(false, null, new FileNewProjectArgs(false)));
                                    Debug.Assert(ProjectManager.Selected != null);
                                }
                            }

                            // ドキュメント生成
                            IReferenceTexture refTexture = new ReferenceTexture(texture);
                            refTexture.FilePath = filePath;
                            TextureDocument textureDoc = new TextureDocument(refTexture);

                            // コマンド実行
                            Command.EditTextureDocumentCommand command =
                                new Command.EditTextureDocumentCommand(
                                    textureDoc, null, textureDoc,
                                    Command.EditTextureDocumentCommand.OpType.Add);
                            TheApp.CommandManager.Execute(command);

                            // Success
                            return textureDoc;
                        }
                    }
                }
            }
#endif
            // Error Exit
            return null;
        }

        /// <summary>
        /// Nsrifドキュメントの読み込み。
        /// </summary>
        private bool ReadNsrifDocument(out IContentDocument document, string filePath)
        {
//TODO:未対応
#if false
            if (!DocumentConstants.IsTgaPath(filePath))
            {
                NsrifReader reader = new NsrifReader();
                nsrifDocument = reader.Read(filePath);
                ReportNsrifVersionUpdated(filePath, reader);
                return reader.Updated;
            }
            else
            {
                if (TgaReader.IsAppendInformation(filePath))
                {
                    nsrifDocument = TgaReader.CreateRtexDocument(filePath);
                    return (nsrifDocument != null);
                }
                else
                {
                    ErrorInvalidTga(filePath);
                    nsrifDocument = null;
                    return false;
                }
            }
#else
            document = null;
            return false;
#endif
        }

        /// <summary>
        /// 同名データの解決（TemplateMethod）。
        /// </summary>
        protected virtual ITextureDocument ResolveSameNamed(
            ITextureDocument sameNamed, IContentDocument document, string filePath, string name)
        {
            ITextureDocument result = null;
#if false
            if (UIMessageBox.YesNoRes("IO_DATA_OVERWRITE_MSG", name))
            {
                result = Overwrite(sameNamed, document);
                if (result == null) { ErrorDataOverwrite(name); }
            }
            else
#endif
            {
                ErrorFileNameCollision(filePath);
            }
            return result;
        }

        /// <summary>
        /// 上書き。
        /// </summary>
        protected ITextureDocument Overwrite(
            ITextureDocument sameNamed, IContentDocument nsrifDocument)
        {
//TODO:未対応
#if true
            if (sameNamed.OverwriteDocument(nsrifDocument))
            {
//				TextureLoadMessage.Open(sameNamed);

                Document.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(sameNamed));
                return sameNamed;
            }
#endif
            return null;
        }

        /// <summary>
        /// 生成（TemplateMethod）。
        /// </summary>
        protected virtual ITextureDocument Create(
            IContentDocument document, string filePath, string name, bool updated)
        {
//TODO:未対応
#if false
            if (DocumentConstants.IsTgaPath(filePath))
            {
                return new TextureDocument(name, document, updated);
            }
            else
#endif
            {
#if false
                // IProjectDocument projDoc, ITexture texture, IParticleModel model
                ITexture texture = DocumentCreator.CreateTexture(name, null);
                return DocumentCreator.CreateTextureDocument(
                    document as CtexDocument, EffectProjectManager.Selected, texture);
#else
                return null;
#endif
            }
        }
    }

    //=========================================================================
    /// <summary>
    /// マージテクスチャオープナ。
    /// </summary>
    public sealed class MergeTextureOpener : TextureOpener
    {
        // TODO: 動作に問題が無ければ消去する

        ///// <summary>
        ///// テクスチャを開く（TemplateMethod）。
        ///// </summary>
        //public override Texture Open(
        //    string basePath, string name, DocumentComponentChangedEventArgs changed)
        //{
        //    string filePath = SearchFile(basePath, name);
        //    if (filePath == null) { return null; }
        //    return Open(filePath, changed);
        //}

        ///// <summary>
        ///// 同名データの解決（TemplateMethod）。
        ///// </summary>
        //protected override Texture ResolveSameNamed(
        //    Texture sameNamed, NsrifDocument nsrifDocument, string filePath, string name)
        //{
        //    Texture result = null;
        //    if (MsgBox.YesNo(StringResource.Get("IO_DATA_OVERWRITE_MSG", name)))
        //    {
        //        result = Overwrite(sameNamed, nsrifDocument);
        //    }
        //    return result;
        //}
    }

    //=========================================================================
    /// <summary>
    /// スタータテクスチャオープナ。
    /// </summary>
    public sealed class StarterTextureOpener : TextureOpener
    {
        /// <summary>
        /// テクスチャリストを開く（TemplateMethod）。
        /// </summary>
        public override bool OpenNameList(string basePath, List<string> nameList,
            DocumentComponentChangedEventArgs changed)
        {
            bool result = true;
            foreach (string name in nameList)
            {
                // スタータ経由の場合はロード済みチェックを行わない
                if (Open(basePath, name, changed) == null) { result = false; }
            }
            return result;
        }

        /// <summary>
        /// 同名データの解決（TemplateMethod）。
        /// </summary>
        protected override ITextureDocument ResolveSameNamed(
            ITextureDocument sameNamed, IContentDocument nsrifDocument, string filePath, string name)
        {
            ITextureDocument result = Overwrite(sameNamed, nsrifDocument);
            if (result == null) { ErrorDataOverwrite(name); }
            return result;
        }
#if false
        /// <summary>
        /// 生成（TemplateMethod）。
        /// </summary>
        protected override TextureDocument Create(
            IContentDocument nsrifDocument, string filePath, string name, bool updated)
        {
            return new TextureDocument(name, nsrifDocument, true);
        }
#endif
    }
}
