﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace App.Data
{
    /// <summary>
    /// Axis Aligned Bounding Box クラス。
    /// </summary>
    public sealed class AABB
    {
        // 最小座標
        private Vector _minimum = new Vector();
        // 最大座標
        private Vector _maximum = new Vector();

        //---------------------------------------------------------------------
        // コンストラクタ
        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public AABB()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public AABB(AABB src)
        {
            Set(src);
        }

        //---------------------------------------------------------------------
        // アトリビュート
        //---------------------------------------------------------------------
        /// <summary>
        /// 最小座標。
        /// </summary>
        public Vector Minimum
        {
            get { return _minimum; }
            set { _minimum.Set(value); }
        }

        /// <summary>
        /// 最大座標。
        /// </summary>
        public Vector Maximum
        {
            get { return _maximum; }
            set { _maximum.Set(value); }
        }

        /// <summary>
        /// 位置を設定。
        /// </summary>
        public void SetPosition(Vector position)
        {
            SetPositionAndSize(position, GetSize());
        }

        /// <summary>
        /// サイズを設定。
        /// </summary>
        public void SetSize(Vector size)
        {
            SetPositionAndSize(GetPosition(), size);
        }

        /// <summary>
        /// 位置を取得。
        /// </summary>
        public Vector GetPosition()
        {
            return new Vector(
                ConvertToPosition(_minimum.X, _maximum.X),
                ConvertToPosition(_minimum.Y, _maximum.Y),
                ConvertToPosition(_minimum.Z, _maximum.Z)
            );
        }

        /// <summary>
        /// サイズを取得。
        /// </summary>
        public Vector GetSize()
        {
            return new Vector(
                ConvertToSize(_minimum.X, _maximum.X),
                ConvertToSize(_minimum.Y, _maximum.Y),
                ConvertToSize(_minimum.Z, _maximum.Z)
            );
        }

        /// <summary>
        /// 位置とサイズを設定。
        /// </summary>
        public void SetPositionAndSize(Vector position, Vector size)
        {
            float minX, minY, minZ;
            float maxX, maxY, maxZ;

            ConvertToMinMax(position.X, size.X, out minX, out maxX);
            ConvertToMinMax(position.Y, size.Y, out minY, out maxY);
            ConvertToMinMax(position.Z, size.Z, out minZ, out maxZ);

            _minimum.Set(minX, minY, minZ);
            _maximum.Set(maxX, maxY, maxZ);
        }

        //---------------------------------------------------------------------
        // 設定
        //---------------------------------------------------------------------
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(AABB src)
        {
            _minimum.Set(src._minimum);
            _maximum.Set(src._maximum);
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(Vector minimum, Vector maximum)
        {
            _minimum.Set(minimum);
            _maximum.Set(maximum);
        }

        //---------------------------------------------------------------------
        // 型変換
        //---------------------------------------------------------------------
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            return string.Format("Min:{0} Max:{1} (Pos:{2} Size:{3})",
                _minimum.ToString(),
                _maximum.ToString(),
                GetPosition().ToString(),
                GetSize().ToString()
            );
        }

        //---------------------------------------------------------------------
        // 比較
        //---------------------------------------------------------------------
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(AABB src)
        {
            if (src == null) return false;
            if (src == this) return true;

            return
                _minimum.Equals(src._minimum) &&
                _maximum.Equals(src._maximum);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as AABB);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return
                _minimum.GetHashCode() ^
                _maximum.GetHashCode();
        }

        #region 位置/サイズ変換

        /// <summary>
        /// 最小値/最大値→位置への変換。
        /// </summary>
        private static float ConvertToPosition(float min, float max)
        {
            return (min + max) / 2.0f;
        }

        /// <summary>
        /// 最小値/最大値→サイズへの変換。
        /// </summary>
        private static float ConvertToSize(float min, float max)
        {
            return max - min;
        }

        /// <summary>
        /// 位置/サイズ→最小値/最大値への変換。
        /// </summary>
        private static void ConvertToMinMax(float pos, float size, out float min, out float max)
        {
            min = pos - size / 2.0f;
            max = pos + size / 2.0f;
        }

        #endregion
    }
}
