﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using App.IO;

using NWCore.DataModel;
using NWCore.Serializer;

using NintendoWare.ToolDevelopmentKit.Xml;

namespace App.Data
{
    #region Class for user function information

    /// <summary>
    /// Class for user function information.
    /// </summary>
    public class UserFunctionInfo
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="pageName">Title for the property page.</param>
        /// <param name="userFuncName">Name of the user function.</param>
        /// <param name="iUserFuncID">ID of the user function.</param>
        public UserFunctionInfo( string pageName,
                                 string userFuncName,
                                 uint iUserFuncID )
        {
            this.PropertyPageName = string.Copy( pageName );
            this.UserFuncName     = string.Copy( userFuncName );
            this.UserFuncID       = iUserFuncID;
            this.Enabled          = true;
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source user function info.</param>
        public UserFunctionInfo( UserFunctionInfo src )
        {
            this.PropertyPageName = string.Copy( src.PropertyPageName );
            this.UserFuncName     = string.Copy( src.UserFuncName );
            this.UserFuncID       = src.UserFuncID;
            this.Enabled          = src.Enabled;
        }

        #endregion

        #region Properties

        /// <summary>Title for the property page.</summary>
        public string PropertyPageName { get; private set; }

        /// <summary>Name of the user function.</summary>
        public string UserFuncName     { get; private set; }

        /// <summary>ID of the user function.</summary>
        public uint   UserFuncID       { get; private set; }

        /// <summary>The flag indicating whether to enable this user function or not.</summary>
        public bool   Enabled          { get; set; }

        #endregion
    }

    #endregion

    /// <summary>
    /// Class for game settings document.
    /// </summary>
    public sealed class GameSettingsDocument : BaseProjectPanelDocument,
                                               IGameSettingsDocument,
                                               IDataModelProxyOwner
    {
        #region Constructors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public GameSettingsDocument( EmitterSetDocument emitterSetDoc,
                                     string name ) :
            base( name )
        {
            this.Name            = name;
            this.m_emitterSetDoc = emitterSetDoc;
            this.m_dataModel     = new GameConfigData();
            this.m_dataModel.DocumentID = this.DocumentID;

            ////UIManager UIMgr = TheApp.UIManager;
            ////string    strPageTitle;
            ////string    strUserFuncName;
            ////uint      iUserFuncID;

            ////int iNumUserFuncs =
            ////    UIMgr.GetNumUserDefinedPropertyPages( PropertyPanelID.GameSettingsPropertyPanel );
            ////for ( int i=0;i<iNumUserFuncs;++i )
            ////{
            ////    UIMgr.GetUserFunctionInfo( PropertyPanelID.GameSettingsPropertyPanel,
            ////                               i,
            ////                               out strPageTitle,
            ////                               out strUserFuncName,
            ////                               out iUserFuncID );

            ////    UserFunctionInfo info = new UserFunctionInfo( strPageTitle,
            ////                                                  strUserFuncName,
            ////                                                  iUserFuncID );

            ////    m_userFuncList.Add( info );
            ////}

            // Create and register data model proxy
            m_dataModelProxy = new DataModelProxy( this );
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        public GameSettingsDocument( EmitterSetDocument emitterSetDoc,
                                     GameSettingsDocument srcDoc ) :
            base( srcDoc.Name )
        {
            this.Name            = srcDoc.Name;
            this.m_emitterSetDoc = emitterSetDoc;
            this.m_dataModel     = new GameConfigData( srcDoc.m_dataModel );

            foreach ( UserFunctionInfo info in srcDoc.m_userFuncList )
            {
                this.m_userFuncList.Add( new UserFunctionInfo( info ) );
            }

            // Create and register data model proxy
            m_dataModelProxy = new DataModelProxy( this );
        }

        #endregion

        #region Properties

        /// <summary>
        /// 表示／非表示
        /// </summary>
        public bool IsShow
        {
            get { return m_dataModel.DispPreviewNode; }
            set
            {
                m_dataModel.DispPreviewNode = value;
//                /App.Viewer.Message.SendFunctionInfo.Send(this.OwnerDocument as IEmitterSetDocument);
//#if true
//                TheApp.Logger.Warn.AddMessage(
//                    String.Format("注意：まだ対応していません。表示：{0}", value.ToString()));
//#endif
            }
        }


        /// <summary>
        /// Internal flag indicating if the modify flag should be notified to parent document.
        /// </summary>
        protected override bool ShouldNotifyParentDocModified
        {
            get
            {
                return false;
            }
        }


        /// <summary>
        /// オーバーライド
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.GameSettings; }
        }


        /// <summary>
        /// オーナードキュメント
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return this.EmitterSetDocument; }
        }


        /// <summary>
        /// emitter set this emitter set belongs to
        /// </summary>
        public EmitterSetDocument EmitterSetDocument
        {
            get { return this.m_emitterSetDoc; }
            set { this.m_emitterSetDoc = value; }
        }


        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "GameConfigData";
            }
        }


        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                return this.EmitterSetDocument.DataScrPath +
                       ".GameSettings_uid" +
                       this.DocumentID.ToString();
            }
        }


        /// <summary>
        /// Get the game config data.
        /// </summary>
        public GameConfigData GameConfigData
        {
            get
            {
                return m_dataModel;
            }
        }


        /// <summary>
        /// Get or set the flag indicating if the model linkage page is enabled.
        /// </summary>
        public bool ModelLinkagePageEnabled
        {
            get { return m_bModelLinkagePageEnabled; }
            set
            {
                m_bModelLinkagePageEnabled = value;
                if ( UserFunctionEnableChanged!=null )
                    UserFunctionEnableChanged( this, EventArgs.Empty );
            }
        }


        /// <summary>
        /// Get the data model proxy.
        /// </summary>
        public IDataModelProxy DataModelProxy
        {
            get { return m_dataModelProxy; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string FileExt
        {
            get { return DocumentConstants.Prev; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string BinaryFileExt
        {
            get { return string.Empty; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string FileFilter
        {
            get { return res.Strings.IO_FILEFILTER_CAFE_PREV; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string BinaryFileFilter
        {
            get { return string.Empty; }
        }

        #endregion

        #region User functions

        /// <summary>
        /// Get the number of user functions.
        /// </summary>
        /// <returns>The number of user functions.</returns>
        public int GetNumUserFunctions()
        {
            return m_userFuncList.Count;
        }


        /// <summary>
        /// Get the user function information with the given index.
        /// </summary>
        /// <param name="iIndex">The index to the user function info.</param>
        /// <returns>The user function info.</returns>
        public UserFunctionInfo GetUserFunctionInfo( int iIndex )
        {
            if ( iIndex<0 || iIndex>=m_userFuncList.Count )
                return null;

            return m_userFuncList[iIndex];
        }


        /// <summary>
        /// Enable / disable the specified user function.
        /// </summary>
        /// <param name="iUserFuncID">User function ID.</param>
        /// <param name="bEnable">True to enable the user function.</param>
        public bool EnableUserFunction( uint iUserFuncID,
                                        bool bEnable )
        {
            var hasChanged = false;

            foreach ( UserFunctionInfo info in m_userFuncList )
            {
                if ( info!=null &&
                     info.UserFuncID==iUserFuncID )
                {
                    if ( info.Enabled!=bEnable )
                    {
                        info.Enabled = bEnable;
                        if ( m_dataModelProxy!=null )
                        {
                            DataModelInfo dataModel =
                                m_dataModelProxy.GetDataModel( info.UserFuncName );

                            if (dataModel != null)
                            {
                                dataModel.Enabled = bEnable;
                                hasChanged = true;
                            }
                        }

                        // Trigger event.
                        if ( UserFunctionEnableChanged!=null )
                        {
                            UserFunctionEnableChanged( this, System.EventArgs.Empty );
                        }
                    }
                    return hasChanged;
                }
            }

            return false;
        }


        /// <summary>
        /// Check if the specified user function is enabled.
        /// </summary>
        /// <param name="iUserFuncID">User function ID.</param>
        /// <returns>True if the user function is enabled.</returns>
        public bool IsDataModelEnabled( uint iUserFuncID )
        {
            foreach ( UserFunctionInfo info in m_userFuncList )
            {
                if ( info!=null &&
                     info.UserFuncID==iUserFuncID )
                {
                    return info.Enabled;
                }
            }

            return false;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// notify property has been changed
        /// </summary>
        public void NotifyPropertyChanged()
        {
            ProjectManager.UpdateAll();
        }


        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public override void NotifyAddedToParent()
        {
            base.NotifyAddedToParent();

            DocumentManager.RegisterDocumentPath(this, this.DataScrPath);

            DocumentManager.RegisterDataModelProxy( m_dataModelProxy );
        }


        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public override void NotifyRemovedFromParent()
        {
            // Compose the source data path.
            string docPath = this.DataScrPath + "." +
                             this.RelativeDataScrPath;

            DocumentManager.UnregisterDataModelProxy(m_dataModelProxy);

            DocumentManager.UnregisterDocumentPath( this, this.DataScrPath );

            base.NotifyRemovedFromParent();
        }


        /// <summary>
        /// Copy the data from the given source document data.
        /// </summary>
        /// <param name="srcData">The source document data.</param>
        public void Copy( GameConfigData srcData )
        {
            // Remember the document name first.
            string name = this.Name;

            // Copy the data model.
            this.GameConfigData.Set( srcData );
        }

        #endregion

        #region Save / Load

        /// <summary>
        /// ストリームにドキュメントの書き込みします。
        /// </summary>
        /// <param name="stream">保存先のストリーム。</param>
        /// <param name="bClearModifyFlag">
        /// Trueを指定したら、ドキュメントの変更のフラグを削除します。
        /// </param>
        /// <param name="bNormalizePath">
        /// Trueを指定場合、ドキュメントは記載するファイルパスを相対パスに転換する。
        /// </param>
        /// <returns>成功するとtrueを戻します。</returns>
        public override bool SaveDocumentToStream( System.IO.Stream stream,
                                                   bool bClearModifyFlag = true,
                                                   bool bNormalizePath = true )
        {
            // Create the serializer
            GameConfigDataXml xmlWritable;

            // Serialize the document data.
            SerializeDocument( out xmlWritable, false );

            // Serialize into the stream
            XmlUtility.SerializeToStream( xmlWritable, ref stream );

            if ( bClearModifyFlag==true )
                ClearModifiedFlag( true );

            return true;
        }


        /// <summary>
        /// Serialize the document to the XML data structure.
        /// </summary>
        /// <param name="outputData">The XML structure that being serialized to.</param>
        /// <param name="bUseEsetFullPath">True to store the full path of the emitter set.</param>
        /// <returns>True on success.</returns>
        public bool SerializeDocument( out GameConfigDataXml outputData,
                                       bool bUseEsetFullPath )
        {
            PrepareSavingTemp();

            string esetPath;
            if ( bUseEsetFullPath==true )
                esetPath = this.EmitterSetDocument.FilePath;
            else
                esetPath = this.EmitterSetDocument.FileName;

            // Create the serializer
            outputData = new GameConfigDataXml( esetPath,
                                                this.GameConfigData );

            // Set up the name
            outputData.Name                    = this.Name;
            outputData.Comment                 = this.Comment;
            outputData.ModelLinkagePageEnabled = this.ModelLinkagePageEnabled;

            // User functions
            IList<IDataModelSerializeList> dataModels =
                this.DataModelProxy.GetAllDataModelsForSerialize();
            foreach ( DataModelSerializeList dataModel in dataModels )
            {
                UserFunctionXML userFunc =
                    new UserFunctionXML( dataModel.Name, dataModel.ID );

                // Fields in the user function
                foreach ( DataModelSerializeItem item in dataModel )
                {
                    uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( item.Name );
                    userFunc.Add( item.Name, iNameCRC, item.Value );
                }

                outputData.UserFunctionList.Add( userFunc );
            }

            // Process pre-serialize event.
            outputData.PreSerialize();

            return true;
        }


        /// <summary>
        /// Load the contents from the deserializer.
        /// </summary>
        /// <param name="deserializer">The deserializer.</param>
        /// <returns>True on success.</returns>
        public bool LoadFromDeserializer( GameConfigDataXml deserializer )
        {
            // Set the game config data.
            this.GameConfigData.Set( deserializer.GameConfigData );

            this.Comment = deserializer.Comment;
            this.ModelLinkagePageEnabled = deserializer.ModelLinkagePageEnabled;

            // First disable all the user functions
            foreach ( UserFunctionInfo info in m_userFuncList )
            {
                if ( info.Enabled==true )
                {
                    info.Enabled = false;
                    DataModelInfo dataModel =
                        m_dataModelProxy.GetDataModel( info.UserFuncName );

                    if ( dataModel!=null )
                        dataModel.Enabled = false;
                }
            }

            // Load the user functions.
            foreach ( UserFunctionXML userFunc in deserializer.UserFunctionList )
            {
                (this.DataModelProxy as DataModelProxy).LoadUserFunction( userFunc );

                // Enable the loaded user function
                foreach ( UserFunctionInfo info in m_userFuncList )
                {
                    if ( info.UserFuncID==userFunc.UserFuncID )
                    {
                        info.Enabled = true;
                        DataModelInfo dataModel =
                            m_dataModelProxy.GetDataModel( info.UserFuncName );

                        if ( dataModel!=null )
                            dataModel.Enabled = true;
                    }
                }
            }

            ProcessLoadingTemp();

            // Trigger event to notify the user function has been enabled/disabled.
            if ( UserFunctionEnableChanged!=null )
            {
                UserFunctionEnableChanged( this, System.EventArgs.Empty );
            }

            return true;
        }

        #endregion

        #region Events

        /// <summary>
        /// Triggers when enable/disable any of the user functions.
        /// </summary>
        public event System.EventHandler UserFunctionEnableChanged;

        #endregion

        #region Event handlers

        /// <summary>
        /// Handle linked bone index being changed.
        /// </summary>
        /// <param name="command">The command that changed the source data.</param>
        /// <param name="executionType">The type of the command execution.</param>
        public void OnLinkedBoneIndexChanged( ICommand command,
                                              CommandExecutionTypes executionType )
        {
            int iModelIndex = this.GameConfigData.PreviewModelIndex;
            int iBoneIndex  = this.GameConfigData.BoneIndex;
            if ( iModelIndex<0 )
            {
                this.GameConfigData.BoneName = string.Empty;
                return;
            }

            // Get the preview model data.
            PreviewModelData modelData =
                ProjectManager.ActiveProject.EnvConfigData.PreviewConfigData.GetModelData( iModelIndex );

            if ( modelData==null ||
                 iBoneIndex<0 ||
                 iBoneIndex>=modelData.BoneCount )
            {
                this.GameConfigData.BoneName = string.Empty;
            }
            else
            {
                this.GameConfigData.BoneName = string.Copy( modelData.GetBoneName(iBoneIndex) );
            }
        }

        #endregion

        #region Owner document

        /// <summary>
        /// Set owner document.
        /// This method can move the document to another owner.
        /// </summary>
        /// <param name="owner">The new owner.</param>
        /// <param name="iIndex">The index the document should be placed at.</param>
        /// <returns>True on success.</returns>
        public bool SetOwnerDocument( EmitterSetDocument owner,
                                      int iIndex )
        {
            if ( owner==null )
                return false;

            this.EmitterSetDocument.RemoveGameSettingsDocument( this );

            this.EmitterSetDocument = owner;
            this.EmitterSetDocument.AddGameSettingsDocument( this, iIndex );

            return true;
        }

        #endregion

        #region Drag & drop

        /// <summary>
        /// Check if the document can be dragged on project tree view.
        /// </summary>
        /// <returns>True if can be dragged.</returns>
        public override bool CanDrag()
        {
            return true;
        }


        /// <summary>
        /// Check if the given document can be dropped and inserted before this document.
        /// </summary>
        /// <param name="doc">The dropped document.</param>
        /// <returns>True if the given document can be dropped.</returns>
        public override bool CanDrop( IDocument doc )
        {
            if ( (doc is GameSettingsDocument)==false )
                return false;

            if ( doc.OwnerDocument!=this.OwnerDocument )
                return false;

            return true;
        }

        #endregion

        #region Temporary handlers when saving / loading

        /// <summary>
        /// Prepare the document for saving.
        /// This method is temporarily here before the updater is made.
        /// </summary>
        private void PrepareSavingTemp()
        {
        }


        /// <summary>
        /// Process the document after loading.
        /// This method is temporarily here before the updater is made.
        /// </summary>
        private void ProcessLoadingTemp()
        {
            foreach (GameSettingsDocument doc in m_emitterSetDoc.GameSettingsDocuments)
            {
                float d = (float)Math.PI * 0.0001f;	// 誤差の値
                if (doc.GameConfigData.OffsetRotateX > Math.PI+d)
                {
                    doc.GameConfigData.OffsetRotateX -= (float)Math.PI * 2.0f;
                }
                if (doc.GameConfigData.OffsetRotateY > Math.PI+d)
                {
                    doc.GameConfigData.OffsetRotateY -= (float)Math.PI * 2.0f;
                }
                if (doc.GameConfigData.OffsetRotateZ > Math.PI+d)
                {
                    doc.GameConfigData.OffsetRotateZ -= (float)Math.PI * 2.0f;
                }
            }
        }

        #endregion

        #region Memeber Variables

        private          EmitterSetDocument m_emitterSetDoc = null; // parent/owner
        private readonly GameConfigData     m_dataModel     = null;

        private          DataModelProxy     m_dataModelProxy;

        private          bool               m_bModelLinkagePageEnabled = true;

        private List<UserFunctionInfo>      m_userFuncList  = new List<UserFunctionInfo>();

        #endregion
    }
}
