﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Windows.Forms;
using App.IO;
using NintendoWare.ToolDevelopmentKit.Xml;
using NWCore.DataModel;
using NWCore.Serializer;

namespace App.Data
{
    /// <summary>
    /// エフェクト用ドキュメント
    /// </summary>
    public sealed class EmitterSetDocument : BaseProjectPanelDocument, IEmitterSetDocument
    {
        #region Member Variables

        private readonly List<EmitterDocument>      m_emitterList = new List<EmitterDocument>();
        private readonly List<GameSettingsDocument> m_gameSettingsDocList = new List<GameSettingsDocument>();
        private IProjectDocument                    m_projectDoc = null;         // parent
        private EmitterSetData                      m_emitterSetData = null;     // data model
        private string                              m_fileTitle;
        private string                              m_srcDataPath = string.Empty;

        #endregion

        #region Properties

        /// <summary>
        /// 要求されて読み込んだノードか？
        /// </summary>
        public bool IsLinked
        {
            get;
            set;
        }

        /// <summary>
        /// デバッグ表示するか？
        /// </summary>
        public bool IsShow
        {
            get
            {
                return this.EmitterSetData.EditData.DebugData.DispEmitterSet;
            }
            set
            {
                this.EmitterSetData.EditData.DebugData.DispEmitterSet = value;
            }
        }

        /// <summary>
        /// ファイル名にタイトル（ファイル名の拡張子の無い名前の部分です。）
        /// </summary>
        public override string FileTitle
        {
            get { return m_fileTitle; }
            set { m_fileTitle = value; }
        }

        /// <summary>
        /// 名前
        /// </summary>
        public override string Name
        {
            get
            {
                if (this.EmitterSetData != null)
                {
                    return this.EmitterSetData.Name;
                }
                return String.Empty;
            }
            set
            {
                if (this.EmitterSetData != null)
                {
                    this.EmitterSetData.Name = value;
                }

                if ( (value!=null) &&
                     (value.Length>0) &&
                     (this.FileTitle==null ||
                      this.FileTitle.Length<=0) )
                {
                    this.FileTitle = value;
                }

                // 通知します。
                if (this.Project != null)
                {
                    (this.Project as EffectProjectDocument).NotifyEmitterSetNameChanged(this);
                }
            }
        }

        /// <summary>
        /// オーバーライド：OwnerDocumentです
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return this.m_projectDoc; }
        }

        /// <summary>
        /// オーバーライド：オブジェクト種類のＩＤ
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.EmitterSet; }
        }

        /// <summary>
        /// 関連するEffectProjectDocumet です
        /// </summary>
        public IProjectDocument Project
        {
            get { return this.m_projectDoc; }
            set { this.m_projectDoc = value; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string FileExt
        {
            get { return DocumentConstants.Eset; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string BinaryFileExt
        {
            get { return DocumentConstants.Beset; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string FileFilter
        {
            get { return res.Strings.IO_FILEFILTER_CAFE_ESET; }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override string BinaryFileFilter
        {
            get { return res.Strings.IO_FILEFILTER_CAFE_BPTCL; }
        }

        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "EmitterSetData";
            }
        }

        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                if ( this.Project==null )
                {
                    return "Project_uid0.EmitterSet_uid" +
                           this.DocumentID.ToString();
                }
                else
                {
                    return this.Project.DataScrPath +
                           ".EmitterSet_uid" +
                           this.DocumentID.ToString();
                }
            }
        }

        /// <summary>
        /// accessor for emitter data model encapsulated in this document
        /// </summary>
        public EmitterSetData EmitterSetData
        {
            get
            {
                return this.m_emitterSetData;
            }
        }

        /// <summary>
        /// エミッタ数
        /// </summary>
        public override int Count
        {
            get { return this.m_emitterList.Count; }
        }

        /// <summary>
        /// 子供のドキュメントを纏めて取得します
        /// </summary>
        public IEnumerable<IEmitterDocument> EmitterDocuments
        {
            get
            {
                foreach (EmitterDocument emitter in this.m_emitterList)
                {
                    yield return emitter;
                }
            }
        }

        /// <summary>
        /// 子供のドキュメントを纏めて取得します
        /// </summary>
        public IEnumerable<IGameSettingsDocument> GameSettingsDocuments
        {
            get
            {
                foreach ( GameSettingsDocument doc in this.m_gameSettingsDocList )
                {
                    yield return doc;
                }
            }
        }

        /// <summary>
        /// 子供のドキュメントを纏めて取得します
        /// </summary>
        public override IEnumerable<IDocument> SubDocuments
        {
            get
            {
                foreach (var item in this.EmitterDocuments)
                {
                    yield return item;
                }
                foreach (var item in this.GameSettingsDocuments)
                {
                    yield return item;
                }
            }
        }

        /// <summary>
        /// 関連するテクスチャのパスを取得します
        /// </summary>
        public IEnumerable<string> TexturePaths
        {
            get
            {
                foreach (var emitter in this.EmitterDocuments)
                {
                    foreach (var path in emitter.TexturePaths)
                    {
                        yield return path;
                    }
                }
            }
        }

        /// <summary>
        /// 関連するテクスチャを取得します
        /// </summary>
        public IEnumerable<EmitterAssetInfo> Textures
        {
            get
            {
                foreach ( var emitter in this.EmitterDocuments )
                {
                    foreach ( var texInfo in emitter.Textures )
                    {
                        yield return texInfo;
                    }
                }
            }
        }

        /// <summary>
        /// 関連するプリミティブのパスを取得します
        /// </summary>
        public IEnumerable<string> PrimitivePaths
        {
            get
            {
                foreach (var emitter in this.EmitterDocuments)
                {
                    foreach ( var path in emitter.PrimitivePaths )
                    {
                        yield return path;
                    }
                }
            }
        }

        /// <summary>
        /// 関連するテクスチャを取得します
        /// </summary>
        public IEnumerable<EmitterAssetInfo> Primitives
        {
            get
            {
                foreach ( var emitter in this.EmitterDocuments )
                {
                    foreach ( var info in emitter.Primitives )
                    {
                        yield return info;
                    }
                }
            }
        }

        /// <summary>
        /// テクスチャが読み込まれたか？
        /// </summary>
        public bool IsLoadedTextures
        {
            get;
            set;
        }

        #endregion

        #region Constructors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EmitterSetDocument(string name,
                                  string filePath,
                                  EmitterSetData dataModel,
                                  IProjectDocument projectDoc)
            : base(name)
        {
            this.IsLoadedTextures = false;
            this.m_projectDoc = projectDoc;
            this.FileTitle = Path.GetFileNameWithoutExtension(filePath);
            this.m_projectDoc = projectDoc;
            this.m_emitterSetData = dataModel;
            this.Name = name;

            DocumentManager.NotifyDocumentCreated(this);
        }

        /// <summary>
        /// コンストラクタ (for xml deserialization purpose)
        /// </summary>
        public EmitterSetDocument( IProjectDocument projectDoc )
            : base("")
        {
            if ( projectDoc!=null )
            {
                this.Project = projectDoc;
            }

            this.m_emitterSetData  = new EmitterSetData();

            DocumentManager.NotifyDocumentCreated(this);
        }

        #endregion

        #region Find child

        /// <summary>
        /// Find emitter document by name
        /// </summary>
        /// <param name="name">Name of the emitter document</param>
        /// <returns>The found emitter document or null if not found.</returns>
        public override IDocument FindChild( string name )
        {
            return FindEmitterDocument( name );
        }

        #endregion

        #region Emitter documents

        /// <summary>
        /// エミッタドキュメントのインデックスを取得します。
        /// </summary>
        public int IndexOf(IEmitterDocument emitter)
        {
            return this.m_emitterList.IndexOf(emitter as EmitterDocument);
        }

        /// <summary>
        /// Find emitter document by name
        /// </summary>
        /// <param name="name">Name of the emitter document</param>
        /// <returns>The found emitter document or null if not found.</returns>
        public EmitterDocument FindEmitterDocument( string name )
        {
            foreach ( EmitterDocument doc in EmitterDocuments )
            {
                if ( doc.Name==name )
                    return doc;
            }

            return null;
        }


        /// <summary>
        /// Move the given emitter document to the location the given index points to.
        /// </summary>
        /// <param name="doc">The emitter document to move.</param>
        /// <param name="iIndex">The index to move to.</param>
        /// <returns>True on success.</returns>
        public bool MoveEmitterDocument( EmitterDocument doc,
                                         int iIndex )
        {
            int iOrigIndex = m_emitterList.IndexOf( doc );
            if ( iOrigIndex<0 || iOrigIndex>=m_emitterList.Count )
                return false;

            if ( iOrigIndex==iIndex )
                return true;

            if ( iIndex<0 || iIndex>=m_emitterList.Count )
            {
                // Add the document to the end of the list.
                m_emitterList.Add( doc );
            }
            else
            {
                if ( iIndex<iOrigIndex )
                    ++iOrigIndex;

                // Insert the document back to the desired location.
                m_emitterList.Insert( iIndex, doc );
            }

            // Remove the document from the original location.
            m_emitterList.RemoveAt( iOrigIndex );

            return true;
        }


        /// <summary>
        /// EmitterDocumentを追加します
        /// </summary>
        /// <param name="document">追加するEmitterDocumentです</param>
        /// <param name="iIndex">挿入する位置のインデックスです、-1では最後に追加</param>
        public void AddEmitterDocument( IEmitterDocument _document,
                                        int iIndex = -1 )
        {
            EmitterDocument document = _document as EmitterDocument;

            /*
             *  追加
             */
            document.Enable = true;
            if ( iIndex>=0 &&
                 this.m_emitterList.Count>iIndex )
            {
                this.m_emitterList.Insert( iIndex, document );
            }
            else
            {
                this.m_emitterList.Add( document );
            }

            document.NotifyAddedToParent();
        }

        /// <summary>
        /// EmitterDocumentを削除します。
        /// </summary>
        /// <param name="document">削除するEmitterDocumentです</param>
        /// <returns>挿入されていたインデックスを返します</returns>
        public int RemoveEmitterDocument(IEmitterDocument _document)
        {
            EmitterDocument document = _document as EmitterDocument;

            // リストから削除します。
            int index = this.m_emitterList.IndexOf(document);
            if (index >= 0)
            {
                /*
                 *  削除
                 */
                // Remove the emitter document
                this.m_emitterList.RemoveAt( index );

                // Clear the flag from the emitters
                int iCount = 0;
                foreach (EmitterDocument child in m_emitterList)
                {
                    iCount += child.ModifyCount;
                }

                this.ChildModifyCount = iCount;

                // Notify the document that it's been removed
                document.NotifyRemovedFromParent();
            }

            return index;
        }

        /// <summary>
        /// get emitter document by index
        /// </summary>
        /// <param name="index">index of the document</param>
        /// <returns></returns>
        public IEmitterDocument GetEmitterDocumentByIndex(int index)
        {
            if (index >= 0 && index < this.m_emitterList.Count)
            {
                return this.m_emitterList[index];
            }

            return null;
        }

        /// <summary>
        /// get number of emitter documents
        /// </summary>
        public int GetEmitterDocumentCount()
        {
            return this.m_emitterList.Count;
        }

        /// <summary>
        /// エミッタタイプが変更された
        /// </summary>
        private void OnEmitterTypeChanged(ICommand command,
                                          CommandExecutionTypes executionType)
        {
            using ( NWCore.Viewer.MCSDisableBlock block = new NWCore.Viewer.MCSDisableBlock() )
            {
                // ツリーノードを再構築します。
                this.Project.BuildTreeNodes();
            }
        }

        #endregion

        #region Game settings documents

        /// <summary>
        /// Find game settings document by name
        /// </summary>
        /// <param name="name">Name of the game settings document</param>
        /// <returns>The found game settings document or null if not found.</returns>
        public GameSettingsDocument FindGameSettingsDocument( string name )
        {
            foreach ( GameSettingsDocument doc in m_gameSettingsDocList)
            {
                if ( doc.Name==name )
                    return doc;
            }

            return null;
        }


        /// <summary>
        /// Find the index of the game settings document
        /// </summary>
        /// <param name="name">Name of the game settings document</param>
        /// <returns>The found game settings document or null if not found.</returns>
        public int FindGameSettingsDocumentIndex( GameSettingsDocument doc )
        {
            return m_gameSettingsDocList.IndexOf( doc );
        }


        /// <summary>
        /// Move the given game settings document to the location the given index points to.
        /// </summary>
        /// <param name="doc">The game settings document to move.</param>
        /// <param name="iIndex">The index to move to.</param>
        /// <returns>True on success.</returns>
        public bool MoveGameSettingsDocument( GameSettingsDocument doc,
                                              int iIndex )
        {
            int iOrigIndex = m_gameSettingsDocList.IndexOf( doc );
            if ( iOrigIndex<0 || iOrigIndex>=m_gameSettingsDocList.Count )
                return false;

            if ( iOrigIndex==iIndex )
                return true;

            if ( iIndex<0 || iIndex>=m_gameSettingsDocList.Count )
            {
                // Add the document to the end of the list.
                m_gameSettingsDocList.Add( doc );
            }
            else
            {
                if ( iIndex<iOrigIndex )
                    ++iOrigIndex;

                // Insert the document back to the desired location.
                m_gameSettingsDocList.Insert( iIndex, doc );
            }

            // Remove the document from the original location.
            m_gameSettingsDocList.RemoveAt( iOrigIndex );

            return true;
        }


        /// <summary>
        /// Add game settings document.
        /// </summary>
        /// <param name="document">The game settings document.</param>
        /// <param name="iIndex">The index to insert the document to.</param>
        public void AddGameSettingsDocument( GameSettingsDocument doc,
                                             int iIndex = -1 )
        {
            // Add the document to our list.
            doc.Enable = true;
            if ( iIndex>=0 &&
                 m_gameSettingsDocList.Count>iIndex )
            {
                m_gameSettingsDocList.Insert( iIndex, doc );
            }
            else
            {
                m_gameSettingsDocList.Add( doc );
            }

            // Notify the document that it's been added.
            doc.NotifyAddedToParent();
        }


        /// <summary>
        /// Remove the specified game settings document.
        /// </summary>
        /// <param name="doc">The document to remove.</param>
        /// <returns>
        /// The index of the document that has been removed or
        /// -1 if the document does not exist.
        /// </returns>
        public int RemoveGameSettingsDocument( GameSettingsDocument doc )
        {
            // Find the document for removing.
            int iIndex = m_gameSettingsDocList.IndexOf( doc );
            if ( iIndex<0 || iIndex>=m_gameSettingsDocList.Count )
                return -1;

            // Remove the document
            m_gameSettingsDocList.RemoveAt( iIndex );

            // Notify the document that it's been removed
            doc.NotifyRemovedFromParent();

            return iIndex;
        }


        /// <summary>
        /// Get the number of game settings documents this emitter set has.
        /// </summary>
        /// <returns>The number of game settings documents.</returns>
        public int GetNumGameSettingsDocuments()
        {
            return m_gameSettingsDocList.Count;
        }


        /// <summary>
        /// Get game settings document by index.
        /// </summary>
        /// <param name="iIndex">Index to the document.</param>
        /// <returns>The document or null if the index is invalid.</returns>
        public GameSettingsDocument GetGameSettingsDocumentByIndex( int iIndex )
        {
            if ( iIndex<0 || iIndex>=m_gameSettingsDocList.Count )
                return null;

            return m_gameSettingsDocList[iIndex];
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Verify the textures of this emitter set and its emitters are set.
        /// </summary>
        /// <returns>True if all the texture paths are set.</returns>
        public bool VerifyTexturesAreSet()
        {
            bool bValid = true;
            foreach ( IEmitterDocument emitter in this.EmitterDocuments )
            {
                bValid &= emitter.VerifyTexturePathsAreSet(this.FileLocation);
            }

            return bValid;
        }

        /// <summary>
        /// Verify the texture paths of this emitter set and its emitters.
        /// </summary>
        /// <returns>True if all the texture paths are valid.</returns>
        public bool VerifyTexturesAreReachable()
        {
            bool bValid = true;
            foreach ( IEmitterDocument emitter in this.EmitterDocuments )
            {
                bValid &= emitter.VerifyTexturePathsAreReachable(this.FileLocation);
            }

            return bValid;
        }

        /// <summary>
        /// Verify the texture paths of this emitter set and its emitters.
        /// </summary>
        /// <param name="bCheckTexNotSet">True to check if the texture is set.</param>
        /// <param name="bCheckPath">True to check if the path is reachable.</param>
        /// <returns>True if all the texture paths are set and valid.</returns>
        private bool VerifyPrimitivePaths()
        {
            bool bValid = true;
            foreach ( IEmitterDocument emitter in this.EmitterDocuments )
            {
                bValid &= emitter.VerifyPrimitivePathsAreReachable( this.FileLocation );
            }

            return bValid;
        }


        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public override void NotifyAddedToParent()
        {
            base.NotifyAddedToParent();

            DocumentManager.RegisterDocumentPath( this, this.DataScrPath );

            // Emitter documents
            foreach ( EmitterDocument doc in this.m_emitterList )
            {
                // Notify the document that it has been added
                if ( doc!=null )
                    doc.NotifyAddedToParent();
            }

            // Game settings document.
            foreach ( GameSettingsDocument doc in this.m_gameSettingsDocList )
            {
                // Notify the document that it has been added
                if ( doc!=null )
                    doc.NotifyAddedToParent();
            }

            // Enabling auto save for the emitter sets.
            TheApp.AutoSaveManager.RegisterAutoSaveDoc( this );
        }


        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public override void NotifyRemovedFromParent()
        {
            // Remove the emitter set from the auto save queue.
            TheApp.AutoSaveManager.UnregisterAutoSaveDoc( this );

            // Game settings document.
            foreach ( GameSettingsDocument doc in this.m_gameSettingsDocList )
            {
                // Notify the document that it has been added
                if ( doc!=null )
                    doc.NotifyRemovedFromParent();
            }

            // Emitter documents
            foreach ( EmitterDocument doc in this.m_emitterList )
            {
                // Notify the document that it has been added
                if ( doc!=null )
                    doc.NotifyRemovedFromParent();
            }

            DocumentManager.UnregisterDocumentPath( this, this.DataScrPath );

            base.NotifyRemovedFromParent();
        }


        /// <summary>
        /// Duplicate this emitter set document with the given name and owner project name.
        /// Note that this method only copy the emitter set itself, the child emitters
        /// will not be copied to the new emitter set.
        /// </summary>
        /// <param name="name">Name for the new emitter set document.</param>
        /// <param name="project">The owner project of the new emitter set document.</param>
        /// <returns>The created emitter set document.</returns>
        public IEmitterSetDocument DuplicateSelf( string name,
                                                  IProjectDocument project )
        {
            return DuplicateSelf( name, project, false );
        }


        /// <summary>
        /// Duplicate this emitter set document with the given name and owner project name.
        /// Note that this method only copy the emitter set itself, the child emitters
        /// will not be copied to the new emitter set.
        /// </summary>
        /// <param name="name">Name for the new emitter set document.</param>
        /// <param name="project">The owner project of the new emitter set document.</param>
        /// <param name="bIgnoreChildDocs">
        /// True to skip the copying of the child documents. ( emitters and game settings )
        /// </param>
        /// <returns>The created emitter set document.</returns>
        public IEmitterSetDocument DuplicateSelf( string name,
                                                  IProjectDocument project,
                                                  bool bIgnoreChildDocs = false )
        {
            // Create a new emitter set.
            EmitterSetDocument doc = new EmitterSetDocument( project );
            if ( doc==null )
                return null;

            // Copy the emitter set
            doc.EmitterSetData.Set( this.EmitterSetData );

            doc.Name = string.Copy( name );

            if ( bIgnoreChildDocs==false )
            {
                // Copy emitter documents
                foreach ( EmitterDocument emitterDoc in this.m_emitterList )
                {
                    EmitterDocument newEmitter =
                        new EmitterDocument( doc, emitterDoc.EmitterData );

                    doc.AddEmitterDocument( newEmitter );
                }

                // Copy game settings documents
                foreach ( GameSettingsDocument gameSettingDoc in this.m_gameSettingsDocList )
                {
                    GameSettingsDocument newGameSettingDoc =
                        new GameSettingsDocument( doc, gameSettingDoc );

                    doc.AddGameSettingsDocument( newGameSettingDoc );
                }
            }

            return doc;
        }


        /// <summary>
        /// Check if the given document can be a child of this document.
        /// E.q. Emitter can be emitter set's child, but cannot be a child
        /// of an effect project document.
        /// </summary>
        /// <param name="doc">The child document.</param>
        /// <returns>True if the given document is a valid child document type.</returns>
        public override bool CanAddAsChildDoc( IDocument doc )
        {
            if ( doc is EmitterDocument )
            {
                EmitterSetDocument origOwner = doc.OwnerDocument as EmitterSetDocument;
                if ( origOwner!=null && origOwner.Count<=1 )
                    return false;

                bool bMyChild = ( this.m_emitterList.IndexOf(doc as EmitterDocument)>=0 );

                if ( bMyChild==false && this.Count>=TheApp.MAX_EMITTER_COUNT )
                    return false;

                return true;
            }
            else if ( doc is GameSettingsDocument )
            {
                EmitterSetDocument origOwner = doc.OwnerDocument as EmitterSetDocument;
                if ( origOwner!=null && origOwner.GetNumGameSettingsDocuments()<=1 )
                    return false;

                return true;
            }

            return false;
        }

        #endregion

        #region Modification state

        /// <summary>
        /// Notify that a modification has been done to the document.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        public override void SetModified( bool bUpdateProjectTree = true )
        {
            base.SetModified( bUpdateProjectTree );
            DocumentManager.NotifyDocumentModified( this );
        }


        /// <summary>
        /// Notify that a modification has been undone to the document.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        public override void UndoModify( bool bUpdateProjectTree = true )
        {
            base.UndoModify( bUpdateProjectTree );
            DocumentManager.NotifyDocumentModified( this );
        }


        /// <summary>
        /// Clear the modified flag.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        /// <returns>Number of cleared flag counts.</returns>
        public override int ClearModifiedFlag( bool bUpdateProjectTree )
        {
            // Clear the flag from the emitters
            int iClearCount = 0;
            foreach ( EmitterDocument child in m_emitterList )
            {
                iClearCount += child.ClearModifiedFlag( false );
            }

            bool bOrigModifyFlag = this.Modified;

            this.ChildModifyCount -= iClearCount;

            iClearCount += this.SelfModifyCount;
            this.SelfModifyCount = 0;

            if ( bUpdateProjectTree==true &&
                 (iClearCount>0 ||
                  this.Modified!=bOrigModifyFlag) )
            {
//                TheApp.MainFrame.RequestRebuildNodes();
            }

            return iClearCount;
        }

        #endregion

        #region Drag & drop

        /// <summary>
        /// Check if the document can be dragged on project tree view.
        /// </summary>
        /// <returns>True if can be dragged.</returns>
        public override bool CanDrag()
        {
            return true;
        }


        /// <summary>
        /// Check if the given document can be dropped and inserted before this document.
        /// </summary>
        /// <param name="doc">The dropped document.</param>
        /// <returns>True if the given document can be dropped.</returns>
        public override bool CanDrop( IDocument doc )
        {
            if ( (doc is EmitterSetDocument)==false )
                return false;

            EffectProjectDocument project = this.OwnerDocument as EffectProjectDocument;
            if ( project==null )
                return false;

            EmitterSetDocument eset = project.FindByName( doc.Name );
            if ( eset!=doc )
                return false;

            return true;
        }


        /// <summary>
        /// Move the specified child document to the location
        /// before another child document.
        /// </summary>
        /// <param name="doc">The child document to move.</param>
        /// <param name="beforeDoc">
        /// The child document which "doc" will be moved before,
        /// null to move "doc" to the last child document.
        /// </param>
        /// <returns>The document after moved.</returns>
        public override IDocument MoveChildDocument( IDocument doc,
                                                     IDocument beforeDoc )
        {
            // Same location?
            if ( doc==beforeDoc )
                return doc;

            if ( doc is EmitterDocument )
            {
                #region Move emitter document

                int iDestIndex = -1;
                if ( beforeDoc!=null )
                {
                    EmitterDocument beforeChild = beforeDoc as EmitterDocument;
                    if ( beforeChild==null || beforeChild.EmitterSetDocument!=this )
                        return null;

                    iDestIndex = m_emitterList.IndexOf( beforeChild );
                    if ( iDestIndex<0 || iDestIndex>=m_emitterList.Count )
                        return null;
                }

                EmitterDocument child = doc as EmitterDocument;
                if ( child.EmitterSetDocument!=this )
                {
                    using ( NWCore.Viewer.MCSDisableBlock block = new NWCore.Viewer.MCSDisableBlock() )
                    {
                        DocumentIO.CloneDocumentNode( child, this, iDestIndex, true );


                        if ( iDestIndex<0 )
                            return this.GetEmitterDocumentByIndex( this.Count-1 );
                        else
                            return this.GetEmitterDocumentByIndex( iDestIndex );
                    }
                }
                else
                {
                    int iOrigIndex = m_emitterList.IndexOf( child );
                    if ( iOrigIndex<0 || iOrigIndex>=m_emitterList.Count )
                        return null;

                    if ( beforeDoc==null &&
                         iOrigIndex==m_emitterList.Count-1 )
                    {
                        // Same location
                        return doc;
                    }
                    else if ( beforeDoc!=null &&
                              iOrigIndex==(iDestIndex-1) )
                    {
                        // Same location
                        return doc;
                    }
                }

                #endregion
            }
            else if ( doc is GameSettingsDocument )
            {
                #region Move game settings document

                int iDestIndex = -1;
                if ( beforeDoc!=null )
                {
                    GameSettingsDocument beforeChild = beforeDoc as GameSettingsDocument;
                    if ( beforeChild==null || beforeChild.EmitterSetDocument!=this )
                        return null;

                    iDestIndex = m_gameSettingsDocList.IndexOf( beforeChild );
                    if ( iDestIndex<0 || iDestIndex>=m_gameSettingsDocList.Count )
                        return null;
                }

                GameSettingsDocument child = doc as GameSettingsDocument;
                if ( child.EmitterSetDocument!=this )
                {
                    using ( NWCore.Viewer.MCSDisableBlock block = new NWCore.Viewer.MCSDisableBlock() )
                    {
                        if ( iDestIndex<0 )
                            return this.GetGameSettingsDocumentByIndex(this.GetNumGameSettingsDocuments()-1);
                        else
                            return this.GetGameSettingsDocumentByIndex(iDestIndex);
                    }
                }
                else
                {
                    int iOrigIndex = m_gameSettingsDocList.IndexOf( child );
                    if ( iOrigIndex<0 || iOrigIndex>=m_gameSettingsDocList.Count )
                        return null;

                    if ( beforeDoc==null &&
                         iOrigIndex==m_gameSettingsDocList.Count-1 )
                    {
                        // Same location
                        return doc;
                    }
                    else if ( beforeDoc!=null &&
                              iOrigIndex==(iDestIndex-1) )
                    {
                        // Same location
                        return doc;
                    }
                }

                #endregion
            }
            else
            {
                return null;
            }

            return doc;
        }

        #endregion

        #region Save / Load

        /// <summary>
        /// ストリームにドキュメントの書き込みします。
        /// </summary>
        /// <param name="stream">保存先のストリーム。</param>
        /// <param name="bClearModifyFlag">
        /// Trueを指定したら、ドキュメントの変更のフラグを削除します。
        /// </param>
        /// <param name="bNormalizePath">
        /// Trueを指定場合、ドキュメントは記載するファイルパスを相対パスに転換する。
        /// </param>
        /// <returns>成功するとtrueを戻します。</returns>
        public override bool SaveDocumentToStream( Stream stream,
                                                   bool bClearModifyFlag = true,
                                                   bool bNormalizePath = true )
        {
            PrepareSavingTemp();

            var list = new List<EmitterData>();
            foreach ( IEmitterDocument item in this.EmitterDocuments )
            {
                list.Add( item.EmitterData.Clone() );
            }

            var xmlWritable = new EmitterSetDataXml( this.m_emitterSetData,
                                                     list,
                                                     bNormalizePath );

            // serialize into xml
            XmlUtility.SerializeToStream( xmlWritable, ref stream );

            if ( bClearModifyFlag==true )
                ClearModifiedFlag( true );

            return true;
        }

        /// <summary>
        /// Create emitter set document from deserializer.
        /// </summary>
        /// <param name="project">The project document for the emitter set.</param>
        /// <param name="deserializer">The deserializer.</param>
        /// <returns>The created emitter set document.</returns>
        public static EmitterSetDocument CreateFromDeserializer( IProjectDocument project,
                                                                 EmitterSetDataXml deserializer )
        {
            EmitterSetDocument emitterSetDoc = new EmitterSetDocument( project );

            emitterSetDoc.EmitterSetData.Set( deserializer.EmitterSetData );
//<name>タグを削除
//            emitterSetDoc.Name = deserializer.name;

            foreach ( ComplexEmitterDataXml item in deserializer.EmitterList )
            {
                DocumentIO.ReadTextureFile( item );

                EmitterDocument emitter = new EmitterDocument( emitterSetDoc,
                                                               item.EditData );

                emitterSetDoc.AddEmitterDocument( emitter );
            }

            return emitterSetDoc;
        }

        /// <summary>
        /// ドキュメントをバイナリー保存します
        /// </summary>
        /// <returns>=true..成功</returns>
        public override bool SaveBinary(FileStream writer, bool isOptimize)
        {
            // コンテントをバイナリー保存します
            bool resut = false;
            try
            {
                //resut = exporter.Export(writer, targets);
            }
            catch (Exception ex)
            {
                DebugConsole.WriteLine(ex.Message);
                resut = false;
            }
            finally
            {
            }

            return resut;
        }

        /// <summary>
        /// Finish up the loading process.
        /// </summary>
        public void OnFinishedLoading( bool loadTexture,
                                       bool bSilent )
        {
            ProcessLoadingTemp();

            Dictionary<uint, uint> texturePaths = new Dictionary<uint, uint>();
            foreach ( IEmitterDocument item in this.EmitterDocuments )
            {
                EmitterDocument emitter = item as EmitterDocument;
                if ( emitter==null )
                    continue;

                #region Convert relative primitive paths

                // Convert relative primitive paths to absolute paths.
                {
                    string fullPath;
                    // Emitter primitive file path.
                    if ( string.IsNullOrEmpty(emitter.EmitterData.BillboardPrimitiveFileSource)==false )
                    {
                        if ( DocumentConstants.LocatePrimitiveByFileName(this.FileLocation,
                                                                         emitter.EmitterData.BillboardPrimitiveFileSource,
                                                                         out fullPath)==true )
                        {
                            emitter.EmitterData.BillboardPrimitiveFileSource = fullPath;
                        }
                    }

                    if (string.IsNullOrEmpty(emitter.EmitterData.PrimitiveEmitterFilePath) == false)
                    {
                        if (DocumentConstants.LocatePrimitiveByFileName(this.FileLocation,
                                                                         emitter.EmitterData.PrimitiveEmitterFilePath,
                                                                         out fullPath) == true)
                        {
                            emitter.EmitterData.PrimitiveEmitterFilePath = fullPath;
                        }
                    }

                    // Child primitive file path.
                    if ( string.IsNullOrEmpty(emitter.EmitterData.ChildData.BillboardPrimitiveFileSource)==false )
                    {
                        if ( DocumentConstants.LocatePrimitiveByFileName(this.FileLocation,
                                                                         emitter.EmitterData.ChildData.BillboardPrimitiveFileSource,
                                                                         out fullPath)==true )
                        {
                            emitter.EmitterData.ChildData.BillboardPrimitiveFileSource = fullPath;
                        }
                    }
                }

                #endregion

                // Map the draw path name to the corresponding ID.
                emitter.UpdateDrawPathID( bSilent );

                if ( loadTexture==true )
                {
                    string fullTexPath = String.Empty;

                    #region Load texture files

                    // 対応したテクスチャ０も読み込みます。
                    if ( DocumentIO.ReadTextureFile(this.FileLocation,
                                                    emitter.EmitterData.TexPatData0.UI_texPatFileName,
                                                    emitter.EmitterData.TexRes0,
                                                    ref fullTexPath)==true )
                    {
                        emitter.EmitterData.TexPatData0.UI_texPatFileName = fullTexPath;

                        uint iPathCRC =
                            TheApp.CRC32Helper.ComputeCRC32Str( fullTexPath.ToLower() );
                        if ( texturePaths.ContainsKey(iPathCRC)==false )
                            texturePaths.Add( iPathCRC, iPathCRC );
                    }

                    // テクスチャ１の読み込み
                    if ( DocumentIO.ReadTextureFile(this.FileLocation,
                                                    emitter.EmitterData.TexPatData1.UI_texPatFileName,
                                                    emitter.EmitterData.TexRes1,
                                                    ref fullTexPath)==true )
                    {
                        emitter.EmitterData.TexPatData1.UI_texPatFileName = fullTexPath;

                        uint iPathCRC =
                            TheApp.CRC32Helper.ComputeCRC32Str( fullTexPath.ToLower() );
                        if ( texturePaths.ContainsKey(iPathCRC)==false )
                            texturePaths.Add( iPathCRC, iPathCRC );
                    }

                    // テクスチャ２の読み込み
                    if ( DocumentIO.ReadTextureFile(this.FileLocation,
                                                    emitter.EmitterData.TexPatData2.UI_texPatFileName,
                                                    emitter.EmitterData.TexRes2,
                                                    ref fullTexPath)==true )
                    {
                        emitter.EmitterData.TexPatData2.UI_texPatFileName = fullTexPath;

                        uint iPathCRC =
                            TheApp.CRC32Helper.ComputeCRC32Str( fullTexPath.ToLower() );
                        if ( texturePaths.ContainsKey(iPathCRC)==false )
                            texturePaths.Add( iPathCRC, iPathCRC );
                    }

                    // チャイルドテクスチャーの読み込み
                    if ( DocumentIO.ReadTextureFile(this.FileLocation,
                                                    emitter.EmitterData.ChildTexPatData.UI_texPatFileName,
                                                    emitter.EmitterData.ChildData.TextureRes,
                                                    ref fullTexPath)==true )
                    {
                        emitter.EmitterData.ChildTexPatData.UI_texPatFileName = fullTexPath;

                        uint iPathCRC =
                            TheApp.CRC32Helper.ComputeCRC32Str( fullTexPath.ToLower() );
                        if ( texturePaths.ContainsKey(iPathCRC)==false )
                            texturePaths.Add( iPathCRC, iPathCRC );
                    }

                    #endregion

                    #region Create child documents if needed

                    // 各ドキュメント追加します。
                    if ( emitter.EmitterType==EmitterType.Complex )
                    {
                        emitter.CreateChildDocument();
                        emitter.CreateFieldDocument();
                        emitter.CreateFluctuationDocument();
                    }

                    #endregion

                    // Check if assets are reachable.
                    emitter.CheckAssetsReachability( false );
                    if ( emitter.EmitterType==EmitterType.Complex )
                    {
                        emitter.ChildDocument.CheckAssetsReachability( false );
                    }
                }
            }

            // Check if the loaded textures has same linear edit mode as the application settings.
            ////foreach ( uint iPathCRC in texturePaths.Values )
            ////{
            ////    TheApp.TextureManager.CheckTextureLinearEditMode( iPathCRC, !bSilent );
            ////}

            #region Check if the textures exist in the searching paths.

            if ( bSilent==false &&
                 loadTexture==true &&
                 VerifyTexturesAreReachable() == false)
            {
                string msg = string.Format( res.Strings.WARNING_TEXTURE_NOT_IN_SEARCH_PATH,
                                            this.Name );

                if ( TheApp.ConsoleMode==true )
                {
                    Console.WriteLine( msg );
                }
                else
                {
                    ////ThreadSafeMsgBox.Show( TheApp.MainFrame,
                    ////                       msg,
                    ////                       res.Strings.WARNING_CAPTION,
                    ////                       MessageBoxButtons.OK,
                    ////                       MessageBoxIcon.Exclamation );
                }
            }

            #endregion

            #region Check if the primitives exist in the searching paths.

            if ( bSilent==false &&
                 VerifyPrimitivePaths()==false )
            {
                string msg = string.Format( res.Strings.WARNING_PRIMITIVE_NOT_IN_SEARCH_PATH,
                                            this.Name );

                if ( TheApp.ConsoleMode==true )
                {
                    Console.WriteLine( msg );
                }
                else
                {
                    ////ThreadSafeMsgBox.Show( TheApp.MainFrame,
                    ////                       msg,
                    ////                       res.Strings.WARNING_CAPTION,
                    ////                       MessageBoxButtons.OK,
                    ////                       MessageBoxIcon.Exclamation );
                }
            }

            #endregion

            // テクスチャは読み込んだ？
            this.IsLoadedTextures = loadTexture;

            // update the recent documents list
            Config.EmitterRecentDocuments.NotifyDocumentLoaded(this);
        }

        protected override void OnDocumentSaveFinished(bool isError)
        {
            if (isError)
                return;

            Config.EmitterRecentDocuments.NotifyDocumentLoaded(this);
        }

        #endregion

        #region Temporary handlers when saving / loading

        /// <summary>
        /// Prepare the document for saving.
        /// This method is temporarily here before the updater is made.
        /// </summary>
        private void PrepareSavingTemp()
        {
            foreach ( IEmitterDocument doc in this.EmitterDocuments )
            {
                EmitterDocument emitter = doc as EmitterDocument;
                if ( emitter==null )
                    continue;

                // For internal users beginning to use new data model earlier than
                // external users, these temporary updaters need to stay here for
                // 2 versions.
                #region To be moved to updater on version 1.6.0.0 ( added on 1.4.0.0 )

                // Update codes had been removed.

                #endregion
            }
        }


        /// <summary>
        /// Process the document after loading.
        /// This method is temporarily here before the updater is made.
        /// </summary>
        private void ProcessLoadingTemp()
        {
            foreach ( IEmitterDocument doc in this.EmitterDocuments )
            {
                EmitterDocument emitter = doc as EmitterDocument;
                if ( emitter==null )
                    continue;

                #region To be moved to updater on version 1.7.0.0 ( added on 1.6.0.0 )

                // Update codes had been removed.

                #endregion

                #if BUILD_FOR_CTR

                // For CE2
                if ( doc.EmitterData.AnimEditData.Color0.ColorType==Constances.ColorSettingType.Animation ||
                     doc.EmitterData.AnimEditData.Color0.ColorType==Constances.ColorSettingType.Random )
                {
                    if ( doc.EmitterData.AnimEditData.Color1.ColorType!=Constances.ColorSettingType.Constant )
                    {
                        doc.EmitterData.AnimEditData.Color1.ColorType = Constances.ColorSettingType.Constant;

                        TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                             res.Strings.WARNING_SET_COLOR1_CONSTANT,
                                             this.Name,
                                             doc.Name );
                    }
                }

                #endif
            }
        }

        #endregion
    }
}
