﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.Globalization;
using System.Diagnostics;

namespace App.Data
{
    /// <summary>
    /// Class for storing/loading user data UI definition.
    /// </summary>
    public class UserDataUIDefinition : IXmlSerializable
    {
        public class XmlReadException : Exception
        {
        }

        #region Enumerator for the user data UI control types

        /// <summary>
        /// Enumerator for the user data UI control types.
        /// </summary>
        public enum UserDataUIControlTypes
        {
            Unknow   = -1,

            CheckBox = 0,
            IntSlider,
            FloatSlider
        }

        #endregion

        #region Class for UI group definition for user data parameters

        /// <summary>
        /// Class for UI group definition for user data parameters.
        /// </summary>
        public class UIGroupDefinition
        {
            #region Constructor

            /// <summary>
            /// Default constructor.
            /// </summary>
            public UIGroupDefinition()
            {
            }

            #endregion

            #region Initialize <= the kind of controls be in the group is decided here

            /// <summary>
            /// Initialize the data.
            /// </summary>
            /// <param name="groupType">The group type to initialize to.</param>
            /// <returns>True on success.</returns>
            public bool Init( UserDataUIControlTypes groupType )
            {
                m_groupType = groupType;

                this.Visible = true;

                switch ( groupType )
                {
                    case UserDataUIControlTypes.CheckBox :
                        {
                            this.Label = res.Strings.UI_EMITTER_SET_USERDATA_BIT_GROUP;

                            if ( (m_controls is CheckBoxUserUIDef[])==false ||
                                 (m_controls.Length!=NumCheckBoxDef) )
                            {
                                m_controls = new CheckBoxUserUIDef[NumCheckBoxDef];

                                // Initialize the control definitions.
                                for ( int i=0;i<m_controls.Length;++i )
                                {
                                    BaseUserUIControlDef item = new CheckBoxUserUIDef();
                                    if ( item==null )
                                        continue;

                                    m_controls[i] = item;

                                    // Setup the default label.
                                    item.Index = i;

                                    if ( item.Init()==false )
                                        return false;
                                }
                            }
                            else
                            {
                                // Initialize the control definitions.
                                foreach ( var item in m_controls )
                                {
                                    if ( item.Init()==false )
                                        return false;
                                }
                            }
                        }
                        break;

                    case UserDataUIControlTypes.IntSlider :
                        {
                            this.Label = res.Strings.UI_EMITTER_SET_USERDATA_NUMERIC_GROUP;

                            if ( (m_controls is IntSliderUserUIDef[])==false ||
                                 (m_controls.Length!=NumIntSliderDef) )
                            {
                                m_controls = new IntSliderUserUIDef[NumIntSliderDef];

                                // Initialize the control definitions.
                                for ( int i=0;i<m_controls.Length;++i )
                                {
                                    BaseUserUIControlDef item = new IntSliderUserUIDef();
                                    if ( item==null )
                                        continue;

                                    m_controls[i] = item;

                                    // Setup the default label.
                                    item.Index = i;

                                    if ( item.Init()==false )
                                        return false;
                                }
                            }
                            else
                            {
                                // Initialize the control definitions.
                                foreach (var item in m_controls)
                                {
                                    if (item.Init() == false)
                                        return false;
                                }
                            }
                        }
                        break;

                    case UserDataUIControlTypes.FloatSlider :
                        {
                            this.Label = res.Strings.UI_EMITTER_SET_USERDATA_FLOAT_GROUP;

                            if ( (m_controls is FloatSliderUserUIDef[])==false ||
                                 (m_controls.Length!=NumFloatSliderDef) )
                            {
                                m_controls = new FloatSliderUserUIDef[NumFloatSliderDef];

                                // Initialize the control definitions.
                                for ( int i=0;i<m_controls.Length;++i )
                                {
                                    BaseUserUIControlDef item = new FloatSliderUserUIDef();
                                    if ( item==null )
                                        continue;

                                    m_controls[i] = item;

                                    // Setup the default label.
                                    item.Index = i;

                                    if ( item.Init()==false )
                                        return false;
                                }
                            }
                            else
                            {
                                // Initialize the control definitions.
                                foreach (var item in m_controls)
                                {
                                    if (item.Init() == false)
                                        return false;
                                }
                            }
                        }
                        break;
                }

                return true;
            }

            #endregion

            #region Properties

            /// <summary>Get the index of the UI group.</summary>
            public int Index { get; private set; }

            /// <summary>Get or set the flag indicating whether this group is visible.</summary>
            public bool Visible { get; set; }

            /// <summary>Get or set the label of the group.</summary>
            public string Label { get; set; }

            /// <summary>Get the count of the control definitions.</summary>
            public int ControlDefCount
            {
                get
                {
                    if ( m_controls==null )
                        return 0;
                    return m_controls.Length;
                }
            }

            /// <summary>Get the enumerator for the control definitions in the group.</summary>
            public IEnumerable<BaseUserUIControlDef> ControlDefs
            {
                get
                {
                    if ( m_controls==null )
                    {
                        yield return null;
                    }
                    else
                    {
                        foreach ( BaseUserUIControlDef def in m_controls )
                        {
                            yield return def;
                        }
                    }
                }
            }

            #endregion

            #region Load group definition

            /// <summary>
            /// Load the definition from the given XML element.
            /// </summary>
            /// <param name="xml">The XML element containing the definition.</param>
            /// <returns>True on success.</returns>
            public bool LoadDefinition( XmlElement xml )
            {
                ////// Load the label for the group.
                ////if ( xml.HasAttribute(App.Controls.UIManager.XMLUIPrimaryLabelNodeName)==true )
                ////{
                ////    this.Label = xml.GetAttribute( App.Controls.UIManager.XMLUIPrimaryLabelNodeName );
                ////}
                ////else if ( xml.HasAttribute(App.Controls.UIManager.XMLUISecondaryLabelNodeName)==true )
                ////{
                ////    this.Label = xml.GetAttribute( App.Controls.UIManager.XMLUISecondaryLabelNodeName );
                ////}

                // Load the visibility flag for the group.
                if ( xml.HasAttribute("Visible")==true )
                {
                    bool bVisible = true;
                    if ( bool.TryParse( xml.GetAttribute("Visible"), out bVisible )==true )
                        this.Visible = bVisible;
                }

                // Load the index for the group.
                int iIndex;
                if ( int.TryParse(xml.GetAttribute("Index"),
                                  out iIndex)==false )
                {
                    return false;
                }

                this.Index = iIndex;

                // Load the template definition.
                XmlElement  template = null;
                XmlNodeList nodes    = xml.GetElementsByTagName( "Template" );
                if ( nodes!=null && nodes.Count>0 )
                    template = nodes[0] as XmlElement;

                // Load the control definitions.
                foreach ( XmlNode node in xml.ChildNodes )
                {
                    XmlElement child = node as XmlElement;
                    if ( child==null )
                        continue;

                    if ( child.Name==m_groupType.ToString() )
                    {
                        if ( int.TryParse(child.GetAttribute("Index"), out iIndex)==false )
                            continue;

                        if ( iIndex<0 || iIndex>=m_controls.Length )
                            continue;

                        // The control definitions should all be created and initialized.
                        // Just return false if they were not.
                        // This means someone tries to load definitions without initialize
                        // first.
                        BaseUserUIControlDef def = m_controls[iIndex];
                        if ( def==null )
                            return false;

                        // Load the control definitions.
                        if ( def.LoadDefinition(child, template)==false )
                            return false;
                    }
                }

                return true;
            }

            #endregion

            #region Utility methods

            /// <summary>
            /// Set the visibility of all the definitions to false.
            /// </summary>
            public void HideAll()
            {
                this.Visible = false;

                if ( m_controls!=null )
                {
                    foreach ( BaseUserUIControlDef def in m_controls )
                    {
                        if ( def!=null )
                            def.Visible = false;
                    }
                }
            }

            #endregion

            #region Constants <= the number of controls in the group is defined here

            private const int NumCheckBoxDef    = 16;
            private const int NumIntSliderDef   = 6;
            private const int NumFloatSliderDef = 8;

            #endregion

            #region Member variables

            private UserDataUIControlTypes m_groupType = UserDataUIControlTypes.Unknow;
            private BaseUserUIControlDef[] m_controls  = null;

            #endregion
        }

        #endregion

        #region Constructors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserDataUIDefinition()
        {
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool Init( int iIndex )
        {
            this.Index   = iIndex;
            this.Label   = string.Empty;
            this.Visible = true;

            if ( m_cbGroup==null || m_cbGroup.Init(UserDataUIControlTypes.CheckBox)==false )
                return false;

            if ( m_intSldGroup==null || m_intSldGroup.Init(UserDataUIControlTypes.IntSlider)==false )
                return false;

            if ( m_floatSldGroup==null || m_floatSldGroup.Init(UserDataUIControlTypes.FloatSlider)==false )
                return false;

            return true;
        }

        #endregion

        #region Properties

        /// <summary>Get the index of the UI definition.</summary>
        public int Index { get; private set; }

        /// <summary>
        /// 定義のファイル名
        /// </summary>
        public string Name { get; set; }

        /// <summary>Get the label of the UI definition.</summary>
        public string Label
        {
            get
            {
                if ( string.IsNullOrEmpty(m_label)==true )
                    return string.Format( res.Strings.UI_EMITTER_SET_USERDATA_CALLBACK_ID_ITEM, this.Index );
                return m_label;
            }
            private set { m_label = value; }
        }

        /// <summary>Get or set the flag indicating whether this control is visible.</summary>
        public bool Visible { get; set; }


        /// <summary>
        /// Get the check box group definition.
        /// </summary>
        public UIGroupDefinition CheckBoxGroupDefinition
        {
            get { return m_cbGroup; }
        }


        /// <summary>
        /// Get the integer slider groups definitions.
        /// </summary>
        public UIGroupDefinition IntSliderGroupDefinition
        {
            get { return m_intSldGroup; }
        }


        /// <summary>
        /// Get the float slider group definition.
        /// </summary>
        public UIGroupDefinition FloatSliderGroupDefinition
        {
            get { return m_floatSldGroup; }
        }

        #endregion

        #region UI group definitions

        /// <summary>
        /// Get the UI group definition with the given group type.
        /// </summary>
        /// <param name="groupType">The group type.</param>
        /// <returns>The UI group definition.</returns>
        public UIGroupDefinition GetGroupDef( UserDataUIControlTypes groupType )
        {
            switch ( groupType )
            {
                case UserDataUIControlTypes.CheckBox:
                    return m_cbGroup;

                case UserDataUIControlTypes.IntSlider:
                    return m_intSldGroup;

                case UserDataUIControlTypes.FloatSlider:
                    return m_floatSldGroup;
            }

            return null;
        }


        /// <summary>
        /// Set the visibility of all the definitions to false.
        /// </summary>
        public void HideAll()
        {
            if ( m_cbGroup!=null )
                m_cbGroup.HideAll();

            if ( m_intSldGroup!=null )
                m_intSldGroup.HideAll();

            if ( m_floatSldGroup!=null )
                m_floatSldGroup.HideAll();
        }

        #endregion

        #region Implemented for IXmlSerializable

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml( XmlReader reader )
        {
            // First reset the definitions.
            this.Init( 0 );
            this.HideAll();

            if ( reader.MoveToContent()==XmlNodeType.Element &&
                 reader.LocalName=="UserDataUIDefinition" )
            {
                XmlDocument doc = new XmlDocument();
                doc.LoadXml( reader.ReadOuterXml() );
                SetFromXMLDocument( doc.DocumentElement );
            }
        }

        /// <summary>
        /// Set this according to the XML document
        /// </summary>
        /// <param name="root">Root of the XML document</param>
        private void SetFromXMLDocument( XmlElement root )
        {
            Debug.Assert( root!=null );

            // Load the index of the definition.
            {
                int iIndex;
                if ( int.TryParse( root.GetAttribute( "Index" ),
                                   out iIndex )==false )
                {
                    throw new XmlReadException();
                }

                // Set the index of the definition.
                this.Index = iIndex;
            }

            // Load the visibility flag of the definition.
            {
                bool bVisible;
                if ( bool.TryParse( root.GetAttribute( "Visible" ),
                                    out bVisible )==false )
                {
                    bVisible = true;
                }

                // Set the visibility flag of the definition.
                this.Visible = bVisible;
            }

            ////// Load the label for the definition.
            ////if ( root.HasAttribute( App.Controls.UIManager.XMLUIPrimaryLabelNodeName )==true )
            ////{
            ////    this.Label = root.GetAttribute( App.Controls.UIManager.XMLUIPrimaryLabelNodeName );
            ////}
            ////else if ( root.HasAttribute( App.Controls.UIManager.XMLUISecondaryLabelNodeName )==true )
            ////{
            ////    this.Label = root.GetAttribute( App.Controls.UIManager.XMLUISecondaryLabelNodeName );
            ////}
            ////else
            {
                this.Label = string.Empty;
            }

            foreach ( XmlNode node in root.ChildNodes )
            {
                XmlElement child = node as XmlElement;
                if ( child==null || child.Name!="Group" )
                    continue;

                UserDataUIControlTypes grpType;
                if ( Enum.TryParse( child.GetAttribute( "Type" ),
                                    out grpType )==false )
                {
                    continue;
                }

                UIGroupDefinition group = GetGroupDef( grpType );
                if ( group==null )
                    continue;

                group.Init( grpType );
                group.HideAll();
                group.LoadDefinition( child );
            }
        }


        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml( XmlWriter writer )
        {
            // Not implemented.
        }


        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        #endregion

        #region Member variables

        private const int         m_iNumRBGroups = 4;

        private string            m_label        = string.Empty;

        private UIGroupDefinition m_cbGroup       = new UIGroupDefinition();
        private UIGroupDefinition m_intSldGroup   = new UIGroupDefinition();
        private UIGroupDefinition m_floatSldGroup = new UIGroupDefinition();

        #endregion
    }
}
