﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
//#define CHECK_INIT_ERROR

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Security.Principal;
using System.Threading;
using System.Windows.Forms;
using System.Xml.Serialization;
using App.Data;
using App.Global;
using App.IO;
using App.Utility;

#if ENABLE_EFFECT_COMBINER_EDITOR
using App.Services;
#endif

using NintendoWare.ToolDevelopmentKit.Logs; // OK
using NWCore.src.Remake.DataModel.DataModels;
using NWCore.src.Remake.DataModel.DataModels.Emitter;
using NWCore.Utility;
using NWCore.DataModel;
using NWCore.Viewer;

namespace App
{
    /// <summary>
    /// アプリケーションクラス。
    /// </summary>
    public static class TheApp
    {
        #region Application constants

        const string WorkCGToolFolder    = "work\\cgtool";
        const string WorkMCSFolder       = "work\\mcs";
        const string WorkTempFolder      = "work\\temp";
        const string WorkViewerFolder    = "work\\viewer";
        const string WorkShaderFolder    = "work\\shader";
        const string TempFolderName      = "EffectMaker";
        const string LocalizeDLL_English = "\\en\\{0}.resources.dll";
        const string LocalizeDir_English = "en";
        const string G3dXsdBaseFodler    = @"g3dxsd\";
        const string InitFileExt         = ".ini";
        const string UserConfigFileName  = "NW4F_EffectMaker_UserConfig.ini";

        const string TmpEmitterVSFileName = "emitter.vsh";
        const string TmpEmitterFSFileName = "emitter.fsh";
        const string TmpChildVSFileName   = "child.vsh";
        const string TmpChildFSFileName   = "child.fsh";

        const string HelpFilePath00 = "docs/EffectMaker.chm";
        const string HelpFilePath01 = "EffectMaker.chm";
        const string HelpFilePath02 = "../EffectMaker.chm";
        const string HelpFilePath03 = "../documents/EffectMaker.chm";
        const string HelpDocMapsXml = "docs/EffectMaker_HelpDocMaps.xml";

        const string UserDataParamUIFolder     = "";
        const string UserDataParamUIDefFileExt = ".udd";

        /// <summary>Maximum number of user shader UI definitions.</summary>
        public const int    MAX_NUM_USER_SHADER_UI_DEFINITIONS = 8;

        /// <summary>Maximum number of user data UI definitions.</summary>
        public const int    MAX_NUM_USER_DATA_UI_DEFINITIONS   = 9; // Index = -1, 0 ~ 7

        /// <summary>追加できるEmitterSetの最大数</summary>
        public const int MAX_EMITTERSET_COUNT = 16;

        /// <summary>追加できるEmitterの最大数</summary>
        public const int MAX_EMITTER_COUNT = 16;

        #endregion

        #region Static member varialbles

        /// <summary>デフォルトのコマンド上限数</summary>
        public const int DefaultLimitCommandCount = 100;

        // 英語ＵＩモード
        private static bool _englishUI = false;
        // ＧＵＩフォント
        private static Font _guiFont = null;
        // ＧＵＩフォント（太字）
        private static Font _guiFontBold = null;
        // ＧＵＩ固定幅フォント
        private static Font _guiFixedFont = null;
        // ＧＵＩ固定幅フォント（太字）
        private static Font _guiFixedFontBold = null;

        // メインスレッド
        private static Thread _mainThread = null;

        // 引数ファイルリスト
        private static string                _connectTarget = string.Empty;
        private static readonly List<string> _openFiles     = new List<string>();
        // 例外処理済みフラグ
        private static bool _fatalErrorHandled = false;

        private static AssociationManager _associationManager = null;

        /// <summary>オプション変更イベント。</summary>
        public static event EventHandler OptionChanged = null;

        // HelpFilePath
        private static string _helpFilePath = String.Empty;

        // CRC32 utility
        private static CRC32 _CRC32Helper = null;

        // Animation table manager
        private static AnimationTableManager _animTableManager = null;

        // Auto save manager.
        private static AutoSaveManager _autoSaveManager = null;

#if ENABLE_EFFECT_COMBINER_EDITOR
        // EffectCombinerEditor communication manager
        private static EffectCombinerCommunicationManager _eftCombinerCommManager = null;
#endif
        // Data model updater manager
        private static DataModelUpdaterManager _dataModelUpdaterManager = null;

        // Texture manager
        private static App.Data.TextureManager _textureManager = null;

        // ロガーです。
        private static Logger4C _logger = null;

        // テンポラリーフォルダです。
        private static string _tempFolderPath;
        private static string _applicationPath;
        private static string _workingFolder;

        // 初期設定
        private static InitSettingsData _initSettingsData;

        private static CustomShaderUIDefList _userShaderUIDefList = null;

        private static UserDataUIDefinition[] _userDataUIDefList =
            new UserDataUIDefinition[MAX_NUM_USER_DATA_UI_DEFINITIONS];

        private static bool _bSkipUpdateDrawPathWarningDialog = false;
        private static bool _bUpdateDrawPath                  = false;

        // リニア編集モードが変更された際のイベント
        public delegate void GammaCorrectionChanged();

        /// <summary>
        /// Gamma correction changed event
        /// </summary>
        public static GammaCorrectionChanged OnGammaCorrectionModeChanged;

        #endregion

        #region プロパティ

        /// <summary>
        /// Get the flag indicating whether the application is running on console.
        /// </summary>
        public static bool ConsoleMode { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to enable warning dialogs under console mode.
        /// </summary>
        public static bool EnableConsoleModeWarningDialogs { get; set; }

        /// <summary>
        /// 言語モード。
        /// </summary>
        public static LanguageMode LanguageMode
        {
            get { return Config.Data.Application.LanguageModeAccessor; }
            set { Config.Data.Application.LanguageModeAccessor = value; }
        }

        /// <summary>
        /// 英語ＵＩモードかどうか。
        /// </summary>
        public static bool IsEnglishUI
        {
            get { return _englishUI; }
        }

        /// <summary>
        /// ＧＵＩフォント。
        /// </summary>
        public static Font GuiFont
        {
            get { return _guiFont; }
        }

        /// <summary>
        /// ＧＵＩフォント（太字）。
        /// </summary>
        public static Font GuiFontBold
        {
            get { return _guiFontBold; }
        }

        /// <summary>
        /// ＧＵＩ固定幅フォント。
        /// </summary>
        public static Font GuiFixedFont
        {
            get { return _guiFixedFont; }
        }

        /// <summary>
        /// ＧＵＩ固定幅フォント（太字）。
        /// </summary>
        public static Font GuiFixedFontBold
        {
            get { return _guiFixedFontBold; }
        }

        /// <summary>
        /// Get or set the default preset template folder path.
        /// </summary>
        public static string DefaultPresetFolderPath { get; private set; }

        /// <summary>
        /// Get or set the preset template folder path.
        /// </summary>
        public static string PresetFolderPath
        {
            get
            {
                if (String.IsNullOrEmpty(Config.Data.DocumentIO.PresetTemplateFolder) == true)
                {
                    return DefaultPresetFolderPath;
                }

                return Config.Data.DocumentIO.PresetTemplateFolder;
            }

            set
            {
                if (Config.Data.DocumentIO.PresetTemplateFolder == value)
                {
                    return;
                }

                if (value.ToLower() == DefaultPresetFolderPath.ToLower())
                {
                    Config.Data.DocumentIO.PresetTemplateFolder = string.Empty;
                    return;
                }

                Config.Data.DocumentIO.PresetTemplateFolder = value;
            }
        }

        /// <summary>
        /// Get or set the user shader path for binary converter.
        /// </summary>
        public static string UserShaderPath
        {
            get
            {
                return Config.Data.Option.UserShaderPath;
            }
            set
            {
                if ( Config.Data.Option.UserShaderPath==value )
                    return;

                Config.Data.Option.UserShaderPath = value;

                LoadUserShaderUIDefinitions();
            }
        }

        /// <summary>
        /// Get the iterator for the user shader UI definitions.
        /// </summary>
        public static IEnumerable<UserShaderUIDefinition> UserShaderUIDefinitions
        {
            get
            {
                if (_userShaderUIDefList == null)
                {
                    LoadUserShaderUIDefinitions();
                }

                foreach (CustomShaderUIDefLinkData data in _userShaderUIDefList.DefinitionList)
                {
                    if ( data==null )
                        yield return null;

                    yield return data.DefinitionData;
                }
            }
        }

        /// <summary>
        /// USDのリンク情報リストを取得します.
        /// </summary>
        public static List<CustomShaderUIDefLinkData> UserShaderUIDefLinkList
        {
            get
            {
                if (_userShaderUIDefList == null)
                {
                    LoadUserShaderUIDefinitions();
                }

                return _userShaderUIDefList.DefinitionList;
            }
        }

        /// <summary>
        /// UDDのリンク情報リストを取得します.
        /// </summary>
        public static List<UserDataUIDefinition> CustomActionUIDefLinkList
        {
            get
            {
                if (_userDataUIDefList == null)
                {
                    LoadUserDataUIDefinitions();
                }

                return _userDataUIDefList.ToList();
            }
        }

        /// <summary>
        /// Get or set the user data path.
        /// </summary>
        public static string UserDataPath
        {
            get
            {
                if ( String.IsNullOrEmpty(Config.Data.Option.UserDataPath)==true )
                    return string.Empty;

                return Config.Data.Option.UserDataPath;
            }
            set
            {
                if ( Config.Data.Option.UserDataPath==value )
                    return;

                Config.Data.Option.UserDataPath = value;

                LoadUserDataUIDefinitions();
            }
        }

        /// <summary>
        /// Get the iterator for the user data UI definitions.
        /// </summary>
        public static IEnumerable<UserDataUIDefinition> UserDataUIDefinitions
        {
            get
            {
                foreach ( UserDataUIDefinition def in _userDataUIDefList )
                {
                    yield return def;
                }
            }
        }

        /// <summary>
        /// 保存前コマンドパス
        /// </summary>
        public static string PreSaveCommandPath
        {
            get
            {
                if (String.IsNullOrEmpty(Config.Data.Option.PreSaveCommandPath) == true)
                    return string.Empty;

                return Config.Data.Option.PreSaveCommandPath;
            }
            set
            {
                if (Config.Data.Option.PreSaveCommandPath == value)
                    return;

                Config.Data.Option.PreSaveCommandPath = value;
            }
        }

        /// <summary>
        /// 保存後コマンドパス
        /// </summary>
        public static string PostSaveCommandPath
        {
            get
            {
                if (String.IsNullOrEmpty(Config.Data.Option.PostSaveCommandPath) == true)
                    return string.Empty;

                return Config.Data.Option.PostSaveCommandPath;
            }
            set
            {
                if (Config.Data.Option.PostSaveCommandPath == value)
                    return;

                Config.Data.Option.PostSaveCommandPath = value;
            }
        }

        /// <summary>
        /// 削除前コマンドパス
        /// </summary>
        public static string PreDeleteCommandPath
        {
            get
            {
                if (String.IsNullOrEmpty(Config.Data.Option.PreDeleteCommandPath) == true)
                    return string.Empty;

                return Config.Data.Option.PreDeleteCommandPath;
            }
            set
            {
                if (Config.Data.Option.PreDeleteCommandPath == value)
                    return;

                Config.Data.Option.PreDeleteCommandPath = value;
            }
        }

        /// <summary>
        /// 削除後コマンドパス
        /// </summary>
        public static string PostDeleteCommandPath
        {
            get
            {
                if (String.IsNullOrEmpty(Config.Data.Option.PostDeleteCommandPath) == true)
                    return string.Empty;

                return Config.Data.Option.PostDeleteCommandPath;
            }
            set
            {
                if (Config.Data.Option.PostDeleteCommandPath == value)
                    return;

                Config.Data.Option.PostDeleteCommandPath = value;
            }
        }

        /// <summary>
        /// オープン前コマンドパス
        /// </summary>
        public static string PreOpenCommandPath
        {
            get
            {
                if (String.IsNullOrEmpty(Config.Data.Option.PreOpenCommandPath) == true)
                    return string.Empty;

                return Config.Data.Option.PreOpenCommandPath;
            }
            set
            {
                if (Config.Data.Option.PreOpenCommandPath == value)
                    return;

                Config.Data.Option.PreOpenCommandPath = value;
            }
        }

        /// <summary>
        /// クローズ後コマンドパス
        /// </summary>
        public static string PostCloseCommandPath
        {
            get
            {
                if (String.IsNullOrEmpty(Config.Data.Option.PostCloseCommandPath) == true)
                    return string.Empty;

                return Config.Data.Option.PostCloseCommandPath;
            }
            set
            {
                if (Config.Data.Option.PostCloseCommandPath == value)
                    return;

                Config.Data.Option.PostCloseCommandPath = value;
            }
        }

        /// <summary>
        /// Get the effect combiner communication manager.
        /// </summary>
#if ENABLE_EFFECT_COMBINER_EDITOR
        public static EffectCombinerCommunicationManager EffectCombinerCommManager
        {
            get { return _eftCombinerCommManager; }
        }
#endif

        /// <summary>
        /// Association MessageManager
        /// </summary>
        public static AssociationManager AssociationManager
        {
            get { return _associationManager; }
        }

        /// <summary>
        /// メインスレッド。
        /// </summary>
        public static Thread MainThread
        {
            get { return _mainThread; }
        }


        /// <summary>
        /// Get the texture manager.
        /// </summary>
        public static TextureManager TextureManager
        {
            get { return _textureManager; }
        }

        /// <summary>
        /// Data model updater manager.
        /// </summary>
        public static DataModelUpdaterManager UpdaterManager
        {
            get { return _dataModelUpdaterManager; }
        }

        /// <summary>
        /// Get the animation table manager.
        /// </summary>
        public static AnimationTableManager AnimationTableManager
        {
            get { return _animTableManager; }
        }


        /// <summary>
        /// オートバックアップマネジャを取得します。
        /// </summary>
        public static AutoSaveManager AutoSaveManager
        {
            get { return _autoSaveManager; }
        }

        /// <summary>
        /// ロガー。
        /// </summary>
        public static ILogger Logger
        {
            get { return _logger; }
        }

        /// <summary>
        /// ウィンドウをずらしながら重ねる際のマージン。
        /// </summary>
        public static int WindowCascadingMargin
        {
            get { return SystemInformation.CaptionHeight; }
        }

        /// <summary>
        /// オートバックアップフォルダのパスを取得します。
        /// </summary>
        public static string AutoSaveFolderPath
        {
            get
            {
                // Should be "(Executable Folder)/AutoBackup/"
                string folderPath =
                    System.IO.Path.GetFullPath( System.IO.Path.Combine(TheApp.ApplicationPath, "AutoBackup") );
                if ( Directory.Exists(folderPath)==false )
                {
                    try
                    {
                        Directory.CreateDirectory(folderPath);
                    }
                    catch
                    {
                        OutputLogMsg( NWCore.LogLevels.Error,
                                      res.Strings.WARNING_FAILED_CREATING_AUTOSAVE_FOLDER );
                        return string.Empty;
                    }

                    if ( Directory.Exists(folderPath)==false )
                    {
                        OutputLogMsg( NWCore.LogLevels.Error,
                                      res.Strings.WARNING_FAILED_CREATING_AUTOSAVE_FOLDER );
                        return string.Empty;
                    }
                }

                return folderPath;
            }
        }

        /// <summary>
        /// テンポラリーフォルダのパスを取得します。
        /// </summary>
        public static string TemporaryFolderPath
        {
            get { return _tempFolderPath; }
        }

        /// <summary>
        /// Cache file path for effect browser.
        /// </summary>
        public static string EffectBrowserCachePath { get; private set; }

        /// <summary>
        /// Folder path for unused assets.
        /// </summary>
        public static string UnusedAssetsFolderPath { get; private set; }

        /// <summary>テンポラリーシェーダーパスを取得します。</summary>
        public static string TempShaderFolder
        {
            get { return System.IO.Path.Combine( _tempFolderPath, WorkShaderFolder ); }
        }

        /// <summary>
        /// ＣＧツールのワークフォルダのパスを取得します。
        /// </summary>
        public static string TempCGToolWorkFolderPath
        {
            get { return System.IO.Path.Combine(_tempFolderPath, WorkCGToolFolder); }
        }

        /// <summary>
        /// 通信用テンポラリーフォルダを取得します。
        /// </summary>
        public static string TempMCSWorkFolderPath
        {
            get { return System.IO.Path.Combine(_tempFolderPath, WorkMCSFolder); }
        }

        /// <summary>
        /// G3DXsdフォルダへのパスを取得します。
        /// </summary>
        public static string G3dXsdBasePath
        {
            get
            {
                string path =
                    System.IO.Path.Combine( System.IO.Path.GetDirectoryName(Assembly.GetEntryAssembly().Location),
                                  G3dXsdBaseFodler );

                return Directory.Exists(path) ? path : String.Empty;
            }
        }

        /// <summary>
        /// アプリケーションのパス
        /// </summary>
        public static string ApplicationPath
        {
            get { return _applicationPath; }
        }

        /// <summary>
        /// Working folder path.
        /// </summary>
        public static string WorkingFolder
        {
            get { return _workingFolder; }
        }

        /// <summary>
        /// ヘルプファイルパス。
        /// </summary>
        public static string HelpNamespace
        {
            get
            {
                return _helpFilePath;
            }
        }

        /// <summary>
        /// Releaseでもデバッグメニューを使用するか？
        /// </summary>
        public static bool IsUseDebugMenu { get; set; }

        /// <summary>
        /// コマンドバッファの上限数
        /// </summary>
        public static int LimitCommandCount
        {
            get { return Config.Data.Application.LimitCommandCount; }
        }

        /// <summary>
        /// Check if user is administrator
        /// </summary>
        /// <returns>return true if user is administrator </returns>
        public static bool IsUserAdministrator
        {
            //bool value to hold the return value
            get
            {
                bool isAdmin;
                try
                {
                    //get the currently logged in user
                    WindowsIdentity user = WindowsIdentity.GetCurrent();
                    WindowsPrincipal principal = new WindowsPrincipal(user);
                    isAdmin = principal.IsInRole(WindowsBuiltInRole.Administrator);
                }
                catch (UnauthorizedAccessException)
                {
                    isAdmin = false;
//                    ThreadSafeMsgBox.Show(ex.Message);
                }
                catch (Exception)
                {
                    isAdmin = false;
//                    ThreadSafeMsgBox.Show(ex.Message);
                }

                return isAdmin;
             }
        }

        /// <summary>
        /// 初期設定
        /// </summary>
        public static InitSettingsData InitSettingsData
        {
            get { return _initSettingsData; }
        }


        /// <summary>
        /// Get or set the flag indicating whether gamma correction is enabled.
        /// </summary>
        public static bool IsGammaCorrectionEnabled
        {
            get
            {
                #if BUILD_FOR_CTR
                return false; // Always false for CE2.
                #else
                return Config.Data.Option.GammaCorrection;
                #endif
            }
            set
            {
                #if BUILD_FOR_CTR
                // Always false for CE2.
                Config.Data.Option.GammaCorrection = false;
                NintendoWare.ColorPicker.ColorUtil.IsGammaCorrectionEnabled = false;
                #else
                Config.Data.Option.GammaCorrection = value;
                #endif

                using (new MCSDisableBlock())
                {
                    if (OnGammaCorrectionModeChanged != null)
                    {
                        OnGammaCorrectionModeChanged();
                    }
                }

                TheApp.TriggerConfigChangedEvent();
            }
        }


        /// <summary>
        /// Get or set the file path of the user config file.
        /// </summary>
        public static string UserConfigPath
        {
            get { return Config.Data.Option.UserConfigPath; }
            set
            {
                if ( string.IsNullOrEmpty(value)==true )
                {
                    TheApp.ResetUserConfigData();
                    Config.Data.Option.UserConfigPath = string.Empty;
                    return;
                }

                Config.Data.Option.UserConfigPath = value;

                TheApp.LoadUserConfig( true );
            }
        }


        /// <summary>
        /// Get the user config data.
        /// </summary>
        public static UserConfigData UserConfigData { get; private set; }

        #endregion

        #region マニフェストリソース

        /// <summary>
        /// マニフェストリソースを取得。
        /// </summary>
        public static Stream GetManifestResourceStream(string name)
        {
            string resourceName = "App.res." + name;
            Stream resourceStream = Assembly.GetExecutingAssembly().GetManifestResourceStream(resourceName);
            Debug.Assert(resourceStream != null, string.Format( "マニフェストリソース '{0}' が見つかりません。", resourceName));
            return resourceStream;
        }

        /// <summary>
        /// マニフェストリソースビットマップを取得。
        /// </summary>
        public static Bitmap GetManifestResourceBitmap(string name)
        {
            return new Bitmap(GetManifestResourceStream("bmp." + name));
        }

        /// <summary>
        /// マニフェストリソースアイコンを取得。
        /// </summary>
        public static Icon GetManifestResourceIcon(string name)
        {
            return new Icon(GetManifestResourceStream("ico." + name));
        }

        /// <summary>
        /// マニフェストリソースアイコンを取得。
        /// </summary>
        public static Icon GetManifestResourceIcon(string name, Size size)
        {
            return new Icon(GetManifestResourceStream("ico." + name), size);
        }

        /// <summary>
        /// マニフェストリソースカーソルを取得。
        /// </summary>
        public static Cursor GetManifestResourceCursor(string name)
        {
            return new Cursor(GetManifestResourceStream("cur." + name));
        }

        /// <summary>
        /// マニフェストリソースHTMLを取得。
        /// </summary>
        public static string GetManifestResourceHtml(string name)
        {
            using (StreamReader resourceReader =
                new StreamReader(GetManifestResourceStream("html." + name)))
            {
                return resourceReader.ReadToEnd();
            }
        }

        #endregion

        #region コンフィグ
        /// <summary>
        /// コンフィグ変更通知。
        /// </summary>
        public static void NotifyOptionChanged()
        {
            if (OptionChanged != null)
            {
                OptionChanged(null, EventArgs.Empty);
            }
        }
        #endregion

        #region ユーティリティ

        //---------------------------------------------------------------------
        /// <summary>CRC32 helper</summary>
        //---------------------------------------------------------------------
        public static CRC32 CRC32Helper
        {
            get { return _CRC32Helper; }
        } // End of CRC32Helper for TheApp

        /// <summary>
        /// テンポラリーフォルダへのパス
        /// </summary>
        /// <returns></returns>
        public static string GetTemporaryFolderPath()
        {
            // システムのテンポラリーフォルダへのパスを取得します。
            string folderPath = System.IO.Path.GetTempPath();

            if (String.IsNullOrEmpty(folderPath) == false)
            {
                // ツール名とバージョン番号をコンバインします。
                folderPath = CombineWithToolName(folderPath);
            }
            else
            {
                // "TEMP" の環境変数が取得できないときは、LocalApplicationDataの直下に
                // テンポラリーフォルダを生成します。
                folderPath = GetLocalApplicationDataFolderPath();
            }

            // プロセスＩＤ名のフォルダを生成します。
            folderPath = System.IO.Path.Combine(
                folderPath,
                Process.GetCurrentProcess().Id.ToString("X04"));

            // 注意：RemoveTemporaryFolder() にて１つ上の階層のフォルダから削除するようにしています。
            // ↓ここを修正する場合は注意してください。

            // スレッドＩＤ名のフォルダを生成します。
            folderPath = System.IO.Path.Combine(
                folderPath,
                Thread.CurrentThread.ManagedThreadId.ToString("X04"));

            return System.IO.Path.GetFullPath(folderPath);
        }
        /// <summary>
        /// ツール用の LocalApplicationData フォルダ用パスを取得します
        /// </summary>
        /// <remarks>
        /// 例）"C:\\～\\AppData\\Local\\NintendoWare CreativeStudio\\0.7.0.0"
        /// ツール用設定ファイルなどを保存するためのフォルダを取得します。
        /// </remarks>
        /// <returns>ツール用の LocalApplicationData フォルダへパスを返します。</returns>
        public static string GetLocalApplicationDataFolderPath()
        {
            // LocalApplicationData のフォルダパスを取得します。
            string folderPath = System.Environment.GetFolderPath(
                Environment.SpecialFolder.LocalApplicationData);

            // ツール名とバージョン番号をコンバインします。
            folderPath = CombineWithToolName(folderPath);

            return folderPath;
        }

        /// <summary>
        /// フォルダパスにツール名とバージョン情報をコンパインします。
        /// </summary>
        /// <param name="folderPath">コンバイン元のパスです。後ろに付加されます。</param>
        /// <returns>コンバインされたパスを返します。</returns>
        private static string CombineWithToolName(string folderPath)
        {
            // ツール名をコンバインします。
            string resultPath = System.IO.Path.Combine(
                folderPath, Application.ProductName.Replace(' ','_'));

            // バージョン情報をコンパインします。
            Assembly assemble = Assembly.GetEntryAssembly();
            if (assemble != null)
            {
                FileVersionInfo info = FileVersionInfo.GetVersionInfo(assemble.Location);
                resultPath = System.IO.Path.Combine(resultPath, info.FileVersion);
            }

            return resultPath;
        }


        /// <summary>
        /// テンポラリを生成します。
        /// </summary>
        public static void BuildTemporally()
        {
            if (String.IsNullOrEmpty(_tempFolderPath))
            {
                return;
            }

            Directory.CreateDirectory(TempCGToolWorkFolderPath);
            Directory.CreateDirectory(TempMCSWorkFolderPath);
        }

        /// <summary>
        /// テンポラリをクリア。
        /// </summary>
        public static void ClearTemporally(bool initialize)
        {
            if (!initialize)
            {
                TheApp.ClearDirectory(TemporaryFolderPath);
            }
            TheApp.ClearDirectory(TempMCSWorkFolderPath);
        }

        /// <summary>
        /// ディレクトリをクリア。
        /// </summary>
        public static void ClearDirectory(string dirPath)
        {
            // なければ何もしない
            if (!Directory.Exists(dirPath))
            {
                return;
            }

            // ファイルを削除
            foreach (string filePath in Directory.GetFiles(dirPath))
            {
                try
                {
                    File.Delete(filePath);
                }
                catch (Exception)
                {
                    // 削除できないファイルは残しておく
                }
            }
            // サブディレクトリを削除
            foreach (string subDirPath in Directory.GetDirectories(dirPath))
            {
                try
                {
                    Directory.Delete(subDirPath, true);
                }
                catch (Exception)
                {
                    // 削除できないディレクトリは残しておく
                }
            }
        }

        public static List<EmitterCustomShaderConvertingData> MakeUsdDataList()
        {
            var usdList = new List<EmitterCustomShaderConvertingData>();

            foreach (var usdLink in TheApp.UserShaderUIDefLinkList)
            {
                var usdDef = new EmitterCustomShaderConvertingData
                {
                    Name = System.IO.Path.GetFileNameWithoutExtension(usdLink.FilePath),
                    Visible = usdLink.DefinitionData.Visible,
                    SwitchVisible = usdLink.DefinitionData.RadioBtnGroupDefinition.Visible,
                    ParamVisible = usdLink.DefinitionData.SliderGroupDefinition.Visible,
                };
                foreach (var controlDef in usdLink.DefinitionData.RadioBtnGroupDefinition.ChildGroups)
                {
                    if (controlDef != null)
                    {
                        usdDef.SwitchesVisibleList[controlDef.Index] = controlDef.Visible;
                    }
                }
                foreach (var controlDef in usdLink.DefinitionData.SliderGroupDefinition.ControlDefs)
                {
                    if (controlDef != null)
                    {
                        usdDef.ParamsVisibleList[controlDef.Index] = controlDef.Visible;
                    }
                }

                usdList.Add(usdDef);
            }

            return usdList;
        }

        public static List<CustomActionConvertingData> MakeUddDataList()
        {
            var uddList = new List<CustomActionConvertingData>();

            foreach (var uddData in TheApp.CustomActionUIDefLinkList)
            {
                if (uddData == null)
                {
                    var emptyDef = new CustomActionConvertingData
                    {
                        Name = string.Empty,
                    };
                    uddList.Add(emptyDef);
                    continue;
                }

                var uddDef = new CustomActionConvertingData
                {
                    Name = uddData.Name,
                    Visible = uddData.Visible,
                    FlagsVisible = uddData.CheckBoxGroupDefinition.Visible,
                    IntParamsVisible = uddData.IntSliderGroupDefinition.Visible,
                    FloatParamsVisible = uddData.FloatSliderGroupDefinition.Visible,
                };
                foreach (var controlDef in uddData.CheckBoxGroupDefinition.ControlDefs)
                {
                    if (controlDef != null)
                    {
                        uddDef.FlagsVisibleList[controlDef.Index] = controlDef.Visible;
                    }
                }
                foreach (var controlDef in uddData.IntSliderGroupDefinition.ControlDefs)
                {
                    if (controlDef != null)
                    {
                        uddDef.IntParamsVisibleList[controlDef.Index] = controlDef.Visible;
                    }
                }
                foreach (var controlDef in uddData.FloatSliderGroupDefinition.ControlDefs)
                {
                    if (controlDef != null)
                    {
                        uddDef.FloatParamsVisibleList[controlDef.Index] = controlDef.Visible;
                    }
                }

                uddList.Add(uddDef);
            }

            return uddList;
        }

        #endregion

        #region ヘルプ

        /// <summary>
        /// ヘルプファイルへのパスを取得します。
        /// </summary>
        /// <returns>path</returns>
        private static string GetHelpPath()
        {
            string helpPath =
                System.IO.Path.GetFullPath(System.IO.Path.Combine(Application.StartupPath, HelpFilePath00));
            if (File.Exists(helpPath) == false)
            {
                helpPath =
                    System.IO.Path.GetFullPath(System.IO.Path.Combine(Application.StartupPath, HelpFilePath01));
                if (File.Exists(helpPath) == false)
                {
                    helpPath =
                        System.IO.Path.GetFullPath(System.IO.Path.Combine(Application.StartupPath, HelpFilePath02));
                    if (File.Exists(helpPath) == false)
                    {
                        helpPath =
                            System.IO.Path.GetFullPath(System.IO.Path.Combine(Application.StartupPath, HelpFilePath03));
                        if (File.Exists(helpPath) == false)
                        {
                            helpPath = String.Empty;
                        }
                    }
                }
            }
            return helpPath;
        }

        #endregion

        #region Preset folders

        /// <summary>
        /// Get the preset folder path.
        /// </summary>
        /// <param name="folderKey">The key of the preset folder type.</param>
        /// <returns>The folder path.</returns>
        public static string GetPresetFolder(string folderKey)
        {
            if (folderKey == "Primitive")
                return System.IO.Path.Combine(TheApp.ApplicationPath, "Primitive");

            return string.Empty;
        }

        #endregion

        #region User shader parameter UI definitions

        /// <summary>
        /// Load user shader parameter UI definitions.
        /// </summary>
        /// <returns>True on success.</returns>
        public static bool LoadUserShaderUIDefinitions()
        {
            bool bInitDefaultData = false;
            if ( string.IsNullOrEmpty(TheApp.UserShaderPath)==true )
                bInitDefaultData = true;

            if ( bInitDefaultData==false )
            {
                try
                {
                    if ( File.Exists(TheApp.UserShaderPath)==false )
                    {
                        TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                             res.Strings.ERROR_CUSTOM_SHADER_DEF_LIST_NOT_FOUND,
                                             TheApp.UserShaderPath );
                        bInitDefaultData = true;
                    }
                }
                catch
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_CUSTOM_SHADER_DEF_LIST_NOT_FOUND,
                                         TheApp.UserShaderPath );
                    bInitDefaultData = true;
                }
            }

            // The user shader definition list file is not valid.
            // Just initialize a definition list with default data.
            if ( bInitDefaultData==true )
            {
                _userShaderUIDefList = new CustomShaderUIDefList();
                _userShaderUIDefList.InitDefaultData();

                TheApp.SetupDefaultUserShaderValues();

                return true;
            }

            // Create the XML serializer.
            XmlSerializer serializer =
                new XmlSerializer( typeof(CustomShaderUIDefList) );

            // Load the user shader parameter UI definition file list.
            try
            {
                string workingFolder = Directory.GetCurrentDirectory();
                Directory.SetCurrentDirectory( System.IO.Path.GetDirectoryName(TheApp.UserShaderPath) );

                using ( FileStream stream = new FileStream(TheApp.UserShaderPath, FileMode.Open, FileAccess.Read) )
                {
                    CustomShaderUIDefList data =
                        serializer.Deserialize( stream ) as CustomShaderUIDefList;

                    _userShaderUIDefList = data;
                }

                Directory.SetCurrentDirectory( workingFolder );
            }
            catch ( IOException ex )
            {
                TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                     res.Strings.ERROR_CUSTOM_SHADER_DEF_LIST_FAILED_LOADING,
                                     TheApp.UserShaderPath,
                                     ex.Message );
            }
            catch ( Exception ex )
            {
                string message = string.Empty;
                if ( ex==null )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_CUSTOM_SHADER_DEF_LIST_FAILED_LOADING,
                                         TheApp.UserShaderPath,
                                         string.Empty );
                }
                else if ( ex.InnerException!=null )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_CUSTOM_SHADER_DEF_LIST_FAILED_LOADING,
                                         TheApp.UserShaderPath,
                                         ex.InnerException.Message );
                }
                else if ( ex!=null )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.ERROR_CUSTOM_SHADER_DEF_LIST_FAILED_LOADING,
                                         TheApp.UserShaderPath,
                                         ex.Message );
                }
            }

            if ( _userShaderUIDefList!=null )
            {
                NWCore.NWKernel.SetUserVertexShaderPaths( -1, _userShaderUIDefList.GeneralVertexShaderPaths );
                NWCore.NWKernel.SetUserFragmentShaderPaths( -1, _userShaderUIDefList.GeneralFragmentShaderPaths );
            }

            for ( int i=0;i<MAX_NUM_USER_SHADER_UI_DEFINITIONS;++i )
            {
                UserShaderUIDefinition def = GetUserShaderUIDefinition( i );
                if ( def==null )
                {
                    NWCore.NWKernel.SetUserVertexShaderPaths( i, null );
                    NWCore.NWKernel.SetUserFragmentShaderPaths( i, null );
                }
                else
                {
                    NWCore.NWKernel.SetUserVertexShaderPaths( i, def.VertexShaderPaths );
                    NWCore.NWKernel.SetUserFragmentShaderPaths( i, def.FragmentShaderPaths );
                }
            }

            TheApp.SetupDefaultUserShaderValues();

            return true;
        }


        /// <summary>
        /// Setup the default values of the user shader parameters.
        /// </summary>
        private static void SetupDefaultUserShaderValues()
        {
            int i;

            // Create data structures for storing the default parameters.
            DefaultUserShaderParameter[] defaultParams =
                new DefaultUserShaderParameter[TheApp.MAX_NUM_USER_SHADER_UI_DEFINITIONS];
            for ( i=0;i<defaultParams.Length;++i )
                defaultParams[i] = new DefaultUserShaderParameter();

            // Was the user shader definition being successfully loaded?
            if ( _userShaderUIDefList==null )
            {
                NWCore.NWKernel.SetDefaultUserShaderParams( defaultParams );
                return;
            }

            // Parse the user shader definitions for the default values.
            foreach ( CustomShaderUIDefLinkData data in _userShaderUIDefList.DefinitionList )
            {
                UserShaderUIDefinition def = data.DefinitionData;
                if ( def==null ||
                     def.CallbackId<0 ||
                     def.CallbackId>=defaultParams.Length )
                {
                    continue;
                }

                DefaultUserShaderParameter param = defaultParams[data.CallbackId];

                // Parse the default value of the bit flags ( check boxes ).
                foreach ( BaseUserUIControlDef ctrlDef in def.CheckBoxGroupDefinition.ControlDefs )
                {
                    CheckBoxUserUIDef chkDef = ctrlDef as CheckBoxUserUIDef;
                    if ( chkDef!=null &&
                         chkDef.Index>=0 &&
                         chkDef.Index<param.BitFlags.Length )
                    {
                        param.BitFlags[chkDef.Index] = chkDef.DefaultValue;
                    }
                }

                // Parse the default value of the byte flags ( radio button groups ).
                foreach ( UserShaderUIDefinition.UIGroupDefinition grpDef in def.RadioBtnGroupDefinition.ChildGroups )
                {
                    if ( grpDef==null ||
                         grpDef.Index<0 ||
                         grpDef.Index>=4 )
                    {
                        continue;
                    }

                    foreach ( BaseUserUIControlDef ctrlDef in grpDef.ControlDefs )
                    {
                        RadioButtonUserUIDef rbDef = ctrlDef as RadioButtonUserUIDef;
                        if ( rbDef!=null &&
                             rbDef.DefaultValue==true &&
                             rbDef.Index>=0 &&
                             rbDef.Index<8 )
                        {
                            param.Bytes[grpDef.Index] =
                                (uint)0x00000001 << (rbDef.Index + (grpDef.Index * 8));

                            // Only the first radio button with default value set to true counts.
                            break;
                        }
                    }
                }

                // Parse the default value of the floating point parameters ( sliders ).
                foreach ( BaseUserUIControlDef ctrlDef in def.SliderGroupDefinition.ControlDefs )
                {
                    FloatSliderUserUIDef sldDef = ctrlDef as FloatSliderUserUIDef;
                    if ( sldDef!=null &&
                         sldDef.Index>=0 &&
                         sldDef.Index<param.Params.Length )
                    {
                        param.Params[sldDef.Index] = sldDef.DefaultValue;
                    }
                }
            }

            NWCore.NWKernel.SetDefaultUserShaderParams( defaultParams );
        }


        /// <summary>
        /// Get user shader parameter UI definition.
        /// </summary>
        /// <param name="iIndex">The index to the definition.</param>
        /// <returns>The user shader parameter UI definition.</returns>
        public static UserShaderUIDefinition GetUserShaderUIDefinition( int iIndex )
        {
            if ( _userShaderUIDefList==null ||
                 iIndex<0 ||
                 iIndex>=_userShaderUIDefList.DefinitionList.Count )
            {
                return null;
            }

            CustomShaderUIDefLinkData data = _userShaderUIDefList.DefinitionList[iIndex];
            if ( data==null || data.DefinitionData==null )
                return null;

            return data.DefinitionData;
        }

        #endregion

        #region User data UI definitions

        /// <summary>
        /// Load user data UI definitions.
        /// </summary>
        /// <returns>True on success.</returns>
        public static bool LoadUserDataUIDefinitions()
        {
            // First reset all the user data UI definitions.
            for ( int i=0;i<_userDataUIDefList.Length;++i )
            {
                if ( _userDataUIDefList[i]==null )
                    _userDataUIDefList[i] = new UserDataUIDefinition();

                UserDataUIDefinition def = _userDataUIDefList[i];
                def.Init( i - 1 );
            }

            if (string.IsNullOrEmpty(TheApp.UserDataPath) == false)
            {
                // Compose the directory path for the user data UI definition files.
                string dirPath = TheApp.UserDataPath;
                if ( Directory.Exists(dirPath)==true )
                {
                    // Search for "*.udd" files in the folder.
                    IEnumerable<string> filePaths =
                        Directory.EnumerateFiles( dirPath, "*" + TheApp.UserDataParamUIDefFileExt );

                    // Create the XML serializer.
                    XmlSerializer serializer =
                        new XmlSerializer( typeof( UserDataUIDefinition ) );

                    // Load the user data parameter UI definition files.
                    foreach ( string path in filePaths )
                    {
                        try
                        {
                            using ( FileStream stream = new FileStream(path, FileMode.Open, FileAccess.Read) )
                            {
                                UserDataUIDefinition def =
                                    serializer.Deserialize( stream ) as UserDataUIDefinition;

                                int iIndex = def.Index + 1;
                                if ( def!=null &&
                                     iIndex>=0 &&
                                     iIndex<TheApp.MAX_NUM_USER_DATA_UI_DEFINITIONS )
                                {
                                    _userDataUIDefList[iIndex] = def;
                                    _userDataUIDefList[iIndex].Name = System.IO.Path.GetFileNameWithoutExtension(path);
                                }
                            }
                        }
                        catch ( System.IO.IOException ex )
                        {
                            string message = string.Format( res.Strings.ERROR_CUSTOM_ACTION_FAILED_LOADING,
                                                            System.IO.Path.GetFileName(path),
                                                            ex.Message );

                            DebugConsole.WriteLine( message );
                            TheApp.Logger.Error.AddMessage( message );
                        }
                        catch ( Exception ex )
                        {
                            string message = string.Empty;
                            if ( ex==null )
                            {
                                message = string.Format( res.Strings.ERROR_USERSHADER_UI_DEF_XML_PARSING_FAILED,
                                                         System.IO.Path.GetFileName(path),
                                                         string.Empty );
                            }
                            else if ( ex.InnerException is UserDataUIDefinition.XmlReadException )
                            {
                                message = string.Format( res.Strings.ERROR_USERDATA_DEF_XML_INDEX_INVALID,
                                                         System.IO.Path.GetFileName(path) );
                            }
                            else if ( ex.InnerException!=null )
                            {
                                message = string.Format( res.Strings.ERROR_USERSHADER_UI_DEF_XML_PARSING_FAILED,
                                                         System.IO.Path.GetFileName(path),
                                                         ex.InnerException.Message );
                            }
                            else if ( ex!=null )
                            {
                                message = string.Format( res.Strings.ERROR_CUSTOM_ACTION_FAILED_LOADING,
                                                         System.IO.Path.GetFileName(path),
                                                         ex.Message );
                            }

                            if ( string.IsNullOrEmpty(message)==false )
                            {
                                DebugConsole.WriteLine( message );
                                TheApp.Logger.Error.AddMessage( message );
                            }
                        }
                    }
                }
            }

            TheApp.SetupDefaultUserDataValues();

            ////// Update layout for the property pages to reflect the UI definitions.
            ////UIManager.TriggerLayoutChangedEvent( null );

            return true;
        }


        /// <summary>
        /// Setup the default values of the user data parameters.
        /// </summary>
        private static void SetupDefaultUserDataValues()
        {
            int i;

            // Create data structures for storing the default parameters.
            DefaultEmitterUserData[] defaultParams = new DefaultEmitterUserData[MAX_NUM_USER_DATA_UI_DEFINITIONS];
            for ( i=0;i<defaultParams.Length;++i )
                defaultParams[i] = new DefaultEmitterUserData();

            // Was the user data definition being successfully loaded?
            if ( _userDataUIDefList==null )
            {
                NWCore.NWKernel.SetDefaultEmitterUserData( defaultParams );
                return;
            }

            // Parse the user data definitions for the default values.
            foreach ( UserDataUIDefinition def in _userDataUIDefList )
            {
                int iIndex = def.Index + 1;
                if ( def==null ||
                     iIndex<0 ||
                     iIndex>=defaultParams.Length )
                {
                    continue;
                }

                DefaultEmitterUserData param = defaultParams[iIndex];

                // Parse the default value of the bit flags ( check boxes ).
                foreach ( BaseUserUIControlDef ctrlDef in def.CheckBoxGroupDefinition.ControlDefs )
                {
                    CheckBoxUserUIDef chkDef = ctrlDef as CheckBoxUserUIDef;
                    if ( chkDef!=null &&
                         chkDef.Index>=0 &&
                         chkDef.Index<param.BitFlags.Length )
                    {
                        param.BitFlags[chkDef.Index] = chkDef.DefaultValue;
                    }
                }

                // Parse the default value of the integer sliders.
                foreach ( BaseUserUIControlDef ctrlDef in def.IntSliderGroupDefinition.ControlDefs )
                {
                    IntSliderUserUIDef sldDef = ctrlDef as IntSliderUserUIDef;
                    if ( sldDef!=null &&
                         sldDef.Index>=0 &&
                         sldDef.Index<param.ParamsI.Length )
                    {
                        param.ParamsI[sldDef.Index] = (uint)( sldDef.DefaultValue<0 ? 0 : sldDef.DefaultValue );
                    }
                }

                // Parse the default value of the integer sliders.
                foreach ( BaseUserUIControlDef ctrlDef in def.FloatSliderGroupDefinition.ControlDefs )
                {
                    FloatSliderUserUIDef sldDef = ctrlDef as FloatSliderUserUIDef;
                    if ( sldDef!=null &&
                         sldDef.Index>=0 &&
                         sldDef.Index<param.ParamsF.Length )
                    {
                        param.ParamsF[sldDef.Index] = sldDef.DefaultValue;
                    }
                }
            }

            NWCore.NWKernel.SetDefaultEmitterUserData( defaultParams );
        }


        /// <summary>
        /// Get user data parameter UI definition.
        /// </summary>
        /// <param name="iIndex">The index to the definition.</param>
        /// <returns>The user data parameter UI definition.</returns>
        public static UserDataUIDefinition GetUserDataUIDefinition( int iIndex )
        {
            iIndex += 1; // iIndex = -1, 0 ~ 7, make it 0 ~ 8
            if ( _userDataUIDefList==null ||
                 iIndex<0 ||
                 iIndex>=_userDataUIDefList.Length )
            {
                return null;
            }

            return _userDataUIDefList[iIndex];
        }

        #endregion

        #region User config

        /// <summary>
        /// Event triggered when the user config has been changed.
        /// </summary>
        public static event EventHandler UserConfigChanged;


        /// <summary>
        /// Trigger UserConfigChanged event.
        /// </summary>
        public static void TriggerUserConfigChangedEvent()
        {
            if ( UserConfigChanged==null )
                return;

            TheApp.ResetUpdateDrawPathDialogFlags();

            UserConfigChanged( null, EventArgs.Empty );

            // Reset the flags again just to be safe
            // if someone use these flags without resetting the flags.
            TheApp.ResetUpdateDrawPathDialogFlags();
        }


        /// <summary>
        /// Reset the update draw path dialog flags.
        /// </summary>
        public static void ResetUpdateDrawPathDialogFlags()
        {
            _bSkipUpdateDrawPathWarningDialog = false;
            _bUpdateDrawPath                  = false;
        }


        /// <summary>
        /// Set the action for all the dialog asking user whether to update the draw path.
        /// </summary>
        /// <param name="bUpdateDrawPath">Flag indicating whether to update the draw path.</param>
        public static void SetUpdateDrawPathDialogAction( bool bUpdateDrawPath )
        {
            _bSkipUpdateDrawPathWarningDialog = true;
            _bUpdateDrawPath                  = bUpdateDrawPath;
        }


        /// <summary>
        /// Get the flag indicating whether should skip the update draw path dialog.
        /// </summary>
        /// <returns>True to skip.</returns>
        public static bool ShouldSkipUpdateDrawPathDialog()
        {
            return _bSkipUpdateDrawPathWarningDialog;
        }


        /// <summary>
        /// Get the flag indicating whether should update the draw path.
        /// </summary>
        /// <returns>True to update.</returns>
        public static bool ShouldUpdateDrawPath()
        {
            return _bUpdateDrawPath;
        }


        /// <summary>
        /// Load user config file to the application.
        /// </summary>
        /// <param name="bSilent">True to prevent any log message while loading.</param>
        /// <returns>True on success.</returns>
        public static bool LoadUserConfig( bool bSilent )
        {
            TheApp.UserConfigData = LoadUserConfig( TheApp.UserConfigPath, bSilent );
            if ( TheApp.UserConfigData==null )
            {
                TheApp.ResetUserConfigData();
                return false;
            }

            // Set the default value to data model kernel.
            NWCore.NWKernel.DefaultDrawPath              = TheApp.UserConfigData.DefaultDrawPath;
            NWCore.NWKernel.DefaultDrawPathID            = TheApp.UserConfigData.DefaultDrawPathID;
            NWCore.NWKernel.DefaultUserShaderCompileDef1 = TheApp.UserConfigData.DefaultShaderCompileDef1;
            NWCore.NWKernel.DefaultUserShaderCompileDef2 = TheApp.UserConfigData.DefaultShaderCompileDef2;

            // Trigger UserConfigChanged event.
            TriggerUserConfigChangedEvent();

            return true;
        }


        /// <summary>
        /// Load the user config file.
        /// </summary>
        /// <param name="path">The directory or file path to the user config file.</param>
        /// <param name="bSilent">True to prevent any log message while loading.</param>
        /// <returns>The loaded user config data or null on failure.</returns>
        public static UserConfigData LoadUserConfig( string path,
                                                     bool bSilent )
        {
            if ( string.IsNullOrEmpty(path)==true )
                return null;

            string filePath;
            if ( Directory.Exists(path)==true )
                filePath = System.IO.Path.Combine( path, TheApp.UserConfigFileName );
            else
                filePath = path;

            if ( System.IO.Path.IsPathRooted(filePath)==false )
            {
                if ( bSilent==false )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         StringResource.Get("WARNING_USER_CONFIG_FILE_DOESNT_EXIST",
                                                            filePath) );
                }
                return null;
            }

            UserConfigData data = null;

            try
            {
                if ( string.IsNullOrEmpty(filePath)==true )
                    return null;

                // Check if the file exists.
                if ( File.Exists(filePath)==false )
                {
                    if ( bSilent==false )
                    {
                        TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                             StringResource.Get("WARNING_USER_CONFIG_FILE_DOESNT_EXIST",
                                                                filePath) );
                    }
                    return null;
                }

                // Load the user config file with the serializer.
                using ( FileStream stream = new FileStream(filePath, FileMode.Open, FileAccess.Read) )
                {
                    XmlSerializer serializer = new XmlSerializer( typeof(UserConfigData) );

                    // Deserialize
                    data = serializer.Deserialize( stream ) as UserConfigData;

                    stream.Close();

                    if ( data==null )
                    {
                        if ( bSilent==false )
                        {
                            TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                                 StringResource.Get("WARNING_USER_CONFIG_FILE_FAIL_LOADING",
                                                                    filePath,
                                                                    res.Strings.ERRMSG_USER_CONFIG_FILE_DRAW_PATH_EMPTY) );
                        }
                        return null;
                    }
                    else if ( data.DrawPaths==null )
                    {
                        // Just give it an empty draw path list.
                        data.DrawPaths = new UserConfigData.DrawPathData[0];

                        data.DefaultDrawPathID        = -1;
                        data.DefaultShaderCompileDef1 = string.Empty;
                        data.DefaultShaderCompileDef2 = string.Empty;

                        return data;
                    }
                }
            }
            catch ( Exception ex )
            {
                if ( bSilent==false )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         StringResource.Get("WARNING_USER_CONFIG_FILE_FAIL_LOADING",
                                                            filePath,
                                                            ex.Message) );
                }
                return null;
            }

            // The maximum number of draw paths is limited to 32.
            const int iMaxNumDrawPaths = 32;
            if ( data.DrawPaths.Length>iMaxNumDrawPaths )
            {
                if ( bSilent==false )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Warning,
                                         StringResource.Get("WARNING_USER_CONFIG_FILE_DRAW_PATH_EXCEED_MAX_NUM") );
                }

                UserConfigData.DrawPathData[] origArray = data.DrawPaths;
                UserConfigData.DrawPathData[] newArray  = new UserConfigData.DrawPathData[iMaxNumDrawPaths];
                for ( int i=0;i<iMaxNumDrawPaths;++i )
                {
                    newArray[i] = origArray[i];
                }

                data.DrawPaths = newArray;
            }

            // Get the default draw path and its ID.
            int    iDefaultID               = 0;
            string defaultPath              = string.Empty;
            string defaultShaderCompileDef1 = string.Empty;
            string defaultShaderCompileDef2 = string.Empty;
            if ( string.IsNullOrEmpty(data.DefaultDrawPath)==true )
            {
                if ( bSilent==false )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         StringResource.Get("WARNING_USER_CONFIG_FILE_INVALID_DEFAULT_PATH",
                                                            filePath,
                                                            res.Strings.ERRMSG_USER_CONFIG_FILE_DRAW_PATH_EMPTY) );
                }
                return null;
            }
            else
            {
                defaultPath = data.DefaultDrawPath;

                UserConfigData.DrawPathData pathData =
                    data.FindDrawPathByText(defaultPath);
                if ( pathData==null )
                {
                    if ( bSilent==false )
                    {
                        TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                             StringResource.Get("WARNING_USER_CONFIG_FILE_INVALID_DEFAULT_PATH",
                                                                filePath,
                                                                res.Strings.ERRMSG_USER_CONFIG_FILE_DRAW_PATH_EMPTY) );
                    }
                    return null;
                }

                iDefaultID               = pathData.ID;
                defaultShaderCompileDef1 = pathData.ShaderCompileDef1;
                defaultShaderCompileDef2 = pathData.ShaderCompileDef2;
            }

            data.DefaultDrawPathID        = iDefaultID;
            data.DefaultShaderCompileDef1 = defaultShaderCompileDef1;
            data.DefaultShaderCompileDef2 = defaultShaderCompileDef2;

            // Check the user shader compile definitions.
            bool bBadShaderCompileDefStr = false;
            foreach ( UserConfigData.DrawPathData pathData in data.DrawPaths )
            {
                // Shader compile definition 1
                if ( pathData.ShaderCompileDef1==null )
                {
                    pathData.ShaderCompileDef1 = string.Empty;
                }
                else if ( pathData.ShaderCompileDef1.Length>16 )
                {
                    // Longer than 16 chars?
                    pathData.ShaderCompileDef1 = string.Empty;
                    bBadShaderCompileDefStr = true;
                    break;
                }
                else
                {
                    // Any non-ASCII characters?
                    foreach ( char ch in pathData.ShaderCompileDef1 )
                    {
                        if ( ch<0 || ch>127 )
                        {
                            bBadShaderCompileDefStr = true;
                            break;
                        }
                    }

                    if ( bBadShaderCompileDefStr==true )
                    {
                        pathData.ShaderCompileDef1 = string.Empty;
                        break;
                    }
                }

                // Shader compile definition 2
                if ( pathData.ShaderCompileDef2==null )
                {
                    pathData.ShaderCompileDef2 = string.Empty;
                }
                else if ( pathData.ShaderCompileDef2.Length>16 )
                {
                    // Longer than 16 chars?
                    pathData.ShaderCompileDef2 = string.Empty;
                    bBadShaderCompileDefStr = true;
                    break;
                }
                else
                {
                    // Any non-ASCII characters?
                    foreach ( char ch in pathData.ShaderCompileDef2 )
                    {
                        if ( ch<0 || ch>127 )
                        {
                            bBadShaderCompileDefStr = true;
                            break;
                        }
                    }

                    if ( bBadShaderCompileDefStr==true )
                    {
                        pathData.ShaderCompileDef2 = string.Empty;
                        break;
                    }
                }
            }

            // Do we have any bad formatted shader compile def strings?
            if ( bBadShaderCompileDefStr==true )
            {
                TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                     res.Strings.WARNING_USER_CONFIG_FILE_INVALID_USER_SHADER_COMPILE_DEF );
            }

            return data;
        }


        /// <summary>
        /// Reset the user config data to default values.
        /// </summary>
        public static void ResetUserConfigData()
        {
            TheApp.UserConfigData = null;

            // Set the default value to data model kernel.
            NWCore.NWKernel.DefaultDrawPath              = string.Empty;
            NWCore.NWKernel.DefaultDrawPathID            = 0;
            NWCore.NWKernel.DefaultUserShaderCompileDef1 = string.Empty;
            NWCore.NWKernel.DefaultUserShaderCompileDef2 = string.Empty;

            // Trigger UserConfigChanged event.
            TriggerUserConfigChangedEvent();
        }

        #endregion

        #region Config changed event

        /// <summary>
        /// Event triggered when config has been changed.
        /// </summary>
        public static event EventHandler ConfigChanged = null;


        /// <summary>
        /// The method should be called when you changed the config data
        /// and want to notify others.
        /// </summary>
        public static void TriggerConfigChangedEvent()
        {
            if ( ConfigChanged!=null )
                ConfigChanged( null, EventArgs.Empty );
        }

        #endregion

        #region Main

        /// <summary>
        /// アプリケーションのメインエントリポイント。
        /// </summary>
        [STAThread]
        public static int Main(string[] args)
        {
            int result = 0;

            TheApp.EnableConsoleModeWarningDialogs = false;

#if CHECK_INIT_ERROR
            int iCheckPointID = 0;
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : App.TheApp.Main()", "調査", MessageBoxButtons.OK);
#endif
            _workingFolder   = Environment.CurrentDirectory;
            _applicationPath = System.IO.Path.GetDirectoryName( Application.ExecutablePath );

            Environment.CurrentDirectory = _applicationPath;

            EffectBrowserCachePath = System.IO.Path.Combine( _applicationPath,
                                                   "Cache\\EffectBrowser.cache" );

            UnusedAssetsFolderPath = System.IO.Path.Combine( _applicationPath,
                                                   "UnUsed\\" );

            DefaultPresetFolderPath = System.IO.Path.Combine(_applicationPath, "Preset\\");

            // VisualStyleを有効にする（.manifestは必要なし）
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);
            Application.DoEvents();

            // CRC32 helper
            _CRC32Helper = new CRC32();
            _CRC32Helper.Initialize();

            // Initialize NW4F kernel
            NWCore.NWKernel.Initialize();
            NWCore.NWKernel.RegisterLogHandler( new NWCore.LogHandler(OutputLogMsg) );

            // Are we going to run at command line mode?
            ConsoleApplication.InitArgumentList();
            TheApp.ConsoleMode = ConsoleApplication.CheckArguments( args );

            // Create texture manager.
            _textureManager = new TextureManager();
            if ( TheApp.ConsoleMode==false )
            {
                _textureManager.IsCheckModTimeEnabled = true;
            }

            _associationManager = new AssociationManager();
            if ( TheApp.ConsoleMode==false && args.Length>0 )
            {
                // if another instance of EffectMaker already exists,
                // use it to open the file
                if (_associationManager.SendFilePath(args) == true)
                    return 1;
            }

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : new AssociationManager()", "調査", MessageBoxButtons.OK);
#endif

            // ＵＩ言語初期化
            string enDllFnm =
                Application.StartupPath + String.Format(
                    LocalizeDLL_English,
                    System.IO.Path.GetFileNameWithoutExtension(Application.ExecutablePath));
            if (File.Exists(enDllFnm))
            {
#if true
                CultureInfo cultureInfo = CultureInfo.CreateSpecificCulture("en-us");
                Thread.CurrentThread.CurrentCulture = cultureInfo;
                Thread.CurrentThread.CurrentUICulture = cultureInfo;
#else
                Thread.CurrentThread.CurrentUICulture = new CultureInfo(LocalizeDir_English, false);
#endif
                _englishUI = true;
            }

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : LocalizeDLL", "調査", MessageBoxButtons.OK);
#endif

            // ロガー
            _logger = new Logger4C();

            // コンフィグの初期化
            if (Config.Initialize() == false)
            {
                // エラー：コンフィグファイルが読み込めない
                string filePath = Config.ConfigFilePath;
                ShellUtility.OpenPathByExplorer(filePath);
                return 1;
            }


#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : Config.Initialize()", "調査", MessageBoxButtons.OK);
#endif
            // 初期ファイルの初期化
            if (Config.Data.DocumentIO != null &&
                String.IsNullOrEmpty(Config.Data.DocumentIO.InitPath))
            {
                // デフォルトのファイル名を登録
                Config.Data.DocumentIO.InitPath =
                    System.IO.Path.GetFileNameWithoutExtension( Application.ExecutablePath ) + InitFileExt;
            }

            // バイナリー化をCreativeStudioConsoleを利用するかフラグを設定します。
            IsUseDebugMenu =
                Config.Data.Application.IsUseDebugMenu;

            // メインスレッドの取得
            _mainThread = Thread.CurrentThread;

            // ヘルプファイルのパスを設定
            _helpFilePath = GetHelpPath();

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : GetHelpPath()", "調査", MessageBoxButtons.OK);
#endif
            // テンポラリフォルダを設定します。
            _tempFolderPath = GetTemporaryFolderPath();
            TheApp.BuildTemporally();

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : CheckArguments()", "調査", MessageBoxButtons.OK);
#endif

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : HelpManager.Initialize()", "調査", MessageBoxButtons.OK);
#endif

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : new CRC32()", "調査", MessageBoxButtons.OK);
#endif
            // Animation table manager
            _animTableManager = new App.Data.AnimationTableManager();

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : new AnimationTableManager()", "調査", MessageBoxButtons.OK);
#endif
            // Data model updater manager
            _dataModelUpdaterManager = new App.Data.DataModelUpdaterManager();

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : new DataModelUpdaterManager()", "調査", MessageBoxButtons.OK);
#endif

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : new UIManager()", "調査", MessageBoxButtons.OK);
#endif

#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : new FavoriteDirectoryManager()", "調査", MessageBoxButtons.OK);
#endif
            // デバッグコンソール作成
            DebugConsole.Initialize(null);
#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : DebugConsole.Initialize()", "調査", MessageBoxButtons.OK);
#endif
            DocumentManager.Initialize();
#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : DocumentManager.Initialize()", "調査", MessageBoxButtons.OK);
#endif
            ProjectManager.Initialize();
#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : ProjectManager.Initialize()", "調査", MessageBoxButtons.OK);
#endif
#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : NWCore.Converter.Main.Initialize()", "調査", MessageBoxButtons.OK);
#endif
#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : NWModelLoader.Init()", "調査", MessageBoxButtons.OK);
#endif
            // 初期ファイルの読み込み
            LoadInitFile();
#if CHECK_INIT_ERROR
            ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : LoadInitFile()", "調査", MessageBoxButtons.OK);
#endif

            #if BUILD_FOR_CTR
            IconManager.SetFileAssociatedIcon(DocumentConstants.DotCmdl, Document.DocumentIconImageList[DocumentConstants.Cmdl]);
            #else
            #endif

            TheApp.IsGammaCorrectionEnabled = TheApp.IsGammaCorrectionEnabled;

            // Now we know the application wasn't ran on command line,
            // it's time for us to apply the cache limits for the
            // texture manager.
            TheApp.TextureManager.MaxTextureCacheSize = Config.Data.Option.MaxTextureCacheSize;

#if ENABLE_EFFECT_COMBINER_EDITOR
            _eftCombinerCommManager = new EffectCombinerCommunicationManager();
            _eftCombinerCommManager.Init();
#endif

            // Initialize log message mapper.
            App.Global.LogMessageMapper.Init();

            // コンソール実行
            if ( TheApp.ConsoleMode==true )
            {
                result = ConsoleApplication.Execute(args);
            }
/*            else
            {
                // 引数は全てファイル名とする
                foreach ( string arg in args )
                {
                    if ( arg.IndexOf( "-Connect=" )==0 )
                        _connectTarget = arg.Substring( 9 );
                    else
                        _openFiles.Add( arg );
                }

                // ＧＵＩ処理
                result = WinMain( args );
            }*/

#if ENABLE_EFFECT_COMBINER_EDITOR
            _eftCombinerCommManager.Deinit();
#endif

            // 初期ファイルの読み込み
//            SaveInitFile();
            // デバッグコンソール終了
            DebugConsole.Terminate();

            // テンポラリの消去
            RemoveTemporaryFolder(_tempFolderPath);
            // コンフィグ後始末
            Config.Cleanup();

            return result;
        }

        #endregion

        #region WinMain

        /// <summary>
        /// ＧＵＩ処理開始。
        /// </summary>
 /*       private static int WinMain(string[] args)
        {
            // スレッド例外を補足する
            Application.ThreadException += new ThreadExceptionEventHandler(Event_Application_ThreadException);
            int result = 0;
            bool hasProblemMCS = false;

#if !DEBUG
            try
            {
#endif
#if USE_CSCONSOLE
                // Creative Studio を初期化
                App.IO.CreativeStudio.Instanse.Initialize();
#endif
                // ＧＵＩフォント作成
                _guiFont = new Font("Tahoma", 8.25F, FontStyle.Regular, GraphicsUnit.Point, 0);
                _guiFontBold = new Font(_guiFont, FontStyle.Bold);
                _guiFixedFont = new Font("MS UI Gothic", 9.0f, FontStyle.Regular, GraphicsUnit.Point, 0);
                _guiFixedFontBold = new Font(_guiFixedFont, FontStyle.Bold);

#if CHECK_INIT_ERROR
                ThreadSafeMsgBox.Show("Init Level " + (iCheckPointID++) + " : new Font()", "調査", MessageBoxButtons.OK);
#endif

                // メインフレーム作成
                hasProblemMCS = true;
                _mainFrame = new MainFrame( _openFiles.ToArray(),
                                            _connectTarget );
                hasProblemMCS = false;

                // テンプレートファイルのチェック
//                CheckTemplateFiles(_openFiles);

                // メインフレームクライアントの作成
                // ・MainFrame のインスタンス作成後
                // ・プロファイルの読み込み後
                // に行う事！
                _mainFrame.InitializeClientControl();

                // Load user defined UI
                UIManager.LoadUIPlugins();

                // メッセージループ開始
                Application.Run(_mainFrame);
#if !DEBUG
            }
            catch (Exception error)
            {
                if (hasProblemMCS)
                {
                    // 必要なインストールがされていないときの警告メッセージを表示します。
                    ThreadSafeMsgBox.Show( res.Strings.WARNING_INSTALL,
                                           res.Strings.WARNING_CAPTION, MessageBoxButtons.OK);
                }
                else
                {
                    // エラーダイアログを表示します。
                    OnFatalErrorOccured(error, false);
                }
                result = 1;
            }
            finally
            {
            }
#endif
#if USE_CSCONSOLE
            // 子プロセスを終了させます。
            App.IO.CreativeStudio.Instanse.Dispose();
#endif
                // MCSマネージャを終了させます
            if (hasProblemMCS == false)
            {
                MCSManager.Dispose();
            }

            // テンポラリの消去
            TheApp.ClearTemporally(false);

            return result;
        }*/

        #endregion

        #region 初期ファイル

        static void InitFileInitialize()
        {
            _initSettingsData = new InitSettingsData();
        }

        /// <summary>
        /// 初期ファイルの設定
        /// </summary>
        static void LoadInitFile()
        {
            if (String.IsNullOrEmpty(Config.Data.DocumentIO.InitPath))
            {
                Logger.Error.AddMessage(
                    String.Format(
                        res.Strings.ERROR_CANNOT_SAVE_INITFILE,
                        Config.Data.DocumentIO.InitPath));
                return;
            }

            string fullPath = System.IO.Path.Combine( ApplicationPath,
                                            System.IO.Path.GetFileName( Config.Data.DocumentIO.InitPath ) );

            if ( File.Exists( fullPath ) == false )
            {
                // 初期のビューアプロセス
                InitFileInitialize();

                // 初期ファイルの生成
//                SaveInitFile();
            }
            else
            {
                // 初期ファイルの読み込み
                _initSettingsData = InitSettingsData.LoadFrom( fullPath );
                if (_initSettingsData == null)
                {
                    // 初期のビューアプロセス
                    InitFileInitialize();

                    // ログ
                    Logger.Error.AddMessage( String.Format( res.Strings.ERROR_CANNOT_LOAD_INITFILE,
                                                            fullPath ) );
                }
            }
        }

        /// <summary>
        /// 初期設定ファイルに書き込み
        /// </summary>
        static void SaveInitFile()
        {
            if (String.IsNullOrEmpty(Config.Data.DocumentIO.InitPath))
            {
                Logger.Error.AddMessage(
                    String.Format(
                        res.Strings.ERROR_CANNOT_SAVE_INITFILE,
                        Config.Data.DocumentIO.InitPath));
                return;
            }

            string fullPath = System.IO.Path.Combine( ApplicationPath,
                                            System.IO.Path.GetFileName( Config.Data.DocumentIO.InitPath ) );
            if ( InitSettingsData.SaveTo( fullPath, _initSettingsData )==false )
            {
                Logger.Error.AddMessage( String.Format( res.Strings.ERROR_CANNOT_SAVE_INITFILE,
                                                        fullPath ) );
            }
        }

        #endregion

        #region テンポラリーフォルダ関係

        /// <summary>
        /// テンポラリーフォルダを削除します。
        /// </summary>
        /// <remarks>
        /// プロセスＩＤのフォルダ以下を削除します。
        /// </remarks>
        /// <returns>通常は true を返し、削除に失敗した場合は false を返します。</returns>
        private static bool RemoveTemporaryFolder(string tempFolder)
        {
            if (String.IsNullOrEmpty(tempFolder))
            {
                return false;
            }

            // プロセスＩＤのフォルダから下を削除します。
            tempFolder = System.IO.Path.GetDirectoryName(tempFolder);
            if (System.IO.Directory.Exists(tempFolder) == true)
            {
                string strProccesID = Process.GetCurrentProcess().Id.ToString("X04");

                // １つ上のフォルダがプロセスＩＤのフォルダになっているかチェックをします。
                if (tempFolder.EndsWith(
                    strProccesID,
                    StringComparison.CurrentCultureIgnoreCase) == true)
                {
                    // プロセスＩＤのフォルダ以下を削除します。
                    try
                    {
                        System.IO.Directory.Delete(tempFolder, true);
                    }
                    catch
                    {
                        // エラーが発生して削除に失敗した場合です。
                        return false;
                    }
                }
            }

            return true;
        }

        #endregion

        #region Log & exception handling

        /// <summary>
        /// Output log message.
        /// </summary>
        /// <param name="level">The log level.</param>
        /// <param name="msg">The message to log out.</param>
        /// <param name="args">Message formatting arguments.</param>
        public static void OutputLogMsg( NWCore.LogLevels level,
                                         string msg,
                                         params object[] args )
        {
            if ( string.IsNullOrEmpty(msg)==true )
                return;

            if ( (level & NWCore.LogLevels.MsgForMapping)==NWCore.LogLevels.MsgForMapping )
            {
                // Remove LogLevels.MsgForMapping flag first.
                level &= ~NWCore.LogLevels.MsgForMapping;

                // Convert the message to the mapped string.
                msg = LogMessageMapper.Map( msg );
            }

            string output = msg;
            if ( args!=null && args.Length>0 )
                output = string.Format( msg, args );

            if ( TheApp.ConsoleMode==true )
            {
                TheApp.OutputConsoleMsg( level, output );
                return;
            }

            if ( level==NWCore.LogLevels.Fatal )
            {
                OutputExceptionLog( output );
            }
            if ( TheApp.Logger==null )
                return;

            switch ( level )
            {
                case NWCore.LogLevels.Debug:
                    DebugConsole.WriteLine(output);
                    TheApp.Logger.Debug.AddMessage(output);
                    break;

                case NWCore.LogLevels.Information:
                    TheApp.Logger.Info.AddMessage(output);
                    break;

                case NWCore.LogLevels.Warning :
                    TheApp.Logger.Warn.AddMessage(output);
                    break;

                case NWCore.LogLevels.Error :
                    TheApp.Logger.Error.AddMessage(output);
                    break;

                case NWCore.LogLevels.Fatal:
                    TheApp.Logger.Fatal.AddMessage(output);
                    break;

                case NWCore.LogLevels.OpenLogFile:
                    TheApp.OpenLogFileWithTextEditor(output);
                    break;

                case NWCore.LogLevels.Console:
                    TheApp.OutputConsoleMsg(level, output);
                    break;
            }
        }

        /// <summary>
        /// Output log message to console.
        /// </summary>
        /// <param name="level">The log level.</param>
        /// <param name="msg">The message to log out.</param>
        private static void OutputConsoleMsg( NWCore.LogLevels level,
                                              string msg )
        {
            if ( string.IsNullOrEmpty(msg)==true )
                return;

            if ( level==NWCore.LogLevels.Fatal )
            {
                OutputExceptionLog( msg );
                Console.WriteLine( msg );
            }
            else
            {
                if ( TheApp.Logger==null )
                    return;

                ConsoleColor color = Console.ForegroundColor;

                switch ( level )
                {
                    case NWCore.LogLevels.Information:
                        Console.ForegroundColor = ConsoleColor.White;
                        Console.WriteLine( msg );
                        break;

                    case NWCore.LogLevels.Warning:
                        Console.ForegroundColor = ConsoleColor.Yellow;
                        Console.WriteLine( msg );
                        break;

                    case NWCore.LogLevels.Error:
                        Console.ForegroundColor = ConsoleColor.Red;
                        Console.WriteLine( msg );
                        break;

                    case NWCore.LogLevels.Debug:
                        #if DEBUG
                        Console.ForegroundColor = ConsoleColor.Blue;
                        DebugConsole.WriteLine( msg );
                        Console.WriteLine( msg );
                        #endif
                        break;

                    case NWCore.LogLevels.OpenLogFile:
                        // Do not open the file on console mode.
                        break;

                    case NWCore.LogLevels.Console:
                        Console.WriteLine(msg);
                        break;
                }

                Console.ForegroundColor = color;
            }
        }


        /// <summary>
        /// Open the specified log file with external text editor.
        /// The text editor can be assigned in the application options.
        /// </summary>
        /// <param name="filePath">The path to the log file.</param>
        /// <returns>True on success.</returns>
        private static bool OpenLogFileWithTextEditor( string filePath )
        {
            // External text editor path.
            string editorPath = Config.Data.Option.TextEditorPath;

            if ( string.IsNullOrEmpty(editorPath)==true )
            {
                // External text editor was not set, use Notepad instead.
                editorPath = System.Environment.SystemDirectory + "\\notepad.exe";
                if ( File.Exists(editorPath)==false )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Warning,
                                         res.Strings.WARNING_NOTEPAD_EXECUTABLE_NOT_FOUND );
                    return false;
                }
            }
            else if ( File.Exists(editorPath)==false )
            {
                // The assigned text editor executable does not exist.
                TheApp.OutputLogMsg( NWCore.LogLevels.Warning,
                                     res.Strings.WARNING_EXTERNAL_TEXT_EDIT_EXECUTABLE_NOT_FOUND + editorPath );

                // Use Notepad by default.
                editorPath = System.Environment.SystemDirectory + "\\notepad.exe";
                if ( File.Exists(editorPath)==false )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Warning,
                                         res.Strings.WARNING_NOTEPAD_EXECUTABLE_NOT_FOUND );
                    return false;
                }
            }

            try
            {
                // Launch external text editor to show the shader.
                Process textEditorProcess = Process.Start( editorPath, filePath );

                if ( textEditorProcess==null )
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                         res.Strings.WARNING_FAILED_LAUNCHING_TEXT_EDITOR );
                    return false;
                }
            }
            catch ( System.Exception ex )
            {
                TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                     res.Strings.WARNING_FAILED_LAUNCHING_TEXT_EDITOR );
                TheApp.OutputLogMsg( NWCore.LogLevels.Error, ex.Message );

                return false;
            }

            return true;
        }


        /// <summary>
        /// Output the exception to error log file.
        /// </summary>
        /// <param name="msg">The message to output.</param>
        public static void OutputExceptionLog( string msg )
        {
            try
            {
                string logFilePath =
                    System.IO.Path.GetDirectoryName( Application.ExecutablePath ) + "\\error.log";

                FileInfo info = new FileInfo( logFilePath );
                if ( info.Exists==true && info.Length>=4194304 )
                    File.Delete( logFilePath );

                StreamWriter writer = new StreamWriter( logFilePath, true );
                writer.WriteLine( "--------------------------------------------------------------------------------" );
                writer.WriteLine( "Error occurred at {0} {1}",
                                  DateTime.Now.ToShortDateString(),
                                  DateTime.Now.ToShortTimeString() );
                writer.WriteLine( "--------------------------------------------------------------------------------" );
                writer.WriteLine( msg );
                writer.Close();
            }
            // ログファイル出力失敗時はもうどうしようもない
            catch ( Exception )
            {
            }
        }


        /// <summary>
        /// Output the exception to error log file.
        /// </summary>
        /// <param name="ex">The exception.</param>
        public static void OutputExceptionLog( Exception ex )
        {
            try
            {
                string logFilePath =
                    System.IO.Path.GetDirectoryName( Application.ExecutablePath ) + "\\error.log";

                FileInfo info = new FileInfo( logFilePath );
                if ( info.Exists==true && info.Length>=4194304 )
                    File.Delete( logFilePath );

                StreamWriter writer = new StreamWriter( logFilePath, true );
                writer.WriteLine( "--------------------------------------------------------------------------------" );
                writer.WriteLine( "Error occurred at {0} {1}",
                                  DateTime.Now.ToShortDateString(),
                                  DateTime.Now.ToShortTimeString() );
                writer.WriteLine( "--------------------------------------------------------------------------------" );
                writer.WriteLine( GetErrorMessage( ex ) );
                writer.WriteLine( ex.StackTrace );
                writer.Close();
            }
            // ログファイル出力失敗時はもうどうしようもない
            catch ( Exception )
            {
            }
        }


        /// <summary>
        /// 致命的なエラー発生時の処理。
        /// </summary>
        private static void OnFatalErrorOccured(Exception error, bool fromThreadException)
        {
            // 一度処理したら処理させない
            if (_fatalErrorHandled) { return; }

            string errorMessage = StringResource.Get("APP_FATAL_ERROR_MSG");
            // ログファイルを出力
            try
            {
                string logFilePath =
                    System.IO.Path.GetDirectoryName( Application.ExecutablePath ) + "\\error.log";

                FileInfo info = new FileInfo( logFilePath );
                if ( info.Exists==true && info.Length>=4194304 )
                    File.Delete( logFilePath );

                StreamWriter writer = new StreamWriter( logFilePath, true );
                writer.WriteLine( "--------------------------------------------------------------------------------" );
                writer.WriteLine( "Error occurred at {0} {1}",
                                  DateTime.Now.ToShortDateString(),
                                  DateTime.Now.ToShortTimeString() );
                writer.WriteLine( "--------------------------------------------------------------------------------" );
                writer.WriteLine(GetErrorMessage(error));
                writer.WriteLine(error.StackTrace);
                writer.Close();

                errorMessage += "\n\n\"" + logFilePath + "\"\n" + StringResource.Get("APP_FATAL_ERROR_LOG_MSG");
            }
            // ログファイル出力失敗時はもうどうしようもない
            catch (Exception) { }
#if DEBUG
            // デバッグ用付加情報
            errorMessage += "\n\n[From]: " + (fromThreadException ? "ThreadException" : "WinMain");
            errorMessage += "\n\n[Message]: " + GetErrorMessage(error);
            errorMessage += "\n\n[StackTrace]: " + error.StackTrace;
#endif
            // エラーメッセージ表示
            ////ThreadSafeMsgBox.Show( errorMessage,
            ////                       res.Strings.ERROR_CAPTION,
            ////                       System.Windows.Forms.MessageBoxButtons.OK,
            ////                       System.Windows.Forms.MessageBoxIcon.Error );
            _fatalErrorHandled = true;
        }

        /// <summary>
        /// デバッグ用にpublic に変更
        /// </summary>
        public static string GetErrorMessage(Exception error)
        {
            string result = error.Message + "\n";
            if (error.InnerException != null)
            {
                result += "\n    " + GetErrorMessage(error.InnerException);
            }
            return result;
        }

        /// <summary>
        /// スタックトレース
        /// </summary>
        public static string GetStackTrace(Exception error)
        {
            string result = error.StackTrace + "\n";
            if (error.InnerException != null)
            {
                return GetStackTrace(error.InnerException) + "    " + result;
            }
            return result;
        }

        #endregion

        #region イベントハンドラ

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private static void Event_Application_ThreadException(object sender, ThreadExceptionEventArgs e)
        {
            OnFatalErrorOccured(e.Exception, true);

            // メッセージループを抜けないので強制終了させる
            Application.Exit();
        }

        #endregion
    }

    #region LanguageMode

    /// <summary>
    /// 言語モード。
    /// </summary>
    public enum LanguageMode
    {
        /// <summary>日本語</summary>
        Japanese = 0,
        /// <summary>英語</summary>
        English,
    }

    #endregion

    #region Class for AppExitPreventBlock

    /// <summary>
    /// Class for prevent application from quiting.
    /// </summary>
    public class AppExitPreventBlock : IDisposable
    {
        private static bool   s_bCanExit   = true;
        private static string s_warningMsg = string.Empty;

        private bool   m_bOriginalFlag = true;
        private string m_originalMsg   = string.Empty;

        /// <summary>
        /// Constructor.
        /// </summary>
        public AppExitPreventBlock( string warningMsg )
        {
            m_bOriginalFlag = s_bCanExit;
            m_originalMsg   = s_warningMsg;
            s_bCanExit      = false;
            s_warningMsg    = warningMsg;
        }

        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            s_bCanExit   = m_bOriginalFlag;
            s_warningMsg = m_originalMsg;
        }

        /// <summary>
        /// Static property to get or set the flag indicating
        /// whether can exit the application or not.
        /// </summary>
        public static bool CanExitApp
        {
            get { return s_bCanExit; }
        }

        /// <summary>
        /// The warning message while attempting to quit application.
        /// </summary>
        public static string WarningAttemptExitApp
        {
            get { return s_warningMsg; }
        }
    }

    #endregion
}
