﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Converters;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the binary data field definition.
    /// </summary>
    public class BinaryFieldDefinitionViewModel : ViewModelBase
    {
        /// <summary>The binary field group definition view model.</summary>
        private BinaryFieldGroupDefinitionViewModel fieldGroupViewModel = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public BinaryFieldDefinitionViewModel(
            ViewModelBase parent,
            BinaryFieldDefinition dataModel) :
            base(parent, dataModel)
        {
            // Create executables.
            this.OnCommitEditingExecutable = new AnonymousExecutable(this.OnCommitEditing);
            this.OnAddConnectionSource = new AnonymousExecutable(this.AddConnectionSource);
            this.OnRemoveConnectionSource = new AnonymousExecutable(this.RemoveConnectionSource);

            // Create the view model for the field group definition.
            if (dataModel.GroupDefinition != null)
            {
                this.fieldGroupViewModel =
                    new BinaryFieldGroupDefinitionViewModel(this, dataModel.GroupDefinition);
            }
        }

        /// <summary>
        /// Get the target definition.
        /// </summary>
        public DefinitionBase TargetDefinition
        {
            get
            {
                var myDataModel = this.DataModel as BinaryFieldDefinition;
                if (myDataModel == null)
                {
                    return null;
                }

                return WorkspaceManager.FindDefinition(myDataModel.OutputFieldGuid, true);
            }
        }

        /// <summary>
        /// Get the flag indicating whether if this field is a binary data.
        /// </summary>
        public bool IsBinaryData
        {
            get { return this.TargetDefinition is BinaryDataDefinition; }
        }

        /// <summary>
        /// Get or set the description of the target.
        /// </summary>
        public string TargetDescription
        {
            get
            {
                var def = this.TargetDefinition;
                if (def is RuntimeDataModelPropertyDefinition)
                {
                    return ((RuntimeDataModelPropertyDefinition)def).Description;
                }
                else if (def is RuntimeDataModelDefinition)
                {
                    return ((RuntimeDataModelDefinition)def).Description;
                }
                else if (def is BinaryDataDefinition)
                {
                    return ((BinaryDataDefinition)def).Name;
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        /// <summary>
        /// Get or set the name of the target.
        /// </summary>
        public string TargetName
        {
            get
            {
                var def = this.TargetDefinition;
                if (def is RuntimeDataModelPropertyDefinition)
                {
                    return ((RuntimeDataModelPropertyDefinition)def).Name;
                }
                else if (def is RuntimeDataModelDefinition)
                {
                    return ((RuntimeDataModelDefinition)def).Name;
                }
                else if (def is BinaryDataDefinition)
                {
                    return ((BinaryDataDefinition)def).Name;
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        /// <summary>
        /// Get or set the type of the target.
        /// </summary>
        public string TargetType
        {
            get
            {
                var def = this.TargetDefinition;
                if (def is BinaryDataDefinition)
                {
                    return ((BinaryDataDefinition)def).Name;
                }
                else if (def is RuntimeDataModelDefinition)
                {
                    return ((RuntimeDataModelDefinition)def).Name;
                }
                else if ((def is RuntimeDataModelPropertyDefinition) == false)
                {
                    return string.Empty;
                }

                var propertyDef = (RuntimeDataModelPropertyDefinition)def;
                if (string.IsNullOrEmpty(propertyDef.TypeName) == true)
                {
                    return string.Empty;
                }

                string composedName = string.Empty;

                bool typeExists = TypeManager.ComposeRuntimeTypeFullName(
                    propertyDef.TypeNamespace,
                    propertyDef.TypeName,
                    out composedName);

                int arraySize;
                if (int.TryParse(propertyDef.ArraySize, out arraySize) == true &&
                    arraySize > 1)
                {
                    composedName += "[" + propertyDef.ArraySize + "]";
                }
                else if (string.IsNullOrEmpty(propertyDef.ArraySize) == false)
                {
                    composedName += "[" + propertyDef.ArraySize + "]";
                }

                return composedName;
            }
        }

        /// <summary>
        /// Get or set the converter used for converting the data from the data model property.
        /// </summary>
        public ConverterInfo ConverterInfo
        {
            get
            {
                object result;

                var binder = new EffectMakerGetMemberBinder("Converter");
                this.TryGetMember(binder, out result);

                if (result is string)
                {
                    return ConverterManager.FindConverter((string)result);
                }
                else
                {
                    return ConverterManager.DefaultConverter;
                }
            }

            set
            {
                ConverterInfo converter = null;
                if (value == null)
                {
                    converter = ConverterManager.DefaultConverter;
                }
                else
                {
                    converter = value;
                }

                var binder = new EffectMakerSetMemberBinder("Converter");
                this.TrySetMember(binder, converter.FullName);
            }
        }

        /// <summary>
        /// Get or set the parameters for the converter.
        /// </summary>
        public Dictionary<string, string> ConverterParamKeyValuePairs
        {
            get
            {
                var map = new Dictionary<string, string>();

                object result;
                var binder = new EffectMakerGetMemberBinder("ConverterParameters");
                if (this.TryGetMember(binder, out result) == false)
                {
                    return map;
                }

                var list = result as List<ConverterParameterDefinition>;
                if (list == null || list.Count <= 0)
                {
                    return map;
                }

                list.ForEach(item => map.Add(item.Name, item.Value));

                return map;
            }

            set
            {
                var myDataModel = this.DataModel as BinaryFieldDefinition;
                if (myDataModel == null)
                {
                    return;
                }

                myDataModel.ConverterParameters.Clear();

                if (value == null)
                {
                    return;
                }

                foreach (var pair in value)
                {
                    // Only add parameters that the value is not empty.
                    if (string.IsNullOrEmpty(pair.Value) == false)
                    {
                        myDataModel.ConverterParameters.Add(
                            new ConverterParameterDefinition(pair.Key, pair.Value));
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the binary field group view model.
        /// </summary>
        public BinaryFieldGroupDefinitionViewModel BinaryFieldGroupViewModel
        {
            get { return this.fieldGroupViewModel; }
            set { this.SetValue(ref this.fieldGroupViewModel, value); }
        }

        /// <summary>
        /// Enumerate the Guid of the connected sources.
        /// </summary>
        public IEnumerable<Guid> ConnectedSources
        {
            get
            {
                var myDataModel = this.DataModel as BinaryFieldDefinition;
                if (myDataModel == null)
                {
                    return null;
                }

                this.UpdateConnections();

                return myDataModel.InputPropertyGuidList;
            }
        }

        /// <summary>
        /// Get or set the executable for adding a connection source.
        /// </summary>
        public IExecutable OnAddConnectionSource { get; private set; }

        /// <summary>
        /// Get or set the executable for removing a connection source.
        /// </summary>
        public IExecutable OnRemoveConnectionSource { get; private set; }

        /// <summary>
        /// Get or set the executable for committing modification of the binary field.
        /// </summary>
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Update connections.
        /// Remove all the connections that the source no longer exists.
        /// </summary>
        public void UpdateConnections()
        {
            var myDataModel = this.DataModel as BinaryFieldDefinition;
            if (myDataModel == null)
            {
                return;
            }

            bool isPropertyChanged = false;

            // Find the connected source, remove the connection if the source is not found.
            for (int i = myDataModel.InputPropertyGuidList.Count - 1; i >= 0; --i)
            {
                // If the connected source definition doesn't exist anymore, remove the connection.
                var def = WorkspaceManager.FindDefinition(
                    myDataModel.InputPropertyGuidList[i],
                    true);
                if (def == null)
                {
                    myDataModel.InputPropertyGuidList.RemoveAt(i);
                    isPropertyChanged = true;
                }
            }

            if (isPropertyChanged == true)
            {
                this.OnPropertyChanged("ConnectedSources");
            }
        }

        /// <summary>
        /// Clear all the connected sources of this binary field.
        /// </summary>
        public void ClearConnectedSources()
        {
            var myDataModel = this.DataModel as BinaryFieldDefinition;
            if (myDataModel == null)
            {
                return;
            }

            myDataModel.InputPropertyGuidList.Clear();
            this.OnPropertyChanged("ConnectedSources");
        }

        /// <summary>
        /// Handle execution when committing modification of the binary data.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCommitEditing(object parameter)
        {
            var values = parameter as KeyValuePair<string, object>[];
            if (values == null)
            {
                return;
            }

            try
            {
                foreach (var pair in values)
                {
                    var binder = new EffectMakerSetMemberBinder(pair.Key, true);
                    this.TrySetMember(binder, pair.Value);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "BinaryFieldDefinitionViewModel.OnCommitEditing => Failed setting values to the binary field.");
                Logger.Log(LogLevels.Error, "Exception : {0}, message : {1}", ex.GetType().ToString(), ex.Message);
            }
        }

        /// <summary>
        /// Add a new connection source for the binary field.
        /// </summary>
        /// <param name="parameter">The parameter.</param>
        private void AddConnectionSource(object parameter)
        {
            if ((parameter is Guid) == false)
            {
                return;
            }

            var guid = (Guid)parameter;

            var myDataModel = this.DataModel as BinaryFieldDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // Is this source Guid already in the list?
            if (myDataModel.InputPropertyGuidList.Any(id => id.Equals(guid)) == true)
            {
                return;
            }

            myDataModel.InputPropertyGuidList.Add(guid);

            this.OnPropertyChanged("ConnectedSources");
        }

        /// <summary>
        /// Remove a existing connection source from the binary field.
        /// </summary>
        /// <param name="parameter">The parameter.</param>
        private void RemoveConnectionSource(object parameter)
        {
            if ((parameter is IEnumerable<Guid>) == false)
            {
                return;
            }

            var myDataModel = this.DataModel as BinaryFieldDefinition;
            if (myDataModel == null)
            {
                return;
            }

            foreach (Guid guid in (IEnumerable<Guid>)parameter)
            {
                // Remove this source Guid.
                myDataModel.InputPropertyGuidList.RemoveAll(id => id.Equals(guid));
            }

            this.OnPropertyChanged("ConnectedSources");
        }
    }
}
