﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the binary data definition.
    /// </summary>
    public class BinaryDataDefinitionViewModel : ViewModelBase
    {
        /// <summary>The binary field view models.</summary>
        private List<BinaryFieldDefinitionViewModel> fieldViewModels =
            new List<BinaryFieldDefinitionViewModel>();

        /// <summary>The source data model instance view model.</summary>
        private SourceDataModelInstanceViewModel srcDataModelInstanceViewModel = null;

        /// <summary>
        /// The field view model that is currently selected.
        /// </summary>
        private BinaryFieldDefinitionViewModel selectedFieldViewModel = null;

        /// <summary>The data model instance with the correct type.</summary>
        private BinaryDataDefinition typedDataModel;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public BinaryDataDefinitionViewModel(
            ViewModelBase parent,
            BinaryDataDefinition dataModel) :
            base(parent, dataModel)
        {
            this.typedDataModel = dataModel;

            this.OnCommitEditingExecutable = new AnonymousExecutable(this.OnCommitEditing);
            this.OnCancelEditingExecutable = new AnonymousExecutable(this.OnCancelEditing);

            this.OnCreateBinaryFieldExecutable =
                new AnonymousExecutable(this.OnCreateField);

            this.OnRemoveSelectedBinaryFieldExecutable =
                new AnonymousExecutable(this.OnRemoveSelectedField);

            this.OnMoveBinaryFieldExecutable =
                new AnonymousExecutable(this.OnMoveSelectedField);

            foreach (BinaryFieldDefinition def in dataModel.Fields)
            {
                var child = new BinaryFieldDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.fieldViewModels.Add(child);
            }

            if (dataModel.SourceDataModelInstance != null)
            {
                this.srcDataModelInstanceViewModel =
                    new SourceDataModelInstanceViewModel(this, dataModel.SourceDataModelInstance);
            }

            // If any of the data model is modified, update the source data model instance.
            RootEditorDataModelDefinitionViewModel.DataModelListChanged += (s, e) => this.OnUpdateSourceDataModelInstance();

            // If any of the property is modified, update the source data model instance.
            EditorDataModelDefinitionViewModel.DataModelPropertyListChanged += (s, e) => this.OnUpdateSourceDataModelInstance();

            // If any of the property is modified, update the source data model instance.
            EditorDataModelPropertyDefinitionViewModel.DataModelPropertyModified += (s, e) => this.OnUpdateSourceDataModelInstance();

            // The binary data list is changed, update the fields.
            RootBinaryDataDefinitionViewModel.BinaryDataListChanged += (s, e) => this.UpdateBinaryFields();

            // The runtime data model list is changed, update the fields.
            RootRuntimeDataModelDefinitionViewModel.DataModelListChanged += (s, e) => this.UpdateBinaryFields();

            // The runtime data model is modified, update the fields.
            RuntimeDataModelDefinitionViewModel.DataModelModified += (s, e) => this.UpdateBinaryFields();

            // The runtime data model is modified, update the fields.
            RuntimeDataModelPropertyDefinitionViewModel.DataModelPropertyModified += (s, e) => this.UpdateBinaryFields();

            this.OnUpdateSourceDataModelInstance();
            this.UpdateBinaryFields();
        }

        /// <summary>
        /// Get the available source editor data models.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableSourceDataModels
        {
            get
            {
                yield return new KeyValuePair<string, object>(
                        string.Empty,
                        Guid.Empty);

                foreach (EditorTypeInfo info in TypeManager.EditorTypes.OrderBy(it => it.FullName))
                {
                    if (info.IsDataModel == false)
                    {
                        continue;
                    }

                    yield return new KeyValuePair<string, object>(
                        info.FullName,
                        info.DataModelDefinition.Guid);
                }
            }
        }

        /// <summary>
        /// Get or set the Guid of the source data model.
        /// </summary>
        public object SourceDataModelGuid
        {
            get
            {
                if (this.srcDataModelInstanceViewModel == null)
                {
                    return Guid.Empty;
                }

                return this.srcDataModelInstanceViewModel.DataModelDefinition.Guid;
            }

            set
            {
                // The value has to be a Guid.
                if ((value is Guid) == false)
                {
                    return;
                }

                this.UpdateSourceDataModelInstance((Guid)value);

                this.OnPropertyChanged();
                this.OnPropertyChanged("SourceDataModelInstanceViewModel");
            }
        }

        /// <summary>
        /// Get the view model for the source data model instance.
        /// </summary>
        public SourceDataModelInstanceViewModel SourceDataModelInstanceViewModel
        {
            get { return this.srcDataModelInstanceViewModel; }
        }

        /// <summary>
        /// Enumerate the binary field view models.
        /// </summary>
        public IEnumerable<BinaryFieldDefinitionViewModel> BinaryFieldViewModels
        {
            get { return this.fieldViewModels; }
        }

        /// <summary>
        /// Get or set the selected field group view model.
        /// </summary>
        public object SelectedFieldViewModel
        {
            get
            {
                return this.selectedFieldViewModel;
            }

            set
            {
                this.SetValue(
                    ref this.selectedFieldViewModel,
                    value as BinaryFieldDefinitionViewModel);
            }
        }

        /// <summary>
        /// Get or set the executable for committing modification of the binary data.
        /// </summary>
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling modification of the binary data.
        /// </summary>
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for creating a new field.
        /// </summary>
        public IExecutable OnCreateBinaryFieldExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for removing the selected field.
        /// </summary>
        public IExecutable OnRemoveSelectedBinaryFieldExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for moving the selected field.
        /// </summary>
        public IExecutable OnMoveBinaryFieldExecutable { get; set; }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            this.typedDataModel = null;

            if (this.srcDataModelInstanceViewModel != null)
            {
                this.srcDataModelInstanceViewModel.Dispose();
                this.srcDataModelInstanceViewModel = null;
            }

            this.selectedFieldViewModel = null;
            foreach (var vm in this.fieldViewModels)
            {
                vm.Dispose();
            }

            base.Dispose();
        }

        /// <summary>
        /// Handle execution when committing modification of the binary data.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCommitEditing(object parameter)
        {
            var values = parameter as KeyValuePair<string, object>[];
            if (values == null)
            {
                return;
            }

            try
            {
                foreach (var pair in values)
                {
                    var binder = new EffectMakerSetMemberBinder(pair.Key, true);
                    this.TrySetMember(binder, pair.Value);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "BinaryDataDefinitionViewModel.OnCommitEditing => Failed setting values to the binary data.");
                Logger.Log(LogLevels.Error, "Exception : {0}, message : {1}", ex.GetType().ToString(), ex.Message);
            }

            var rootBinaryDataVM = this.Parent as RootBinaryDataDefinitionViewModel;
            if (rootBinaryDataVM != null)
            {
                rootBinaryDataVM.NotifyBinaryDataNameChanged(this);
            }
        }

        /// <summary>
        /// Handle execution when canceling modification of the binary data.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCancelEditing(object parameter)
        {
            var myParent = this.Parent as RootBinaryDataDefinitionViewModel;
            if (myParent == null)
            {
                return;
            }

            myParent.DeleteBinaryData(this);
        }

        /// <summary>
        /// Handle execution for creating binary field.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCreateField(object parameter)
        {
            // Get the target definition.
            var targetDef = parameter as DefinitionBase;
            if (targetDef == null)
            {
                return;
            }

            var myDataModel = this.DataModel as BinaryDataDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // Search for all the field view models, check if any one of their data model
            // the same target Guid as the one we are about to create.
            if (myDataModel.Fields.Any(fd => fd.OutputFieldGuid.Equals(targetDef.Guid)) == true)
            {
                return;
            }

            // Add the binary field definitions if the target is a data model.
            BinaryFieldDefinition def = null;
            if (targetDef is RuntimeDataModelDefinition)
            {
                var dataModelDef = (RuntimeDataModelDefinition)targetDef;
                def = this.CreateBinaryFieldDefinitions(
                    myDataModel.Fields,
                    null,
                    dataModelDef);
            }
            else if (targetDef is BinaryDataDefinition)
            {
                def = new BinaryFieldDefinition()
                {
                    OutputFieldGuid = targetDef.Guid
                };

                myDataModel.Fields.Add(def);
            }

            // Create the view model for the group definition and add it to the children.
            var child = new BinaryFieldDefinitionViewModel(this, def);
            this.Children.Add(child);
            this.fieldViewModels.Add(child);

            // Notify that the field group view model list is modified.
            this.OnPropertyChanged("BinaryFieldViewModels");

            // Make the new view model selected.
            this.SelectedFieldViewModel = child;
        }

        /// <summary>
        /// Recursively add all the field definitions of the data model to the binary field list.
        /// </summary>
        /// <param name="fieldList">The binary field list to add the created field to.</param>
        /// <param name="propertyDef">The runtime data model property definition.</param>
        /// <param name="dataModelDef">The runtime data model definition.</param>
        /// <returns>The created field definition.</returns>
        private BinaryFieldDefinition CreateBinaryFieldDefinitions(
            IList<BinaryFieldDefinition> fieldList,
            RuntimeDataModelPropertyDefinition propertyDef,
            RuntimeDataModelDefinition dataModelDef)
        {
            // Create a field for the data model, with the super classes and
            // properties stored in the field group.
            var fieldDef = new BinaryFieldDefinition()
            {
                OutputFieldGuid = propertyDef == null ? dataModelDef.Guid : propertyDef.Guid,
                GroupDefinition = new BinaryFieldGroupDefinition()
                {
                    TargetGuid = dataModelDef.Guid
                }
            };

            fieldList.Add(fieldDef);

            return this.CreateBinaryFieldContents(fieldDef, propertyDef, dataModelDef);
        }

        /// <summary>
        /// Recursively create all the field definition contents.
        /// </summary>
        /// <param name="fieldDef">The binary field to create the contents to.</param>
        /// <param name="propertyDef">The runtime data model property definition.</param>
        /// <param name="dataModelDef">The runtime data model definition.</param>
        /// <returns>The created field definition.</returns>
        private BinaryFieldDefinition CreateBinaryFieldContents(
            BinaryFieldDefinition fieldDef,
            RuntimeDataModelPropertyDefinition propertyDef,
            RuntimeDataModelDefinition dataModelDef)
        {
            // Loop through super classes.
            foreach (Guid guid in dataModelDef.SuperClassGuidList)
            {
                var superClassDef =
                    WorkspaceManager.FindDefinition(guid) as RuntimeDataModelDefinition;
                if (superClassDef == null)
                {
                    continue;
                }

                this.CreateBinaryFieldDefinitions(
                    fieldDef.GroupDefinition.Fields,
                    null,
                    superClassDef);
            }

            // Loop through the properties.
            foreach (var property in dataModelDef.Properties)
            {
                RuntimeTypeInfo info =
                    TypeManager.FindRuntimeType(property.TypeNamespace, property.TypeName);
                if (info != null && info.IsStructure == true)
                {
                    this.CreateBinaryFieldDefinitions(
                        fieldDef.GroupDefinition.Fields,
                        property,
                        info.StructureDefinition);
                }
                else
                {
                    fieldDef.GroupDefinition.Fields.Add(new BinaryFieldDefinition()
                    {
                        OutputFieldGuid = property.Guid
                    });
                }
            }

            return fieldDef;
        }

        /// <summary>
        /// Handle execution when requested to move a binary field.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnMoveSelectedField(object parameter)
        {
            if (this.selectedFieldViewModel == null)
            {
                return;
            }

            var myDataModel = this.DataModel as BinaryDataDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // Get the field view model from parameter.
            // The selected group should move to the front of it.
            var nextViewModel = parameter as BinaryFieldDefinitionViewModel;

            if (this.selectedFieldViewModel == nextViewModel)
            {
                // The selected view model is at exactly the same location as requested,
                // so nothing to do.
                return;
            }

            int originalIndex =
                this.fieldViewModels.IndexOf(this.selectedFieldViewModel);

            // First remove the selected view model.
            this.Children.Remove(this.selectedFieldViewModel);
            this.fieldViewModels.Remove(this.selectedFieldViewModel);

            var def = this.selectedFieldViewModel.DataModel as BinaryFieldDefinition;
            if (def != null)
            {
                myDataModel.Fields.Remove(def);
            }

            // Compute the index to insert it back.
            int childInsertIndex;
            int fieldInsertIndex;
            int definitionIndex = myDataModel.Fields.Count;
            if (nextViewModel == null)
            {
                childInsertIndex = this.Children.Count;
                fieldInsertIndex = this.fieldViewModels.Count;
            }
            else
            {
                childInsertIndex = this.Children.IndexOf(nextViewModel);
                fieldInsertIndex = this.fieldViewModels.IndexOf(nextViewModel);

                var nextDef = nextViewModel.DataModel as BinaryFieldDefinition;
                if (nextDef != null)
                {
                    definitionIndex = myDataModel.Fields.IndexOf(nextDef);
                }
            }

            // Insert the selected view model back.
            this.Children.Insert(childInsertIndex, this.selectedFieldViewModel);
            this.fieldViewModels.Insert(fieldInsertIndex, this.selectedFieldViewModel);

            myDataModel.Fields.Insert(definitionIndex, def);

            if (fieldInsertIndex != originalIndex)
            {
                // Notify that the field group view model list is modified.
                this.OnPropertyChanged("BinaryFieldViewModels");
            }
        }

        /// <summary>
        /// Handle execution when removing the selected binary field.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnRemoveSelectedField(object parameter)
        {
            if (this.selectedFieldViewModel != null)
            {
                this.RemoveBinaryFieldInternal(this.selectedFieldViewModel);

                // Notify that the field view model list is modified.
                this.OnPropertyChanged("BinaryFieldViewModels");
            }
        }

        /// <summary>
        /// Remove the specified binary field.
        /// </summary>
        /// <param name="vm">The binary field view model.</param>
        private void RemoveBinaryFieldInternal(BinaryFieldDefinitionViewModel vm)
        {
            this.Children.Remove(vm);
            this.fieldViewModels.Remove(vm);

            var viewModel = vm;
            if (this.selectedFieldViewModel == vm)
            {
                this.selectedFieldViewModel = null;
            }

            var myDataModel = this.DataModel as BinaryDataDefinition;
            var def = viewModel.DataModel as BinaryFieldDefinition;
            if (myDataModel != null && def != null)
            {
                myDataModel.Fields.Remove(def);
            }

            viewModel.Dispose();
            def.Dispose();
        }

        /// <summary>
        /// Update all the connections of the fields, recursively.
        /// </summary>
        /// <param name="fields">The binary fields to update.</param>
        private void UpdateBinaryFieldConnections(
            IEnumerable<BinaryFieldDefinitionViewModel> fields)
        {
            if (fields == null)
            {
                return;
            }

            foreach (BinaryFieldDefinitionViewModel field in fields)
            {
                field.UpdateConnections();
                if (field.BinaryFieldGroupViewModel != null)
                {
                    this.UpdateBinaryFieldConnections(
                        field.BinaryFieldGroupViewModel.BinaryFieldViewModels);
                }
            }
        }

        /// <summary>
        /// Clear all the connections of the fields, recursively.
        /// </summary>
        /// <param name="fields">The binary fields to update.</param>
        private void ClearBinaryFieldConnections(
            IEnumerable<BinaryFieldDefinitionViewModel> fields)
        {
            if (fields == null)
            {
                return;
            }

            foreach (BinaryFieldDefinitionViewModel field in fields)
            {
                field.ClearConnectedSources();
                if (field.BinaryFieldGroupViewModel != null)
                {
                    this.ClearBinaryFieldConnections(
                        field.BinaryFieldGroupViewModel.BinaryFieldViewModels);
                }
            }
        }

        /// <summary>
        /// Update fields when their target runtime data model property or binary data
        /// is changed.
        /// </summary>
        private void UpdateBinaryFields()
        {
            for (int i = this.fieldViewModels.Count - 1; i >= 0; --i)
            {
                var field = this.fieldViewModels[i];
                if (field.TargetDefinition == null)
                {
                    this.RemoveBinaryFieldInternal(field);
                }
                else if (field.BinaryFieldGroupViewModel != null)
                {
                    field.BinaryFieldGroupViewModel.UpdateFromTargetDefinition();
                }
            }
        }

        /// <summary>
        /// Handle events that requires this binary data to update it's source data model instances.
        /// </summary>
        private void OnUpdateSourceDataModelInstance()
        {
            if (this.srcDataModelInstanceViewModel != null &&
                this.srcDataModelInstanceViewModel.DataModelDefinition != null)
            {
                this.UpdateSourceDataModelInstance(
                    this.srcDataModelInstanceViewModel.DataModelDefinition.Guid);
            }
        }

        /// <summary>
        /// Update binary conversion source data model instance
        /// data model and corresponding view model.
        /// </summary>
        /// <param name="dataModelDefGuid">The Guid of the data model definition.</param>
        private void UpdateSourceDataModelInstance(Guid dataModelDefGuid)
        {
            // Is the assigned data model definition different from the one we are using?
            if (dataModelDefGuid.Equals(this.SourceDataModelGuid) == false)
            {
                // Remove the original data model instance.
                if (this.typedDataModel.SourceDataModelInstance != null)
                {
                    this.typedDataModel.SourceDataModelInstance.Dispose();
                    this.typedDataModel.SourceDataModelInstance = null;
                }

                // Create a new empty data model instance.
                this.typedDataModel.SourceDataModelInstance = new SourceDataModelInstanceDefinition()
                {
                    DataModelDefinitionGuid = dataModelDefGuid
                };
            }

            // First remove the source data model instance view model.
            if (this.srcDataModelInstanceViewModel != null)
            {
                this.srcDataModelInstanceViewModel.Dispose();
                this.srcDataModelInstanceViewModel = null;
            }

            // Look for the data model definition.
            var dataModelDef =
                WorkspaceManager.FindDefinition(dataModelDefGuid) as EditorDataModelDefinition;
            if (dataModelDef == null)
            {
                this.ClearBinaryFieldConnections(this.fieldViewModels);
                return;
            }

            Stack<EditorDataModelDefinition> parents = new Stack<EditorDataModelDefinition>();

            // Update the source data model instance.
            this.UpdateSourceDataModelInstanceRecursive(
                dataModelDef,
                this.typedDataModel.SourceDataModelInstance,
                parents);

            // Create the view model for the data model instance.
            this.srcDataModelInstanceViewModel =
                new SourceDataModelInstanceViewModel(this, this.typedDataModel.SourceDataModelInstance);

            // Update the binary field connections.
            this.UpdateBinaryFieldConnections(this.fieldViewModels);
        }

        /// <summary>
        /// Recursively update source data model instance with the given data model definition.
        /// </summary>
        /// <param name="dataModelDef">The data model definition.</param>
        /// <param name="dataModelInst">The data model instance definition to update.</param>
        /// <param name="parents">A stack that holds all the parent definitions.</param>
        private void UpdateSourceDataModelInstanceRecursive(
            EditorDataModelDefinition dataModelDef,
            SourceDataModelInstanceDefinition dataModelInst,
            Stack<EditorDataModelDefinition> parents)
        {
            parents.Push(dataModelDef);

            // Remove the property instances that their definition no longer exist.
            for (int i = dataModelInst.Properties.Count - 1; i >= 0; --i)
            {
                SourcePropertyInstanceDefinition propertyInst = dataModelInst.Properties[i];
                if (dataModelDef.Properties.Any(
                    p => p.Guid == propertyInst.PropertyDefinitionGuid) == false)
                {
                    // Remove the property instance because the
                    // property definition doesn't exist anymore.
                    dataModelInst.Properties.RemoveAt(i);
                }
            }

            // Temporarily save the property instances to a map
            // with the property definition Guid as the key.
            var tmpPropertyInstanceMap = dataModelInst.Properties.ToDictionary(
                p => p.PropertyDefinitionGuid);

            // Clear the property instance list.
            // We will add them back in the order of the property definitions,
            // so that they will all be in the correct order.
            dataModelInst.Properties.Clear();

            // Loop through the properties in the data model definition.
            foreach (EditorDataModelPropertyDefinition propertyDef in dataModelDef.Properties)
            {
                // Find the instance for the property definition.
                SourcePropertyInstanceDefinition propertyInst = null;
                if (tmpPropertyInstanceMap.TryGetValue(propertyDef.Guid, out propertyInst) == true)
                {
                    // Remove this property instance from the temporary map.
                    tmpPropertyInstanceMap.Remove(propertyDef.Guid);
                }
                else
                {
                    // The property instance is not found, create the property instance.
                    propertyInst = new SourcePropertyInstanceDefinition()
                    {
                        PropertyDefinitionGuid = propertyDef.Guid,
                    };
                }

                EditorDataModelDefinition origUnderlyingDataModelDef = null;
                EditorDataModelDefinition currUnderlyingDataModelDef = null;

                if (propertyInst.DataModelInstance != null)
                {
                    // Get the original underlying data model definition of the property.
                    origUnderlyingDataModelDef = WorkspaceManager.FindDefinition(
                        propertyInst.DataModelInstance.DataModelDefinitionGuid) as EditorDataModelDefinition;
                }

                // If the property type is a data model, create the
                // underlying data model instance recursively.
                EditorTypeInfo info =
                    TypeManager.FindEditorType(propertyDef.Namespace, propertyDef.Type);
                if (info != null &&
                    info.IsDataModel == true &&
                    info.DataModelDefinition != null)
                {
                    currUnderlyingDataModelDef = info.DataModelDefinition;
                }
                else if (info != null && info.IsGeneric == true)
                {
                    // The type is generic, check if the element type is data model.
                    EditorTypeInfo elementInfo = TypeManager.FindEditorType(
                        propertyDef.ElementNamespace,
                        propertyDef.ElementType);

                    if (elementInfo != null &&
                        elementInfo.IsDataModel == true &&
                        elementInfo.DataModelDefinition != null)
                    {
                        currUnderlyingDataModelDef = elementInfo.DataModelDefinition;
                    }
                }

                // Do not create the underlying data model instance if any of the parent
                // data model instance use this definition to prevent looping.
                // E.g. EmitterData <-------┐
                //          │               │ Looping!
                //          ├ EmitterList --┘
                //          ├     ┇
                if (parents.Any(def => def == currUnderlyingDataModelDef) == true)
                {
                    currUnderlyingDataModelDef = null;
                }

                // Has underlying data model definition been changed?
                if (origUnderlyingDataModelDef != currUnderlyingDataModelDef)
                {
                    // First clear the original underlying data model instance.
                    if (origUnderlyingDataModelDef != null)
                    {
                        propertyInst.DataModelInstance.Dispose();
                        propertyInst.DataModelInstance = null;
                    }

                    // Create the new underlying data model instance.
                    if (currUnderlyingDataModelDef != null)
                    {
                        propertyInst.DataModelInstance = new SourceDataModelInstanceDefinition()
                        {
                            DataModelDefinitionGuid = currUnderlyingDataModelDef.Guid
                        };
                    }
                }

                // Update the underlying data model instance.
                if (propertyInst.DataModelInstance != null)
                {
                    this.UpdateSourceDataModelInstanceRecursive(
                        currUnderlyingDataModelDef,
                        propertyInst.DataModelInstance,
                        parents);
                }

                // Add the property instance to the parent data model instance.
                dataModelInst.Properties.Add(propertyInst);
            }

            parents.Pop();
        }
    }
}
