﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace EffectMaker.DataModelMaker.UILogic.Utilities
{
    /// <summary>
    /// Utility class for "names". (e.g. creating default names.)
    /// </summary>
    public class NamingUtility
    {
        /// <summary>
        /// Create a unique name with the specified base name and the existing names
        /// to prevent name collision.
        /// </summary>
        /// <param name="baseName">The base name to use.</param>
        /// <param name="existingNames">The existing names.</param>
        /// <returns>The generated unique name.</returns>
        public static string CreateUniqueName(
            string baseName,
            IEnumerable<string> existingNames)
        {
            var regex = new Regex("^" + baseName + "([0-9]+)$");

            List<int> existingSerialList = new List<int>();

            // Collect all the numbers that is used.
            foreach (string name in existingNames)
            {
                Match match = regex.Match(name);
                if (match.Groups.Count < 2)
                {
                    continue;
                }

                int iCurrSerial;
                if (int.TryParse(match.Groups[1].Value, out iCurrSerial) == true)
                {
                    existingSerialList.Add(iCurrSerial);
                }
            }

            return baseName + FindUnusedSerialNumber(existingSerialList).ToString();
        }

        /// <summary>
        /// Find an unused serial number from the given used serial number list.
        /// </summary>
        /// <param name="usedSerialNumList">The serial numbers that has already being used.</param>
        /// <returns>An unused serial number.</returns>
        public static int FindUnusedSerialNumber(List<int> usedSerialNumList)
        {
            int iSerial = 1;
            bool bFound = true;
            while (bFound == true)
            {
                bFound = false;
                foreach (int iNum in usedSerialNumList)
                {
                    if (iNum == iSerial)
                    {
                        ++iSerial;
                        bFound = true;
                        break;
                    }
                }
            }

            return iSerial;
        }
    }
}
