﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.DataModelMaker.Core.DataTypes;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView.Editor
{
    /// <summary>
    /// Class for rendering editor-side data model property information.
    /// </summary>
    internal class EditorDataModelInfoPropertyRenderable : DataModelInfoPropertyRenderable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public EditorDataModelInfoPropertyRenderable(DataModelInfoRenderable owner) :
            base(owner)
        {
            this.Size = new SizeF(0.0f, 32.0f);

            // Set up bindings.
            this.Bindings.Add(new Binder(this, "PropertyType", "PropertyTypeWithoutNamespace"));
            this.Bindings.Add(new Binder(this, "IsPropertyTypeInvalid", "IsPropertyTypeInvalid"));
            this.Bindings.Add(new Binder(this, "PropertyName", "Name"));
            this.Bindings.Add(new Binder(this, "PropertyDesc", "Description"));
        }

        /// <summary>
        /// Get or set the property's value type.
        /// </summary>
        public string PropertyType { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether property type is invalid.
        /// </summary>
        public bool IsPropertyTypeInvalid { get; set; }

        /// <summary>
        /// Get or set the property name.
        /// </summary>
        public string PropertyName { get; set; }

        /// <summary>
        /// Get or set the property description.
        /// </summary>
        public string PropertyDesc { get; set; }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            SizeF valueTypeTextSize = g.MeasureString(
                this.PropertyType,
                DataModelInfoPropertyRenderable.ItemSmallFont);

            // The value type text rectangle.
            RectangleF valueTypeTextRect = this.TransformedBounds;
            valueTypeTextRect.Width = Math.Min(valueTypeTextSize.Width, this.TransformedBounds.Width);
            valueTypeTextRect.Height = valueTypeTextSize.Height;

            // The property name occupies the right size of the item.
            RectangleF propertyNameRect = this.TransformedBounds;
            propertyNameRect.Y += valueTypeTextSize.Height;
            propertyNameRect.Height -= valueTypeTextSize.Height;

            using (var strFormat = new StringFormat())
            {
                // Bottom-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                g.DrawString(
                    this.PropertyType,
                    DataModelInfoPropertyRenderable.ItemSmallFont,
                    this.IsPropertyTypeInvalid == true ? Brushes.Red : Brushes.DimGray,
                    valueTypeTextRect,
                    strFormat);

                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Near;

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.PropertyName,
                    DataModelInfoPropertyRenderable.ItemLargeFont,
                    Brushes.Blue,
                    propertyNameRect,
                    strFormat);
            }
        }
    }
}
