﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView
{
    /// <summary>
    /// The viewport class for rendering the data model information.
    /// </summary>
    internal class DataModelInfoViewport : Viewport, ILogicalTreeElement
    {
        /// <summary>Constant margin of the child renderable.</summary>
        private const float ChildMargin = 5.0f;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>
        /// Backing field for Controls property.
        /// (should be empty, just to satisfy ILogicalTreeElement)
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The renderable that renders the data model information.</summary>
        private DataModelInfoRenderable dataModelInfoRenderable = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="renderingControl">The control to be rendered to.</param>
        public DataModelInfoViewport(Control renderingControl) :
            base(renderingControl)
        {
            this.Bindings = new BindingContainer(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.FillColor = Color.White;

            this.Bindings.Add(new Binder(this, "DataModelRenderableDataContext", "DisplayedDataModel"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return null; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Get the data model information renderable.
        /// </summary>
        public DataModelInfoRenderable DataModelInfoRenderable
        {
            get { return this.dataModelInfoRenderable; }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the data context for the data model info renderable.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataModelRenderableDataContext
        {
            get
            {
                if (this.dataModelInfoRenderable == null)
                {
                    return null;
                }

                return this.dataModelInfoRenderable.DataContext;
            }

            set
            {
                if (value is EditorDataModelDefinitionViewModel)
                {
                    if (this.dataModelInfoRenderable == null)
                    {
                        this.dataModelInfoRenderable =
                            new Editor.EditorDataModelInfoRenderable(this);

                        this.dataModelInfoRenderable.Bounds = new RectangleF(
                            ChildMargin,
                            ChildMargin,
                            this.Width - (ChildMargin * 2.0f),
                            0.0f);

                        this.AddRenderable(this.dataModelInfoRenderable);
                    }
                    else if (this.dataModelInfoRenderable.Visible == false)
                    {
                        this.dataModelInfoRenderable.Visible = true;
                    }

                    this.dataModelInfoRenderable.DataContext = value;
                }
                else if (value is RuntimeDataModelDefinitionViewModel)
                {
                    if (this.dataModelInfoRenderable == null)
                    {
                        this.dataModelInfoRenderable =
                            new Runtime.RuntimeDataModelInfoRenderable(this);

                        this.dataModelInfoRenderable.Bounds = new RectangleF(
                            ChildMargin,
                            ChildMargin,
                            this.Width - (ChildMargin * 2.0f),
                            0.0f);

                        this.AddRenderable(this.dataModelInfoRenderable);
                    }
                    else if (this.dataModelInfoRenderable.Visible == false)
                    {
                        this.dataModelInfoRenderable.Visible = true;
                    }

                    this.dataModelInfoRenderable.DataContext = value;
                }
                else
                {
                    if (this.dataModelInfoRenderable != null)
                    {
                        this.dataModelInfoRenderable.DataContext = null;
                        this.dataModelInfoRenderable.Visible = false;
                    }

                    if (value != null)
                    {
                        Logger.Log(LogLevels.Error, "DataModelInfoViewport.DataModelRenderableDataContext : The data model renderable's data context is not of valid type.");
                    }
                }
            }
        }

        /// <summary>
        /// Get the size of all the list items.
        /// </summary>
        public SizeF ContentSize
        {
            get
            {
                if (this.dataModelInfoRenderable == null)
                {
                    return SizeF.Empty;
                }

                SizeF size = this.dataModelInfoRenderable.Size;
                size.Width += ChildMargin * 2.0f;
                size.Height += ChildMargin * 2.0f;

                return size;
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            if (this.dataModelInfoRenderable != null)
            {
                this.dataModelInfoRenderable.Size = new SizeF(
                    this.Width - (ChildMargin * 2.0f),
                    this.dataModelInfoRenderable.Height);
            }

            base.UpdateChildren(context);
        }
    }
}
