﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView
{
    /// <summary>
    /// Class for rendering data model property information.
    /// </summary>
    internal abstract class DataModelInfoPropertyRenderable : RectangleShape, ILogicalTreeElement
    {
        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The flag indicating if mouse is over this item.</summary>
        private bool isMouseOver = false;

        /// <summary>The flag indicating if the item is selected.</summary>
        private bool isSelected = false;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static DataModelInfoPropertyRenderable()
        {
            // Set up the fonts for rendering.
            ItemSmallFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            ItemLargeFont = new Font(FontFamily.GenericSansSerif, 9.0f);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public DataModelInfoPropertyRenderable(DataModelInfoRenderable owner) :
            base()
        {
            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;

            this.Owner = owner;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the small font.
        /// </summary>
        public static Font ItemSmallFont { get; private set; }

        /// <summary>
        /// Get the large font.
        /// </summary>
        public static Font ItemLargeFont { get; private set; }

        /// <summary>
        /// Get the owner data model collection.
        /// </summary>
        public DataModelInfoRenderable Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as DataModelInfoRenderable; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the flag indicating if mouse is over this item.
        /// </summary>
        public bool IsMouseOver
        {
            get
            {
                return this.isMouseOver;
            }

            set
            {
                if (this.isMouseOver != value)
                {
                    this.isMouseOver = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if this item is selected.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        protected virtual void UpdateAppearance()
        {
            if (this.IsSelected == true)
            {
                this.BorderColor = Color.Red;
                this.BorderThickness = 2.0f;
                this.FillColor = Color.Khaki;
            }
            else if (this.IsMouseOver == true)
            {
                this.BorderColor = Color.Transparent;
                this.BorderThickness = 0.0f;
                this.FillColor = Color.Khaki;
            }
            else
            {
                this.BorderColor = Color.Transparent;
                this.BorderThickness = 0.0f;
                this.FillColor = Color.Transparent;
            }
        }
    }
}
