﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;

using EffectMaker.DataModelMaker.UIControls.Interfaces;
using EffectMaker.DataModelMaker.UIControls.SourceCodeEditor;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor
{
    /// <summary>
    /// User control class for editor data model property editing.
    /// </summary>
    public partial class EditorDataModelPropertyEditor : UIUserControl, IDataModelEditor
    {
        /// <summary>The versions and the corresponding actions.</summary>
        private Dictionary<Version, VersionActions> versionsAndActions =
            new Dictionary<Version, VersionActions>();

        /// <summary>Holds property name and its value to monitor value changes.</summary>
        private Dictionary<string, object> propertyValueMap =
            new Dictionary<string, object>();

        /// <summary>The suggestion strings for the property type text box.</summary>
        private AutoCompleteStringCollection typeSuggestions =
            new AutoCompleteStringCollection();

        /// <summary>The selected version.</summary>
        private Version selectedVersion = null;

        /// <summary>The flag indicating whether to suppress the update of the version list.</summary>
        private bool suppressVersionListUpdate = false;

        /// <summary>The custom getter source code of the editing version.</summary>
        private string customGetterSourceCode = string.Empty;

        /// <summary>The custom setter source code of the editing version.</summary>
        private string customSetterSourceCode = string.Empty;

        /// <summary>The version converter source code of the editing version.</summary>
        private string versionConverterSourceCode = string.Empty;

        /// <summary>
        /// Constructor.
        /// </summary>
        public EditorDataModelPropertyEditor()
        {
            this.InitializeComponent();

            this.typeSuggestions.AddRange(
                (from tp in TypeManager.EditorTypes select tp.FullName).ToArray());
            this.txtType.AutoCompleteCustomSource = this.typeSuggestions;

            this.chkCustomGetter.CheckedChanged += (s, e) =>
            {
                this.chkBackingField.Enabled = this.CustomGetterEnabled | this.CustomSetterEnabled;
            };

            this.chkCustomSetter.CheckedChanged += (s, e) =>
            {
                this.chkBackingField.Enabled = this.CustomGetterEnabled | this.CustomSetterEnabled;
            };

            this.versionListView.SizeChanged += (s, e) =>
            {
                this.versionListView.Columns[0].Width = this.versionListView.ClientSize.Width;
            };

            this.editCustomGetterBtn.Click += (s, e) => this.ShowCustomGetterSourceCodeEditor();
            this.editCustomSetterBtn.Click += (s, e) => this.ShowCustomSetterSourceCodeEditor();
            this.editVersionConverterBtn.Click += (s, e) => this.ShowVersionConverterSourceCodeEditor();

            this.versionListContextMenu.Opening += this.OnVersionListContextMenuOpening;

            this.lblTypeWarning.Visibility = Visibility.Collapsed;
            this.lblNameWarning.Visibility = Visibility.Collapsed;
            this.lblDefaultValueWarning.Visibility = Visibility.Collapsed;

            // These are the properties we want to monitor for the value modifications.
            this.AddBinderAndValueMonitor("DataModelName", "DataModelName");
            this.AddBinderAndValueMonitor("PropertyType", "PropertyType");
            this.AddBinderAndValueMonitor("PropertyName", "Name");
            this.AddBinderAndValueMonitor("XmlIgnore", "XmlIgnore");
            this.AddBinderAndValueMonitor("EnableBackingField", "EnableFieldDeclaration");
            this.AddBinderAndValueMonitor("EnableNullOriginal", "EnableNullOriginal");
            this.AddBinderAndValueMonitor("PropertyDefaultValue", "DefaultValue");
            this.AddBinderAndValueMonitor("PropertyDescription", "Description");
            this.AddBinderAndValueMonitor("CustomSetterEnabled", "CustomSetterEnabled");
            this.AddBinderAndValueMonitor("CustomGetterEnabled", "CustomGetterEnabled");
            this.AddBinderAndValueMonitor("AlwaysApplyVersionConverter", "AlwaysApplyVersionConverter");

            // We don't monitor the value modification for the properties below.
            this.AddBinding("CustomSetter", "CustomSetter");
            this.AddBinding("CustomGetter", "CustomGetter");
            this.AddBinding("VersionConverter", "VersionConverter");
            this.AddBinding("IsTypeWarningVisible", "IsPropertyTypeInvalid");
            this.AddBinding("VersionsAndActions", "VersionsAndActions");

            this.AddBinding("OnCommitEditingExecutable", "OnCommitEditingExecutable");
            this.AddBinding("OnCancelEditingExecutable", "OnCancelEditingExecutable");
            this.AddBinding("OnLockEditingVersionExecutable", "OnLockEditingVersionExecutable");
            this.AddBinding("OnUnlockEditingVersionExecutable", "OnUnlockEditingVersionExecutable");
            this.AddBinding("OnAddVersionExecutable", "OnAddVersionExecutable");
            this.AddBinding("OnRemoveVersionExecutable", "OnRemoveVersionExecutable");
            this.AddBinding("OnMoveVersionExecutable", "OnMoveVersionExecutable");
        }

        /// <summary>
        /// Get or set the name of the owner data model of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string DataModelName { get; set; }

        /// <summary>
        /// Get or set the value type of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string PropertyType
        {
            get { return this.txtType.Text; }
            set { this.txtType.Text = value; }
        }

        /// <summary>
        /// Get or set the value type of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsTypeWarningVisible
        {
            get { return this.lblTypeWarning.Visible; }
            set { this.lblTypeWarning.Visible = value; }
        }

        /// <summary>
        /// Get or set the name of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string PropertyName
        {
            get { return this.txtName.Text; }
            set { this.txtName.Text = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to ignore this property while serialization.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool XmlIgnore
        {
            get { return this.chkIgnoreXml.Checked; }
            set { this.chkIgnoreXml.Checked = value; }
        }

        /// <summary>
        /// Get or set flag indicating whether to generate backing field for the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool EnableBackingField
        {
            get { return this.chkBackingField.Checked; }
            set { this.chkBackingField.Checked = value; }
        }

        /// <summary>
        /// ReadXml時に規定値をnullとして扱うか否かを取得または設定します(ユーザーデータインスタンス用)
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool EnableNullOriginal
        {
            get { return this.chkNullOriginal.Checked; }
            set { this.chkNullOriginal.Checked = value; }
        }

        /// <summary>
        /// Get or set the default value of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string PropertyDefaultValue
        {
            get { return this.txtDefaultValue.Text; }
            set { this.txtDefaultValue.Text = value; }
        }

        /// <summary>
        /// Get or set the description of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string PropertyDescription
        {
            get { return this.txtDescription.Text; }
            set { this.txtDescription.Text = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to enable custom setter.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool CustomSetterEnabled
        {
            get { return this.chkCustomSetter.Checked; }
            set { this.chkCustomSetter.Checked = value; }
        }

        /// <summary>
        /// Get or set the custom setter for the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string CustomSetter
        {
            get
            {
                return this.customSetterSourceCode;
            }

            set
            {
                this.customSetterSourceCode = value;
                this.chkCustomSetter.ForeColor =
                    string.IsNullOrEmpty(value) == false ? Color.Black : Color.Gray;
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether to enable custom getter.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool CustomGetterEnabled
        {
            get { return this.chkCustomGetter.Checked; }
            set { this.chkCustomGetter.Checked = value; }
        }

        /// <summary>
        /// Get or set the custom getter for the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string CustomGetter
        {
            get
            {
                return this.customGetterSourceCode;
            }

            set
            {
                this.customGetterSourceCode = value;
                this.chkCustomGetter.ForeColor =
                    string.IsNullOrEmpty(value) == false ? Color.Black : Color.Gray;
            }
        }

        /// <summary>
        /// Get or set the version converter for the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string VersionConverter
        {
            get
            {
                return this.versionConverterSourceCode;
            }

            set
            {
                this.versionConverterSourceCode = value;
                this.versionConverterLabel.ForeColor =
                    string.IsNullOrEmpty(value) == false ? Color.Black : Color.Gray;
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether to always apply the version converter.
        /// The version converter only generated to the source code at this version if
        /// this flag is false, otherwise, the version converter will be applied to all
        /// the versions after this version.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool AlwaysApplyVersionConverter
        {
            get { return this.chkAlwaysApplyConverter.Checked; }
            set { this.chkAlwaysApplyConverter.Checked = value; }
        }

        /// <summary>
        /// Get or set the list of versions and the action took to the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IEnumerable<KeyValuePair<Version, VersionActions>> VersionsAndActions
        {
            set
            {
                if (this.suppressVersionListUpdate == true)
                {
                    return;
                }

                this.selectedVersion = null;
                this.versionsAndActions.Clear();

                ListViewItem lastItem = null;
                ListViewItem itemToSelect = null;

                Version selectedVersion = null;
                if (this.versionListView.SelectedItems.Count > 0)
                {
                    selectedVersion = this.versionListView.SelectedItems[0].Tag as Version;
                }

                this.versionListView.BeginUpdate();

                this.versionListView.Items.Clear();
                foreach (KeyValuePair<Version, VersionActions> entry in value)
                {
                    var item = new ListViewItem();
                    item.Tag = entry.Key;

                    switch (entry.Value)
                    {
                        case VersionActions.Create:
                            lastItem = item;
                            item.ForeColor = System.Drawing.Color.DodgerBlue;
                            item.Text = string.Format(Properties.Resources.PropertyVersionAndActionCreated, entry.Key.ToString());
                            break;

                        case VersionActions.Modify:
                            lastItem = item;
                            item.ForeColor = System.Drawing.Color.Orange;
                            item.Text = string.Format(Properties.Resources.PropertyVersionAndActionModified, entry.Key.ToString());
                            break;

                        case VersionActions.Delete:
                            lastItem = item;
                            item.ForeColor = System.Drawing.Color.Crimson;
                            item.Text = string.Format(Properties.Resources.PropertyVersionAndActionDeleted, entry.Key.ToString());
                            break;

                        default:
                            item.ForeColor = System.Drawing.Color.LightGray;
                            item.Text = entry.Key.ToString();
                            break;
                    }

                    if (entry.Key == selectedVersion)
                    {
                        itemToSelect = item;
                    }

                    this.versionListView.Items.Add(item);
                    this.versionsAndActions.Add(entry.Key, entry.Value);
                }

                this.versionListView.EndUpdate();

                // Select the originally selected version or the last editable version.
                if (itemToSelect != null)
                {
                    itemToSelect.Selected = true;
                }
                else if (lastItem != null)
                {
                    lastItem.Selected = true;
                }
            }
        }

        /// <summary>
        /// Get or set the executable for committing edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for locking the editing version of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnLockEditingVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for unlocking the editing version of the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnUnlockEditingVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for adding new version to the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnAddVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for removing version from the property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnRemoveVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for moving a version of the property to another version.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnMoveVersionExecutable { get; set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public override object DataContext
        {
            get
            {
                return base.DataContext;
            }

            set
            {
                base.DataContext = value;
                this.CollectOriginalValues();
                this.chkBackingField.Enabled = this.CustomGetterEnabled | this.CustomSetterEnabled;
            }
        }

        /// <summary>
        /// Confirm the edited contents and commit the modification to the data source.
        /// </summary>
        /// <returns>
        /// False if there are problems in the modifications and should cancel the action.
        /// </returns>
        public bool CommitEditing()
        {
            if (this.mainSlipContainer.Panel2.Enabled == true &&
                this.OnCommitEditingExecutable != null)
            {
                this.OnCommitEditingExecutable.Execute(new KeyValuePair<string, object>[]
                {
                    new KeyValuePair<string, object>("PropertyType", this.PropertyType),
                    new KeyValuePair<string, object>("Name", this.PropertyName),
                    new KeyValuePair<string, object>("XmlIgnore", this.XmlIgnore),
                    new KeyValuePair<string, object>("EnableFieldDeclaration", this.EnableBackingField),
                    new KeyValuePair<string, object>("EnableNullOriginal", this.EnableNullOriginal),
                    new KeyValuePair<string, object>("DefaultValue", this.PropertyDefaultValue),
                    new KeyValuePair<string, object>("Description", this.PropertyDescription),
                    new KeyValuePair<string, object>("CustomSetterEnabled", this.CustomSetterEnabled),
                    new KeyValuePair<string, object>("CustomGetterEnabled", this.CustomGetterEnabled),
                    new KeyValuePair<string, object>("AlwaysApplyVersionConverter", this.AlwaysApplyVersionConverter),
                });
            }

            // Is the property type valid?
            if (this.IsTypeWarningVisible == true)
            {
                return false;
            }

            // Unlock the editing version.
            this.OnUnlockEditingVersionExecutable.Execute(null);

            return true;
        }

        /// <summary>
        /// Cancel the edited contents.
        /// </summary>
        /// <param name="deleteContents">True to delete the editing contents.</param>
        public void CancelEditing(bool deleteContents)
        {
            // Unlock the editing version.
            this.OnUnlockEditingVersionExecutable.Execute(null);

            if (deleteContents == true &&
                this.OnCancelEditingExecutable != null)
            {
                this.OnCancelEditingExecutable.Execute(null);
            }
        }

        /// <summary>
        /// Handle Opening event for the context menu of the version list view.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguements.</param>
        private void OnVersionListContextMenuOpening(object sender, CancelEventArgs e)
        {
            if (this.selectedVersion == null)
            {
                e.Cancel = true;
                return;
            }

            // First find out the created and deleted version of the property.
            var createdVersion = (from pair in this.versionsAndActions
                                  where pair.Value == VersionActions.Create
                                  select pair.Key).FirstOrDefault();

            var deletedVersion = (from pair in this.versionsAndActions
                                  where pair.Value == VersionActions.Delete
                                  select pair.Key).FirstOrDefault();

            var selectedVersionAction = this.versionsAndActions[this.selectedVersion];
            if (selectedVersionAction != VersionActions.None)
            {
                // The property already has action for the selected version.
                this.addModifyVersionContextMenuItem.Enabled = false;
                this.addDeleteVersionContextMenuItem.Enabled = false;

                // The created version cannot be removed.
                this.deleteHistoryVersionContextMenuItem.Enabled =
                    selectedVersionAction != VersionActions.Create;

                IEnumerable<Version> availableVersions = null;
                switch (selectedVersionAction)
                {
                    case VersionActions.Create:
                        {
                            // Find the first modify or delete version.
                            // First select all the Modify and Delete versions,
                            // then find the one with the smallest version.
                            var tmpVersions = from pair in this.versionsAndActions
                                              where pair.Value == VersionActions.Modify ||
                                                    pair.Value == VersionActions.Delete
                                              select pair.Key;

                            var minVersion = tmpVersions.Any() ? tmpVersions.Min() : null;

                            // The available versions are the versions before the first modify version.
                            availableVersions = from pair in this.versionsAndActions
                                                where (minVersion == null || pair.Key < minVersion) &&
                                                      (pair.Value == VersionActions.None)
                                                select pair.Key;
                            break;
                        }

                    case VersionActions.Modify:
                        availableVersions = from pair in this.versionsAndActions
                                            where (pair.Key > createdVersion) &&
                                                  (deletedVersion == null || pair.Key < deletedVersion) &&
                                                  (pair.Value == VersionActions.None)
                                            select pair.Key;
                        break;

                    case VersionActions.Delete:
                        {
                            // Find the first modify or create version.
                            // First select all the Modify and Create versions,
                            // then find the one with the largest version.
                            var tmpVersions = from pair in this.versionsAndActions
                                              where pair.Value == VersionActions.Modify ||
                                                    pair.Value == VersionActions.Create
                                              select pair.Key;

                            var maxVersion = tmpVersions.Any() ? tmpVersions.Max() : null;

                            // The available versions are the versions before the first modify version.
                            availableVersions = from pair in this.versionsAndActions
                                                where (maxVersion == null || pair.Key > maxVersion) &&
                                                      (pair.Value == VersionActions.None)
                                                select pair.Key;
                            break;
                        }
                }

                // First clear the available versions from the drop down menu.
                this.moveHistoryVersionContextMenuItem.DropDownItems.Clear();

                if (availableVersions == null ||
                    availableVersions.Any() == false)
                {
                    // The version history has no available versions to move to.
                    this.moveHistoryVersionContextMenuItem.Enabled = false;
                }
                else
                {
                    // The version history has available versions to move to.
                    // Add the versions to the drop down menu.
                    this.moveHistoryVersionContextMenuItem.Enabled = false;
                    this.moveHistoryVersionContextMenuItem.Enabled = true;
                    this.moveHistoryVersionContextMenuItem.DropDownItems.AddRange(
                        availableVersions.Select(v => new ToolStripMenuItem(v.ToString(), null, this.OnMoveVersionHistory) { Tag = v }).ToArray());
                }
            }
            else
            {
                // The selected version has no action for this property yet.
                this.deleteHistoryVersionContextMenuItem.Enabled = false;
                this.moveHistoryVersionContextMenuItem.Enabled = false;

                if (deletedVersion == null)
                {
                    // The property has no deleted version.
                    this.addDeleteVersionContextMenuItem.Enabled = true;

                    // The modify version has to place after the created version.
                    this.addModifyVersionContextMenuItem.Enabled =
                        this.selectedVersion > createdVersion;
                }
                else
                {
                    // Deleted version already exists.
                    this.addDeleteVersionContextMenuItem.Enabled = false;

                    // Modify version has to place between created and deleted versions.
                    if (this.selectedVersion > createdVersion &&
                        this.selectedVersion < deletedVersion)
                    {
                        this.addModifyVersionContextMenuItem.Enabled = true;
                    }
                    else
                    {
                        this.addModifyVersionContextMenuItem.Enabled = false;
                    }
                }
            }
        }

        /// <summary>
        /// Handle Click event for "Add Modify Version" context menu item.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguements.</param>
        private void OnAddModifyVersion(object sender, EventArgs e)
        {
            if (this.selectedVersion == null)
            {
                return;
            }

            this.OnAddVersionExecutable.Execute(new object[]
                {
                    this.selectedVersion,
                    VersionActions.Modify
                });
        }

        /// <summary>
        /// Handle Click event for "Add Deleted Version" context menu item.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguements.</param>
        private void OnAddDeletedVersion(object sender, EventArgs e)
        {
            if (this.selectedVersion == null)
            {
                return;
            }

            this.OnAddVersionExecutable.Execute(new object[]
                {
                    this.selectedVersion,
                    VersionActions.Delete
                });
        }

        /// <summary>
        /// Handle Click event for "Move Version History" context menu items.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguements.</param>
        private void OnMoveVersionHistory(object sender, EventArgs e)
        {
            if (this.selectedVersion == null)
            {
                return;
            }

            var item = sender as ToolStripMenuItem;
            if (item == null)
            {
                return;
            }

            this.OnMoveVersionExecutable.Execute(new object[]
                {
                    this.selectedVersion,
                    item.Tag as Version
                });
        }

        /// <summary>
        /// Handle Click event for "Delete Version History" context menu item.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguements.</param>
        private void OnDeleteVersionHistory(object sender, EventArgs e)
        {
            if (this.selectedVersion == null)
            {
                return;
            }

            this.OnRemoveVersionExecutable.Execute(this.selectedVersion);
        }

        /// <summary>
        /// Handle SelectedIndexChanged event for the version list view.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguements.</param>
        private void OnVersionSelected(object sender, EventArgs e)
        {
            if (this.versionListView.SelectedItems.Count <= 0)
            {
                return;
            }

            var version = this.versionListView.SelectedItems[0].Tag as Version;
            if (this.selectedVersion == version)
            {
                return;
            }

            // Check if any values are modified, prompt user to apply the changes,
            // otherwise the user would lost the modifications.
            if (this.IsAnyMonitoredValueChanged() == true)
            {
                DialogResult result = MessageBox.Show(
                    this,
                    Properties.Resources.WarningDataModelPropertyModifiedWhenSelectVersion,
                    Properties.Resources.WarningMessageBoxTitle,
                    MessageBoxButtons.YesNoCancel,
                    MessageBoxIcon.Exclamation);

                bool revertSelection = false;
                if (result == DialogResult.Yes)
                {
                    if (this.CommitEditing() == false)
                    {
                        revertSelection = true;
                    }
                }
                else if (result == DialogResult.Cancel)
                {
                    revertSelection = true;
                }

                // Revert the selected version to the original one.
                if (revertSelection == true)
                {
                    this.SelectVersion(this.selectedVersion);
                    return;
                }
            }

            // Save the selected version.
            this.selectedVersion = version;

            // Check if there is a history at the selected version.
            VersionActions action;
            if (this.versionsAndActions.TryGetValue(version, out action) == true)
            {
                if (action == VersionActions.None ||
                    action == VersionActions.Delete)
                {
                    // No history made, disable controls to prevent user input.
                    this.mainSlipContainer.Panel2.Enabled = false;

                    // Revert the modified values.
                    this.RevertToOriginalValues();

                    return;
                }
                else if (this.mainSlipContainer.Panel2.Enabled == false)
                {
                    // A history is found, enable the controls for user input.
                    this.mainSlipContainer.Panel2.Enabled = true;
                }
            }

            this.suppressVersionListUpdate = true;

            // Request the data model property to lock its editing version at the
            // selected version, otherwise the property would automatically sync
            // its editing version to the data model root definition's editing version.
            this.OnLockEditingVersionExecutable.Execute(this.selectedVersion);

            // Update all the values from the property definition.
            foreach (var binder in this.Bindings)
            {
                binder.UpdateElement();
            }

            this.suppressVersionListUpdate = false;

            // Update the original values with the updated ones.
            this.CollectOriginalValues();
        }

        /// <summary>
        /// Add a binder for the specified property and a value monitor for the property.
        /// </summary>
        /// <param name="myPropertyName">The property name of this class.</param>
        /// <param name="dataSrcPropertyName">The property name of the data source.</param>
        private void AddBinderAndValueMonitor(string myPropertyName, string dataSrcPropertyName)
        {
            // First add the binder.
            this.AddBinding(myPropertyName, dataSrcPropertyName);

            // Add the property name to our dictionary to be monitoring.
            this.propertyValueMap.Add(myPropertyName, null);
        }

        /// <summary>
        /// Collect the values of the properties that we want to monitor
        /// the modifications.
        /// </summary>
        private void CollectOriginalValues()
        {
            Type myType = this.GetType();

            string[] keys = this.propertyValueMap.Keys.ToArray();
            foreach (string propertyName in keys)
            {
                PropertyInfo propertyInfo = myType.GetProperty(propertyName);
                if (propertyInfo == null)
                {
                    // The property is not found, just set the value to null.
                    this.propertyValueMap[propertyName] = null;
                }
                else
                {
                    // Save the current value of the property.
                    this.propertyValueMap[propertyName] = propertyInfo.GetValue(this);
                }
            }
        }

        /// <summary>
        /// Revert the changes to the original values.
        /// </summary>
        private void RevertToOriginalValues()
        {
            Type myType = this.GetType();
            foreach (string propertyName in this.propertyValueMap.Keys)
            {
                PropertyInfo propertyInfo = myType.GetProperty(propertyName);
                if (propertyInfo == null)
                {
                    continue;
                }

                object origValue = this.propertyValueMap[propertyName];

                propertyInfo.SetValue(this, origValue);
            }

            this.chkBackingField.Enabled = this.CustomGetterEnabled | this.CustomSetterEnabled;
        }

        /// <summary>
        /// Check if any of the values of the monitored properties has changed.
        /// </summary>
        /// <returns>True if some values are modified.</returns>
        private bool IsAnyMonitoredValueChanged()
        {
            Type myType = this.GetType();
            foreach (string propertyName in this.propertyValueMap.Keys)
            {
                PropertyInfo propertyInfo = myType.GetProperty(propertyName);
                if (propertyInfo == null)
                {
                    continue;
                }

                object origValue = this.propertyValueMap[propertyName];
                object currValue = propertyInfo.GetValue(this);

                if (object.Equals(origValue, currValue) == false)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Select the specified version as the editing version.
        /// </summary>
        /// <param name="version">The version to select.</param>
        private async void SelectVersion(Version version)
        {
            // We want to wait a bit for the list view to finish
            // the previous SelectedIndexChanged event handling
            // because the WinForm ListView contains a timer that
            // fires the event some time later the user changed
            // the selection.
            await System.Threading.Tasks.Task.Delay(100);

            this.versionListView.SelectedItems.Clear();
            foreach (ListViewItem item in this.versionListView.Items)
            {
                var v = item.Tag as Version;
                if (v != null && v == version)
                {
                    item.Selected = true;
                    return;
                }
            }
        }

        /// <summary>
        /// Show source code editor for custom getter.
        /// </summary>
        private void ShowCustomGetterSourceCodeEditor()
        {
            var title = string.Format(
                Properties.Resources.SourceCodeEditorTitle_CustomPropertyGetter,
                this.PropertyName);

            string template = string.Empty;
            if (this.CustomGetterEnabled == true)
            {
                template += string.Format(
                    Properties.Resources.SourceCodeEditorTitle_CustomPropertyGetterTemplate,
                    "{SourceCodeTemplate:Editable}");
            }

            if (this.CustomSetterEnabled == true)
            {
                if (this.CustomGetterEnabled == true)
                {
                    template += Environment.NewLine + Environment.NewLine;
                }

                template += string.Format(
                    Properties.Resources.SourceCodeEditorTitle_CustomPropertySetterTemplate,
                    RegexUtility.IndentString(this.CustomSetter, Environment.NewLine, 8));
            }

            template = string.Format(
                Properties.Resources.SourceCodeEditorTitle_CustomPropertyTemplate,
                TypeManager.ToShortTypeName(this.PropertyType),
                this.PropertyName,
                template);

            var dialog = new SourceCodeEditorDialog(title, this.DataContext, "CustomGetter")
            {
                SourceCodeTemplate = template,
                SourceCodeIndent = 8,
            };

            dialog.ShowDialog(this);
        }

        /// <summary>
        /// Show source code editor for custom setter.
        /// </summary>
        private void ShowCustomSetterSourceCodeEditor()
        {
            var title = string.Format(
                Properties.Resources.SourceCodeEditorTitle_CustomPropertySetter,
                this.PropertyName);

            string template = string.Empty;
            if (this.CustomGetterEnabled == true)
            {
                template += string.Format(
                    Properties.Resources.SourceCodeEditorTitle_CustomPropertyGetterTemplate,
                    RegexUtility.IndentString(this.CustomGetter, Environment.NewLine, 8));
            }

            if (this.CustomSetterEnabled == true)
            {
                if (this.CustomGetterEnabled == true)
                {
                    template += Environment.NewLine + Environment.NewLine;
                }

                template += string.Format(
                    Properties.Resources.SourceCodeEditorTitle_CustomPropertySetterTemplate,
                    "{SourceCodeTemplate:Editable}");
            }

            template = string.Format(
                Properties.Resources.SourceCodeEditorTitle_CustomPropertyTemplate,
                TypeManager.ToShortTypeName(this.PropertyType),
                this.PropertyName,
                template);

            var dialog = new SourceCodeEditorDialog(title, this.DataContext, "CustomSetter")
            {
                SourceCodeTemplate = template,
                SourceCodeIndent = 8,
            };

            dialog.ShowDialog(this);
        }

        /// <summary>
        /// Show source code editor for version converter method.
        /// </summary>
        private void ShowVersionConverterSourceCodeEditor()
        {
            var title = string.Format(
                Properties.Resources.SourceCodeEditorTitle_PropertyVersionConverter,
                this.PropertyName,
                this.selectedVersion);

            var template = string.Format(
                Properties.Resources.SourceCodeEditorTitle_PropertyVersionConverterTemplate,
                TypeManager.ToShortTypeName(this.PropertyType),
                this.PropertyName,
                this.DataModelName);

            var dialog = new SourceCodeEditorDialog(title, this.DataContext, "VersionConverter")
            {
                SourceCodeTemplate = template,
                SourceCodeIndent = 4,
            };

            dialog.ShowDialog(this);
        }
    }
}
