﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.UIControls.Selection;

using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelMaker.UIControls.ConversionView
{
    /// <summary>
    /// The renderable class for rendering the conversion connections.
    /// (The line segment that connects the connectors.)
    /// </summary>
    internal class ConnectionRenderable : RenderableBase
    {
        /// <summary>The dictionary maps the connectors' Guid with the connections.</summary>
        private static Dictionary<Guid, List<ConnectionRenderable>> connectionMap =
            new Dictionary<Guid, List<ConnectionRenderable>>();

        /// <summary>The source of the connection.</summary>
        private ISourceConnectible source = null;

        /// <summary>The destination of the connection.</summary>
        private IDestinationConnectible destination = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ConnectionRenderable() :
            base()
        {
            this.Vertex1 = PointF.Empty;
            this.Vertex2 = PointF.Empty;
            this.BorderColor = Color.PowderBlue;
            this.BorderThickness = 1.0f;
            this.PickThreshold = 5.0f;
        }

        /// <summary>
        /// Enum for the relationships between items.
        /// </summary>
        private enum RelationTypes
        {
            /// <summary>The item is this item itself.</summary>
            Self,

            /// <summary>The item is related to this item.</summary>
            Related,

            /// <summary>The item has no relationship with this item.</summary>
            None
        }

        /// <summary>
        /// Get or set the boundary of the rectangle.
        /// </summary>
        public new RectangleF Bounds
        {
            get
            {
                return new RectangleF(
                    Math.Min(this.Vertex1.X, this.Vertex2.X),
                    Math.Min(this.Vertex1.Y, this.Vertex2.Y),
                    Math.Abs(this.Vertex1.X - this.Vertex2.X),
                    Math.Abs(this.Vertex1.Y - this.Vertex2.Y));
            }
        }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public PointF Vertex1 { get; set; }

        /// <summary>
        /// Get or set the location of the other end of the curve.
        /// </summary>
        public PointF Vertex2 { get; set; }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public PointF TransformedVertex1
        {
            get { return this.Vertex1; }
        }

        /// <summary>
        /// Get or set the location of the other end of the curve.
        /// </summary>
        public PointF TransformedVertex2
        {
            get { return this.Vertex2; }
        }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public PointF ClippedVertex1 { get; protected set; }

        /// <summary>
        /// Get or set the location of the other end of the curve.
        /// </summary>
        public PointF ClippedVertex2 { get; protected set; }

        /// <summary>
        /// Get or set the threshold for mouse picking.
        /// </summary>
        public float PickThreshold { get; set; }

        /// <summary>
        /// Get or set the source of the connection.
        /// </summary>
        public ISourceConnectible Source
        {
            get { return this.source; }
            set { this.SetConnectible(ref this.source, value); }
        }

        /// <summary>
        /// Get or set the destination of the connection.
        /// </summary>
        public IDestinationConnectible Destination
        {
            get { return this.destination; }
            set { this.SetConnectible(ref this.destination, value); }
        }

        /// <summary>
        /// Find the connections that connect to the given connector.
        /// </summary>
        /// <param name="connector">The connector.</param>
        /// <returns>The connections.</returns>
        public static IEnumerable<ConnectionRenderable> FindConnections(ConnectorRenderable connector)
        {
            if (connector == null)
            {
                return Enumerable.Empty<ConnectionRenderable>();
            }

            List<ConnectionRenderable> connectionList;
            if (connectionMap.TryGetValue(connector.ConnectorID, out connectionList) == false ||
                connectionList == null)
            {
                return Enumerable.Empty<ConnectionRenderable>();
            }

            return connectionList;
        }

        /// <summary>
        /// Test if the given point lies on this object.
        /// </summary>
        /// <param name="p">The point.</param>
        /// <returns>True if the point picks this object.</returns>
        public override bool Pick(PointF p)
        {
            if (this.CanPick == false || this.Visible == false)
            {
                return false;
            }

            if (p.X < this.ClippedVertex1.X || p.X > this.ClippedVertex2.X)
            {
                return false;
            }

            // Compute the distance from the point.
            float distance =
                MathUtility.ComputePointLineDistance(p, this.ClippedVertex1, this.ClippedVertex2);

            return distance <= this.PickThreshold;
        }

        /// <summary>
        /// Render the item.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>False when the item is clipped and need not to be rendered.</returns>
        public override bool Draw(Graphics g)
        {
            if (this.Visible == false)
            {
                return false;
            }

            if (this.BorderThickness <= 0.0f)
            {
                return false;
            }

            if (this.Source == null || this.Destination == null)
            {
                return false;
            }

            this.Vertex1 = this.source.TransformedConnectorLocation;
            this.Vertex2 = this.destination.TransformedConnectorLocation;

            // The vertices are took from the transformed location of the
            // source and destination, hence we don't need to transform again.
            this.TransformedBounds = this.Bounds;

            bool isInViewRectangle = false;

            // Do clipping with the view rectangle.
            PointF p1 = PointF.Empty;
            PointF p2 = PointF.Empty;
            if (this.SavedRenderContext.IsViewRenctangleEmpty == false)
            {
                isInViewRectangle = RenderUtility.ClipLineSegment(
                    this.SavedRenderContext.ViewRectangle,
                    this.TransformedVertex1,
                    this.TransformedVertex2,
                    out p1,
                    out p2);
            }

            this.ClippedVertex1 = p1;
            this.ClippedVertex2 = p2;

            this.ClippedBounds =
                new RectangleF(
                    Math.Min(this.ClippedVertex1.X, this.ClippedVertex2.X),
                    Math.Min(this.ClippedVertex1.Y, this.ClippedVertex2.Y),
                    Math.Abs(this.ClippedVertex1.X - this.ClippedVertex2.X),
                    Math.Abs(this.ClippedVertex1.Y - this.ClippedVertex2.Y));

            if (isInViewRectangle == false)
            {
                // The line segment does not lie in the view rectangle, don't render it.
                return false;
            }

            if ((int)this.ClippedVertex1.X == (int)this.ClippedVertex2.X &&
                (int)this.ClippedVertex1.Y == (int)this.ClippedVertex2.Y)
            {
                // The line segment's length is zero, don't render it.
                return false;
            }

            if (this.IsUpdateAndRenderingSuspended == false)
            {
                this.SuspendUpdateAndRendering();

                this.DrawBackground(g);
                this.DrawForeground(g);
                this.DrawBorder(g);
                this.DrawChildren(g);

                this.ResumeUpdateAndRendering();

                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Release the resources the renderable uses.
        /// </summary>
        /// <param name="isDisposing">True if the renderable is being disposed.</param>
        protected override void Dispose(bool isDisposing)
        {
            this.Source = null;
            this.Destination = null;

            base.Dispose(isDisposing);
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            // Save the render context for rendering.
            this.SavedRenderContext = context.Clone() as RenderContext;

            this.UpdateAppearance();

            // To ensure that we get the updated location of source and destination,
            // we update in Draw() method.
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBorder(Graphics g)
        {
            // Draw the line segment.
            using (Pen pen = new Pen(this.BorderColor, this.BorderThickness))
            {
                g.DrawLine(pen, this.ClippedVertex1, this.ClippedVertex2);
            }
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        private void UpdateAppearance()
        {
            object selectedObj =
                SelectionManager.GetSelectedObject(SelectionGroups.BinaryConvertConnection);

            object mouseOverObj =
                SelectionManager.GetMouseOverObject(SelectionGroups.BinaryConvertConnection);

            RelationTypes selectedObjRelation = this.DetermineRelationship(selectedObj);
            RelationTypes mouseOverObjRelation = this.DetermineRelationship(mouseOverObj);

            if (selectedObjRelation == RelationTypes.Self)
            {
                this.BorderColor = Color.DodgerBlue;
                this.BorderThickness = 2.0f;
            }
            else if (mouseOverObjRelation == RelationTypes.Self)
            {
                this.BorderColor = Color.DeepSkyBlue;
                this.BorderThickness = 2.0f;
            }
            else if (selectedObjRelation == RelationTypes.Related ||
                     mouseOverObjRelation == RelationTypes.Related)
            {
                this.BorderColor = Color.DeepSkyBlue;
                this.BorderThickness = 2.0f;
            }
            else
            {
                this.BorderColor = Color.PowderBlue;
                this.BorderThickness = 1.0f;
            }
        }

        /// <summary>
        /// Determine the relationship between this item and the specified item.
        /// </summary>
        /// <param name="item">The item to determine relationship with.</param>
        /// <returns>The relationship.</returns>
        private RelationTypes DetermineRelationship(object item)
        {
            if (object.ReferenceEquals(item, this) == true)
            {
                return RelationTypes.Self;
            }
            else
            {
                // The connection is either selected nor mouse over.
                // Check if the selected object is related to this connection.
                if (item is ConnectionRenderable)
                {
                    var connection = (ConnectionRenderable)item;
                    foreach (var rd in connection.Destination.Connections)
                    {
                        if (rd == this)
                        {
                            // The item is related to this connection.
                            return RelationTypes.Related;
                        }
                    }
                }
                else if (item is ConnectorRenderable)
                {
                    var connector = (ConnectorRenderable)item;
                    foreach (var rd in ConnectionRenderable.FindConnections(connector))
                    {
                        if (rd == this)
                        {
                            // The connector is related to this connection.
                            return RelationTypes.Related;
                        }
                    }
                }
            }

            return RelationTypes.None;
        }

        /// <summary>
        /// Set connectible.
        /// </summary>
        /// <typeparam name="TValue">The type of the connectible member.</typeparam>
        /// <param name="connectibleMember">The member field to set the connectible to.</param>
        /// <param name="newConnectible">The new connectible.</param>
        private void SetConnectible<TValue>(
            ref TValue connectibleMember,
            TValue newConnectible) where TValue : IConnectible
        {
            if (object.ReferenceEquals(connectibleMember, newConnectible) == true)
            {
                return;
            }

            // First remove the connection from the connection map.
            if (connectibleMember != null &&
                connectibleMember.Connector != null)
            {
                // Get the connector Guid.
                Guid guid = connectibleMember.Connector.ConnectorID;

                // Get the list of connections that connect to this connector.
                List<ConnectionRenderable> connectionList;
                if (connectionMap.TryGetValue(guid, out connectionList) == true)
                {
                    connectionList.Remove(this);
                    if (connectionList.Count <= 0)
                    {
                        connectionMap.Remove(guid);
                    }
                }
            }

            // Set the connectible.
            if (newConnectible == null ||
                newConnectible.Connector == null)
            {
                connectibleMember = default(TValue);
            }
            else
            {
                connectibleMember = newConnectible;
            }

            // Add the new connection to the connection map.
            if (connectibleMember != null &&
                connectibleMember.Connector != null)
            {
                // Get the connector Guid.
                Guid guid = connectibleMember.Connector.ConnectorID;

                // Get the list of connections that connect to this connector.
                List<ConnectionRenderable> connectionList;
                if (connectionMap.TryGetValue(guid, out connectionList) == false)
                {
                    connectionList = new List<ConnectionRenderable>();
                    connectionMap.Add(guid, connectionList);
                }

                connectionList.Add(this);
            }
        }
    }
}
