﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectMaker.DataModelMaker.Core;
using EffectMaker.DataModelMaker.Core.Template;

namespace EffectMaker.DataModelMaker.Core.Writers
{
    /// <summary>
    /// 汎用ソースコードライタークラスです.
    /// </summary>
    /// <typeparam name="T">データ定義クラス</typeparam>
    public class SourceCodeWriter<T> : SourceCodeWriterBase where T : class
    {
        /// <summary>
        /// テンプレートヘルパーです.
        /// </summary>
        private TemplateHelper helper = null;

        /// <summary>
        /// 引数付きコンストラクタです.
        /// </summary>
        /// <param name="template">テンプレートです.</param>
        public SourceCodeWriter(string template)
        {
            this.helper = new TemplateHelper();
            this.helper.ParseTemplate(template, "EffectMaker.DataModelMaker.Core.Definitions.");
        }

        /// <summary>
        /// 引数付きコンストラクタです.
        /// </summary>
        /// <param name="template">テンプレートです.</param>
        /// <param name="nameSpace">検索に利用する.する名前空間です</param>
        public SourceCodeWriter(string template, string nameSpace)
        {
            this.helper = new TemplateHelper();
            this.helper.ParseTemplate(template, nameSpace);
        }

        /// <summary>
        /// Get or set the handler for each token being replaced with the actual value.
        /// The delegation is as below:
        /// <code>
        /// string TemplateReplaceHandler(TemplateToken token, string original)
        /// </code>
        /// Where token is the token being replacing,
        /// original is the original value that is replacing the token,
        /// and the return value is the actual value to replace the token.
        /// </summary>
        public Func<TemplateToken, string, string> TemplateReplaceHandler
        {
            get { return this.helper.TemplateReplaceHandler; }
            set { this.helper.TemplateReplaceHandler = value; }
        }

        /// <summary>
        /// テンプレートトークンを評価します.
        /// </summary>
        /// <param name="definition">データ定義です.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        internal override int EvaluateTemplateTokens(object definition)
        {
            // キャストできるか試す.
            var convertDefinition = definition as T;
            if (convertDefinition == null)
            {
                // 異常終了.
                return 0;
            }

            // トークンを評価.
            string[] tokens;
            int length = this.helper.EvaluateTokens(definition, out tokens);

            // 評価されたトークンを設定.
            this.EvaluatedTokens = tokens;

            // テキストの長さを返却.
            return length;
        }

        /// <summary>
        /// トークンを取得します.
        /// </summary>
        /// <param name="index">トークンのインデックス</param>
        /// <returns>トークンを返却します.</returns>
        internal TemplateToken GetToken(int index)
        {
            return this.helper.GetToken(index);
        }
    }
}
