﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Template;

namespace EffectMaker.DataModelMaker.Core.Writers
{
    /// <summary>
    /// Source code writer class for runtime data model entry.
    /// </summary>
    internal class RuntimeDataModelWriter : SourceCodeWriterBase
    {
        /// <summary>
        /// Static constructor.
        /// </summary>
        public RuntimeDataModelWriter()
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// Generate source code string from defintion.
        /// </summary>
        /// <param name="def">The definition.</param>
        /// <returns>The generated source code.</returns>
        public string Write(RuntimeFileDefinition def)
        {
            // 各ライターのリスト.
            var includeWriters = new List<SourceCodeWriterBase>();
            var structWriters = new List<SourceCodeWriterBase>();
            var namespaceBeginWriters = new List<SourceCodeWriterBase>();
            var namespaceEndWriters = new List<SourceCodeWriterBase>();

            // エントリーリストを生成.
            var entryListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

            // エントリーリストに追加.
            entryListMap.Add("Include", includeWriters);
            entryListMap.Add("Structure", structWriters);
            entryListMap.Add("NamespaceBegin", namespaceBeginWriters);
            entryListMap.Add("NamespaceEnd", namespaceEndWriters);

            var fileDef = new RuntimeFileDefinition();

            // テンプレートを読み込み.
            var fileWriter = new SourceCodeWriter<RuntimeFileDefinition>(Properties.Resources.RDM_Main);
            int length = fileWriter.EvaluateTemplateTokens(def);

            // インクルード定義を評価.
            length += this.EvaluateInclude(def, ref includeWriters);

            // 名前空間を評価.
            length += this.EvaluateNamespaceBegin(def, ref namespaceBeginWriters);

            // 構造体定義を評価.
            length += this.EvaluateStruct(def, ref structWriters);

            // 名前空間を評価.
            length += this.EvaluateNamespaceEnd(def, ref namespaceEndWriters);

            // ストリングビルダーを生成.
            StringBuilder builder = new StringBuilder(length);

            // コードファイルとなる文字列を生成.
            for (int i = 0; i < fileWriter.EvaluatedTokens.Length; ++i)
            {
                string token = fileWriter.EvaluatedTokens[i];
                if (token != null)
                {
                    builder.Append(token);
                }
                else
                {
                    TemplateToken templateToken = fileWriter.GetToken(i);
                    if ((templateToken != null) &&
                        (string.IsNullOrEmpty(templateToken.Name) == false) &&
                        (entryListMap.ContainsKey(templateToken.Name) == true))
                    {
                        this.WriteEntryList(builder, entryListMap[templateToken.Name]);
                    }
                }
            }

            // 生成された文字列を返却.
            return builder.ToString();
        }

        #region Private Methods

        /// <summary>
        /// インクルード定義を評価します.
        /// </summary>
        /// <param name="def">構造体定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        private int EvaluateInclude(RuntimeFileDefinition def, ref List<SourceCodeWriterBase> writers)
        {
            int length = 0;
            for (int i = 0; i < def.Includes.Count; ++i)
            {
                // ライターを生成.
                var writer = new SourceCodeWriter<RuntimeDataModelIncludeDefinition>(Properties.Resources.RDM_Include);

                // ShaderConverterのDLL化に伴う小畑さんから要求された特殊処理.
                // nw/math.hのみ特殊な#ifdef-#endif処理をつける.
                if (def.Includes[i].File == "nw/math.h")
                {
                    def.Includes[i].Header = "#ifndef _EFT_TOOL\r\n";
                    def.Includes[i].Footer = "\r\n#endif//_EFT_TOOL";
                }

                // テンプレートトークンを評価.
                length += writer.EvaluateTemplateTokens(def.Includes[i]);

                // ライターに追加.
                writers.Add(writer);
            }

            // 長さを返却.
            return length;
        }

        /// <summary>
        /// 構造体定義を評価します.
        /// </summary>
        /// <param name="def">ランタイムデータモデル定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        private int EvaluateStruct(RuntimeFileDefinition def, ref List<SourceCodeWriterBase> writers)
        {
            int length = 0;

            // 構造体定義の数だけループ.
            for (int i = 0; i < def.Structures.Count; ++i)
            {
                var structDef = def.Structures[i];

                // トークン数.
                int count = 0;

                // 各ライターのリスト.
                var memberWriters     = new List<SourceCodeWriterBase>();
                var memberFlipWriters = new List<SourceCodeWriterBase>();
                var flipMethodWriters = new List<SourceCodeWriterBase>();
                var superClassWriters = new List<SourceCodeWriterBase>();

                // エントリーリストを生成.
                var entryListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

                // エントリーリストに追加.
                entryListMap.Add("Member", memberWriters);
                entryListMap.Add("MemberFlip", memberFlipWriters);
                entryListMap.Add("FlipMethod", flipMethodWriters);
                entryListMap.Add("SuperClass", superClassWriters);

                // 構造体定義のテンプレートを評価.
                SourceCodeWriter<RuntimeDataModelDefinition> structWriter = null;

                // 継承している場合.
                if (structDef.SuperClassGuidList.Count > 0)
                {
                    // 継承しているテンプレートを設定.
                    structWriter = new SourceCodeWriter<RuntimeDataModelDefinition>(Properties.Resources.RDM_StructureInherit);

                    // トークンを評価.
                    length += structWriter.EvaluateTemplateTokens(structDef);
                    count  += structWriter.EvaluatedTokens.Length;
                }
                else
                {
                    // 継承していないテンプレートを設定.
                    structWriter = new SourceCodeWriter<RuntimeDataModelDefinition>(Properties.Resources.RDM_Structure);

                    // トークンを評価.
                    length += structWriter.EvaluateTemplateTokens(structDef);
                    count  += structWriter.EvaluatedTokens.Length;
                }

                // 上位クラス定義を評価.
                int resultCount = 0;
                length += this.EvaluateSuperClass(structDef, ref superClassWriters, out resultCount);
                count  += resultCount;

                // メンバー定義を評価.
                length += this.EvaluateMember(structDef, ref memberWriters, out resultCount);
                count  += resultCount;

                // メンバーのエンディアンフリップを評価.
                length += this.EvaluateMemberFlip(structDef, ref memberFlipWriters, out resultCount);
                count  += resultCount;

                // エンディアンフリップメソッドを評価.
                length += this.EvaluateFlipMethod(structDef, ref flipMethodWriters, out resultCount);
                count  += resultCount;

                int counter = 0;
                string[] tokens = new string[count];

                for (int j = 0; j < structWriter.EvaluatedTokens.Length; ++j)
                {
                    string token = structWriter.EvaluatedTokens[j];
                    if (token != null)
                    {
                        tokens[counter] = token;
                        counter++;
                    }
                    else
                    {
                        TemplateToken templateToken = structWriter.GetToken(j);
                        if ((templateToken != null)
                            && (string.IsNullOrEmpty(templateToken.Name) == false)
                            && (entryListMap.ContainsKey(templateToken.Name) == true))
                        {
                            foreach (var writer in entryListMap[templateToken.Name])
                            {
                                foreach (string val in writer.EvaluatedTokens)
                                {
                                    tokens[counter] = val;
                                    counter++;
                                }
                            }
                        }
                    }
                }

                // トークンを差し替え.
                structWriter.EvaluatedTokens = tokens;

                // ライターリストに追加.
                writers.Add(structWriter);
            }

            return length;
        }

        /// <summary>
        /// 名前空間のスコープの開始を評価します。
        /// </summary>
        /// <param name="def">ランタイムデータモデル定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <returns></returns>
        private int EvaluateNamespaceBegin(RuntimeFileDefinition def, ref List<SourceCodeWriterBase> writers)
        {
            int length = 0;
            for (int i = 0; i < def.Namespaces.Count; ++i)
            {
                // ライターを生成.
                var writer = new SourceCodeWriter<RuntimeDataModelNamespaceDefinition>(Properties.Resources.RDM_NamespaceBegin);

                // テンプレートトークンを評価.
                length += writer.EvaluateTemplateTokens(def.Namespaces[i]);

                // ライターに追加.
                writers.Add(writer);
            }

            return length;
        }

        /// <summary>
        /// 名前空間のスコープの終わりを評価します。
        /// </summary>
        /// <param name="def">ランタイムデータモデル定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <returns></returns>
        private int EvaluateNamespaceEnd(RuntimeFileDefinition def, ref List<SourceCodeWriterBase> writers)
        {
            int length = 0;
            for (int i = def.Namespaces.Count - 1; i >= 0; i--)
            {
                // ライターを生成
                var writer = new SourceCodeWriter<RuntimeDataModelNamespaceDefinition>(Properties.Resources.RDM_NamespaceEnd);

                // テンプレートトークンを評価
                length += writer.EvaluateTemplateTokens(def.Namespaces[i]);

                // ライターに追加
                writers.Add(writer);
            }

            return length;
        }

        /// <summary>
        /// 親クラスの定義を評価します.
        /// </summary>
        /// <param name="def">構造体定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <param name="tokenCount">トークン数です.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        private int EvaluateSuperClass(RuntimeDataModelDefinition def, ref List<SourceCodeWriterBase> writers, out int tokenCount)
        {
            int count = 0;
            int length = 0;
            for (int i = 0; i < def.SuperClassGuidList.Count; ++i)
            {
                var superClass = new RuntimeSuperClassDefinition(def.SuperClassGuidList[i]);

                var writer = new SourceCodeWriter<RuntimeSuperClassDefinition>(Properties.Resources.RDM_SuperClass);
                length += writer.EvaluateTemplateTokens(superClass);
                count  += writer.EvaluatedTokens.Length;
                writers.Add(writer);
            }

            tokenCount = count;

            return length;
        }

        /// <summary>
        /// メンバー定義を評価します.
        /// </summary>
        /// <param name="def">構造体定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <param name="tokenCount">トークン数です.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        private int EvaluateMember(RuntimeDataModelDefinition def, ref List<SourceCodeWriterBase> writers, out int tokenCount)
        {
            int length = 0;
            int count  = 0;
            for (int i = 0; i < def.Properties.Count; ++i)
            {
                // 配列サイズを持つ場合.
                if (!string.IsNullOrEmpty(def.Properties[i].ArraySize))
                {
                    var writer = new SourceCodeWriter<RuntimeDataModelPropertyDefinition>(Properties.Resources.RDM_MemberArray);
                    length += writer.EvaluateTemplateTokens(def.Properties[i]);
                    count  += writer.EvaluatedTokens.Length;
                    writers.Add(writer);
                }
                else
                {
                    var writer = new SourceCodeWriter<RuntimeDataModelPropertyDefinition>(Properties.Resources.RDM_Member);
                    length += writer.EvaluateTemplateTokens(def.Properties[i]);
                    count  += writer.EvaluatedTokens.Length;
                    writers.Add(writer);
                }
            }

            tokenCount = count;

            return length;
        }

        /// <summary>
        /// メンバー定義を評価します.
        /// </summary>
        /// <param name="def">構造体定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <param name="tokenCount">トークン数です.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        private int EvaluateMemberFlip(RuntimeDataModelDefinition def, ref List<SourceCodeWriterBase> writers, out int tokenCount)
        {
            int length = 0;
            int count = 0;
            for (int i = 0; i < def.Properties.Count; ++i)
            {
                RuntimeTypeInfo typeInfo = TypeManager.FindRuntimeType(def.Properties[i].TypeNamespace, def.Properties[i].TypeName);
                if (typeInfo != null)
                {
                    // ポインタであるかどうかチェック.
                    bool isPointer = typeInfo.TypeName.Contains("*") ? true : false;

                    // 1Byteまたはポインタの場合はフリップさせない.
                    if (typeInfo.DataSize == 1 ||
                        typeInfo.FullName == "nw::eftcom::Guid" ||
                        isPointer == true)
                    {
                        continue;
                    }

                    // 構造体ではない場合.
                    if (!typeInfo.IsStructure)
                    {
                        // 配列サイズを持つ場合.
                        if (!string.IsNullOrEmpty(def.Properties[i].ArraySize))
                        {
                            var writer = new SourceCodeWriter<RuntimeDataModelPropertyDefinition>(Properties.Resources.RDM_MemberArrayFlip);
                            length += writer.EvaluateTemplateTokens(def.Properties[i]);
                            count  += writer.EvaluatedTokens.Length;
                            writers.Add(writer);
                        }
                        else
                        {
                            var writer = new SourceCodeWriter<RuntimeDataModelPropertyDefinition>(Properties.Resources.RDM_MemberFlip);
                            length += writer.EvaluateTemplateTokens(def.Properties[i]);
                            count  += writer.EvaluatedTokens.Length;
                            writers.Add(writer);
                        }
                    }
                }
            }

            // メンバーが1つもない場合.
            if (count == 0)
            {
                // バグじゃないことを知らせるために /* DO_NOTHING */ を入れる.
                var writer = new SourceCodeWriterBase();
                writer.EvaluatedTokens = new string[1] { "        /* DO_NOTHING */\r\n" };
                length += writer.EvaluatedTokens[0].Length;
                count  += writer.EvaluatedTokens.Length;

                // ライターに追加.
                writers.Add(writer);
            }

            tokenCount = count;

            return length;
        }

        /// <summary>
        /// メンバー定義を評価します.
        /// </summary>
        /// <param name="def">構造体定義です.</param>
        /// <param name="writers">ライターのリストです.</param>
        /// <param name="tokenCount">トークン数です.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        private int EvaluateFlipMethod(RuntimeDataModelDefinition def, ref List<SourceCodeWriterBase> writers, out int tokenCount)
        {
            int length = 0;
            int count = 0;
            for (int i = 0; i < def.Properties.Count; ++i)
            {
                RuntimeTypeInfo typeInfo = TypeManager.FindRuntimeType(def.Properties[i].TypeNamespace, def.Properties[i].TypeName);
                if (typeInfo != null)
                {
                    // 構造体の場合.
                    if (typeInfo.IsStructure)
                    {
                        // 配列サイズを持つ場合.
                        if (!string.IsNullOrEmpty(def.Properties[i].ArraySize))
                        {
                            var writer = new SourceCodeWriter<RuntimeDataModelPropertyDefinition>(Properties.Resources.RDM_FlipMethodArray);
                            length += writer.EvaluateTemplateTokens(def.Properties[i]);
                            count  += writer.EvaluatedTokens.Length;
                            writers.Add(writer);
                        }
                        else
                        {
                            var writer = new SourceCodeWriter<RuntimeDataModelPropertyDefinition>(Properties.Resources.RDM_FlipMethod);
                            length += writer.EvaluateTemplateTokens(def.Properties[i]);
                            count  += writer.EvaluatedTokens.Length;
                            writers.Add(writer);
                        }
                    }
                }
            }

            tokenCount = count;

            return length;
        }

        #endregion
    }
}
