﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.DataModelMaker.Core.Converters;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Template;

namespace EffectMaker.DataModelMaker.Core.Writers
{
    /// <summary>
    /// バイナリ変換情報ライターです.
    /// </summary>
    public class BinaryConversionInfoWriter : SourceCodeWriterBase
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public BinaryConversionInfoWriter()
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// 定義データからソースコードを生成します.
        /// </summary>
        /// <param name="def">バイナリデータ定義です.</param>
        /// <returns>生成された文字列を返却します.</returns>
        public string Write(BinaryConversionInfoDefinition def)
        {
            // 各ライターのリスト.
            var structWriters = new List<SourceCodeWriterBase>();

            // エントリーリストを生成します.
            var entryListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

            // エントリーリストに追加します.
            entryListMap.Add("Struct", structWriters);

            // クラスライター
            SourceCodeWriter<BinaryConversionInfoDefinition> classWriter = null;
            int length = 0;

            if (!def.IsUserData)
            {
                // ビルドイン用テンプレートを使用.
                classWriter = new SourceCodeWriter<BinaryConversionInfoDefinition>(Properties.Resources.BCI_Main);
                length += classWriter.EvaluateTemplateTokens(def);
            }
            else
            {
                // ユーザーデータ用テンプレートを使用.
                classWriter = new SourceCodeWriter<BinaryConversionInfoDefinition>(Properties.Resources.BCI_UserMain);
                length += classWriter.EvaluateTemplateTokens(def);
            }

            // フィールド.
            length += this.EvaluateStruct(def, ref structWriters);

            // ストリングビルダーを生成.
            StringBuilder builder = new StringBuilder(length);

            // ビルダーに文字列を追加していく.
            for (int i = 0; i < classWriter.EvaluatedTokens.Length; ++i)
            {
                string token = classWriter.EvaluatedTokens[i];
                if (token != null)
                {
                    builder.Append(token);
                }
                else
                {
                    TemplateToken templateToken = classWriter.GetToken(i);
                    if ((templateToken != null) &&
                        (string.IsNullOrEmpty(templateToken.Name) == false) &&
                        (entryListMap.ContainsKey(templateToken.Name) == true))
                    {
                        this.WriteEntryList(builder, entryListMap[templateToken.Name]);
                    }
                }
            }

            if (builder[builder.Length - 1] != '\n')
            {
                builder.AppendLine();
            }

            // 生成した文字列を返却します.
            return builder.ToString();
        }

        /// <summary>
        /// バイナリ変換構造体を評価します.
        /// </summary>
        /// <param name="def">バイナリデータ定義.</param>
        /// <param name="writers">ライターです.</param>
        /// <returns>評価されたテキストの長さを返却します.</returns>
        private int EvaluateStruct(BinaryConversionInfoDefinition def, ref List<SourceCodeWriterBase> writers)
        {
            int length = 0;
            int count = 0;

            // ライターです.
            var fieldWriters = new List<SourceCodeWriterBase>();

            // 連想マップです.
            var entryListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

            // 連想マップに追加します.
            entryListMap.Add("AddField", fieldWriters);

            // 構造体ライター.
            var structWriter = new SourceCodeWriter<BinaryConversionInfoDefinition>(Properties.Resources.BCI_Struct);

            // 評価します.
            length += structWriter.EvaluateTemplateTokens(def);
            count += structWriter.EvaluatedTokens.Length;

            foreach (var field in def.FieldDefinitions)
            {
                if (field.HasExData)
                {
                    // ライターを生成.
                    var writer = new SourceCodeWriter<BinaryConversionFieldInfoDefinition>(Properties.Resources.BCI_AddField_AF);

                    // 評価します.
                    length += writer.EvaluateTemplateTokens(field);
                    count += writer.EvaluatedTokens.Length;

                    fieldWriters.Add(writer);
                }
                else
                {
                    // ライターを生成.
                    var writer = new SourceCodeWriter<BinaryConversionFieldInfoDefinition>(Properties.Resources.BCI_AddField);

                    // 評価します.
                    length += writer.EvaluateTemplateTokens(field);
                    count += writer.EvaluatedTokens.Length;

                    fieldWriters.Add(writer);
                }
            }

            int counter = 0;
            string[] tokens = new string[count];

            // 適切な文字に置き換える.
            for (int i = 0; i < structWriter.EvaluatedTokens.Length; ++i)
            {
                string token = structWriter.EvaluatedTokens[i];
                if (token != null)
                {
                    tokens[counter] = token;
                    counter++;
                }
                else
                {
                    TemplateToken templateToken = structWriter.GetToken(i);
                    if ((templateToken != null)
                        && (string.IsNullOrEmpty(templateToken.Name) == false)
                        && (entryListMap.ContainsKey(templateToken.Name) == true))
                    {
                        foreach (var writer in entryListMap[templateToken.Name])
                        {
                            foreach (string val in writer.EvaluatedTokens)
                            {
                                tokens[counter] = val;
                                counter++;
                            }
                        }
                    }
                }
            }

            // トークンを差し替え.
            structWriter.EvaluatedTokens = tokens;

            // ライターリストに追加.
            writers.Add(structWriter);

            return length;
        }
    }
}
