﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Serialization;

using EffectMaker.DataModelLogic;

using EffectMaker.DataModelMaker.Core.Collections;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;

namespace EffectMaker.DataModelMaker.Core.Definitions
{
    /// <summary>
    /// ランタイムデータモデルのルートを定義するクラスです.
    /// </summary>
    [XmlType("RuntimeDataDefinition")]
    public class RuntimeDataModelRootDefinition : DefinitionBase
    {
        /// <summary>The runtime data model definitions.</summary>
        private DefinitionList<RuntimeDataModelDefinition> dataModelDefs = null;

        /// <summary>The binary data definitions.</summary>
        private DefinitionList<BinaryDataDefinition> binaryDefs = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeDataModelRootDefinition()
        {
            this.dataModelDefs = new DefinitionList<RuntimeDataModelDefinition>(this);
            this.binaryDefs = new DefinitionList<BinaryDataDefinition>(this);
        }

        /// <summary>
        /// データモデルの定義です.
        /// </summary>
        [XmlElement("DataModel")]
        public DefinitionList<RuntimeDataModelDefinition> DataModels
        {
            get { return this.dataModelDefs; }
            set { this.SetChildDefinitionList(ref this.dataModelDefs, value); }
        }

        /// <summary>
        /// バイナリデータの定義です.
        /// </summary>
        [XmlElement("BinaryData")]
        public DefinitionList<BinaryDataDefinition> BinaryDatas
        {
            get { return this.binaryDefs; }
            set { this.SetChildDefinitionList(ref this.binaryDefs, value); }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChildren(this.DataModels);
            this.DisposeChildren(this.BinaryDatas);
            base.Dispose();
        }
    }

    /// <summary>
    /// ランタイムのデータモデル(構造体)を定義するクラスです.
    /// </summary>
    [XmlType("DataModel")]
    public class RuntimeDataModelDefinition : DefinitionBase, INameDescriptionObject
    {
        /// <summary>The properties of the data model.</summary>
        private DefinitionList<RuntimeDataModelPropertyDefinition> properties = null;

        /// <summary>The include files for the data model.</summary>
        private DefinitionList<RuntimeDataModelIncludeDefinition> includes = null;

        /// <summary>The name of the data model.</summary>
        private string name = string.Empty;

        /// <summary>The namespace of the data model.</summary>
        private string nameSpace = string.Empty;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeDataModelDefinition()
        {
            this.properties = new DefinitionList<RuntimeDataModelPropertyDefinition>(this);
            this.includes = new DefinitionList<RuntimeDataModelIncludeDefinition>(this);
            this.SuperClassGuidList = new List<Guid>();
        }

        /// <summary>
        /// 構造体名を設定/取得します.
        /// </summary>
        [XmlAttribute("Name")]
        public string Name
        {
            get
            {
                return this.name;
            }

            set
            {
                // First remove the registered type.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.RemoveRuntimeType(this.nameSpace, this.name);
                }

                this.name = value;

                // Now register the data model with the new name.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.AddRuntimeDataModel(this);
                }
            }
        }

        /// <summary>
        /// 概要を設定/取得します.
        /// </summary>
        [XmlAttribute("Description")]
        public string Description { get; set; }

        /// <summary>
        /// スーパークラスのGuidを設定/取得します.
        /// </summary>
        [XmlElement("SuperClass")]
        public List<Guid> SuperClassGuidList { get; set; }

        /// <summary>
        /// メンバーの定義を設定/取得します.
        /// </summary>
        [XmlElement("Property")]
        public DefinitionList<RuntimeDataModelPropertyDefinition> Properties
        {
            get { return this.properties; }
            set { this.SetChildDefinitionList(ref this.properties, value); }
        }

        /// <summary>
        /// ファイル名を設定/取得します.
        /// </summary>
        [XmlAttribute("FileName")]
        public string FileName { get; set; }

        /// <summary>
        /// 名前空間を設定/取得します.
        /// </summary>
        [XmlElement("Namespace")]
        public string Namespace
        {
            get
            {
                return this.nameSpace;
            }

            set
            {
                // First remove the registered type.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.RemoveRuntimeType(this.nameSpace, this.name);
                }

                this.nameSpace = value;

                // Now register the data model with the new name.
                if (string.IsNullOrEmpty(this.name) == false)
                {
                    TypeManager.AddRuntimeDataModel(this);
                }
            }
        }

        /// <summary>
        /// インクルード文を設定/取得します.
        /// </summary>
        [XmlElement("Include")]
        public DefinitionList<RuntimeDataModelIncludeDefinition> Includes
        {
            get { return this.includes; }
            set { this.SetChildDefinitionList(ref this.includes, value); }
        }

        /// <summary>
        /// 変数定義とコメントの間のスペース.
        /// </summary>
        [XmlIgnore]
        public string Space1
        {
            get
            {
                // コメントの内容が空っぽだったら、スペースも入れない.
                if (this.Description == string.Empty)
                {
                    return string.Empty;
                }

                return "      ";
            }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            TypeManager.RemoveRuntimeType(this.nameSpace, this.name);

            this.DisposeChildren(this.Includes);
            this.DisposeChildren(this.Properties);
            base.Dispose();
        }
    }

    /// <summary>
    /// インクルード文を定義するクラスです.
    /// </summary>
    [XmlType("Include")]
    public class RuntimeDataModelIncludeDefinition : DefinitionBase
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeDataModelIncludeDefinition()
        {
            this.File = string.Empty;
            this.Header = string.Empty;
            this.Footer = string.Empty;
        }

        /// <summary>
        /// ファイル名を設定/取得します.
        /// </summary>
        [XmlAttribute("File")]
        public string File { get; set; }

        /// <summary>
        /// インクルード文の前に付け足すヘッダです.
        /// </summary>
        [XmlIgnore]
        public string Header { get; set; }

        /// <summary>
        /// インクルード文の後に付け足すフッタです.
        /// </summary>
        [XmlIgnore]
        public string Footer { get; set; }
    }

    [XmlType("Namespace")]
    public class RuntimeDataModelNamespaceDefinition : DefinitionBase
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeDataModelNamespaceDefinition()
        {
            this.Name = string.Empty;
            this.Header = string.Empty;
            this.Footer = string.Empty;
        }

        [XmlIgnore]
        public string Name { get; set; }

        /// <summary>
        /// 名前空間の前に付け足すヘッダです.
        /// </summary>
        [XmlIgnore]
        public string Header { get; set; }

        /// <summary>
        /// 名前空間の後に付け足すフッタです.
        /// </summary>
        [XmlIgnore]
        public string Footer { get; set; }
    }

    /// <summary>
    /// メンバーを定義するクラスです.
    /// </summary>
    [XmlType("Property")]
    public class RuntimeDataModelPropertyDefinition : DefinitionBase, INameDescriptionObject
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeDataModelPropertyDefinition()
        {
            this.Name          = string.Empty;
            this.TypeName      = string.Empty;
            this.TypeNamespace = string.Empty;
            this.Description   = string.Empty;
            this.ArraySize     = string.Empty;
            this.Space1        = string.Empty;
            this.Space2        = string.Empty;
        }

        /// <summary>
        /// 型名を設定/取得します.
        /// </summary>
        [XmlAttribute("Type")]
        public string TypeName { get; set; }

        /// <summary>
        /// メンバーが定義される名前空間です.
        /// </summary>
        [XmlAttribute("Namespace")]
        public string TypeNamespace { get; set; }

        /// <summary>
        /// メンバー名を設定/取得します.
        /// </summary>
        [XmlAttribute("Name")]
        public string Name { get; set; }

        /// <summary>
        /// 概要を設定/取得します.
        /// </summary>
        [XmlAttribute("Description")]
        public string Description { get; set; }

        /// <summary>
        /// 配列サイズを設定/取得します.
        /// </summary>
        [XmlAttribute("ArraySize")]
        public string ArraySize { get; set; }

        /// <summary>
        /// 名前空間を含むフルタイプ名を取得します.
        /// </summary>
        [XmlIgnore]
        public string FullType
        {
            get
            {
                if (string.IsNullOrEmpty(this.TypeNamespace) == true)
                {
                    return this.TypeName;
                }

                return this.TypeNamespace + "::" + this.TypeName;
            }
        }

        /// <summary>
        /// 型名と変数名の間のスペース.
        /// </summary>
        [XmlIgnore]
        public string Space1 { get; set; }

        /// <summary>
        /// 変数名とコメントの間のスペース.
        /// </summary>
        [XmlIgnore]
        public string Space2 { get; set; }

        /// <summary>
        /// 変数定義とコメントの間のスペース.
        /// </summary>
        [XmlIgnore]
        public string Space3
        {
            get
            {
                // コメントの内容が空っぽだったら、スペースも入れない.
                if (this.Description == string.Empty)
                {
                    return string.Empty;
                }

                return " ";
            }
        }
    }

    /// <summary>
    /// バイナリデータを定義するクラスです.
    /// </summary>
    [XmlType("BinaryData")]
    public class BinaryDataDefinition : DefinitionBase
    {
        /// <summary>The binary fields.</summary>
        private DefinitionList<BinaryFieldDefinition> fields = null;

        /// <summary>The binary conversion source data model instance.</summary>
        private SourceDataModelInstanceDefinition srcDataModelInstance = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public BinaryDataDefinition()
        {
            this.fields = new DefinitionList<BinaryFieldDefinition>(this);
            this.FileName = string.Empty;
            this.HasBinaryHeader = true;
            this.IsUserData = false;
        }

        /// <summary>
        /// 名前を設定/取得します.
        /// </summary>
        [XmlAttribute("Name")]
        public string Name { get; set; }

        /// <summary>
        /// ファイル名を設定/取得します.
        /// </summary>
        [XmlAttribute("FileName")]
        public string FileName { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether this binary data generates binary header.
        /// </summary>
        [XmlAttribute("HasBinaryHeader")]
        public bool HasBinaryHeader { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether this binary data is an user data.
        /// </summary>
        [XmlAttribute("IsUserData")]
        public bool IsUserData { get; set; }

        /// <summary>
        /// The binary conversion source data model instance.
        /// </summary>
        public SourceDataModelInstanceDefinition SourceDataModelInstance
        {
            get { return this.srcDataModelInstance; }
            set { this.SetChildDefinition(ref this.srcDataModelInstance, value); }
        }

        /// <summary>
        /// バイナリデータのフィルドです。
        /// </summary>
        [XmlElement("BinaryField")]
        public DefinitionList<BinaryFieldDefinition> Fields
        {
            get { return this.fields; }
            set { this.SetChildDefinitionList(ref this.fields, value); }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChild(this.SourceDataModelInstance);
            this.DisposeChildren(this.Fields);
            base.Dispose();
        }
    }

    /// <summary>
    /// バイナリフィルドグループを定義するクラスです.
    /// </summary>
    [XmlType("BinaryFieldGroup")]
    public class BinaryFieldGroupDefinition : DefinitionBase
    {
        /// <summary>The binary fields.</summary>
        private DefinitionList<BinaryFieldDefinition> fields = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public BinaryFieldGroupDefinition()
        {
            this.fields = new DefinitionList<BinaryFieldDefinition>(this);
            this.TargetGuid = Guid.Empty;
        }

        /// <summary>
        /// Get the flag indicating whether any of the parents are collapsed.
        /// This property also returns true if this data model instance itself is
        /// collapsed.
        /// </summary>
        [XmlIgnore]
        public bool IsAnyParentCollapsed
        {
            get
            {
                DefinitionBase def = this;
                while (def != null)
                {
                    var group = def.Parent as BinaryFieldGroupDefinition;
                    if (group != null && group.IsCollapsed == true)
                    {
                        return true;
                    }

                    def = def.Parent;
                }

                return false;
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether this field group is collapsed on UI.
        /// </summary>
        public bool IsCollapsed { get; set; }

        /// <summary>
        /// グループターゲットのGuidです。(ランタイムデータモデルかバイナリデータのGuid)
        /// </summary>
        [XmlAttribute("TargetGuid")]
        public Guid TargetGuid { get; set; }

        /// <summary>
        /// バイナリデータのフィルドです。
        /// </summary>
        [XmlElement("BinaryField")]
        public DefinitionList<BinaryFieldDefinition> Fields
        {
            get { return this.fields; }
            set { this.SetChildDefinitionList(ref this.fields, value); }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChildren(this.Fields);
            base.Dispose();
        }
    }

    /// <summary>
    /// Class for storing and serializing/deserializing the converter parameters.
    /// </summary>
    [Serializable]
    public class ConverterParameterDefinition
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public ConverterParameterDefinition()
        {
            this.Name = string.Empty;
            this.Value = string.Empty;
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="name">The name of the parameter.</param>
        /// <param name="value">The value of the parameter.</param>
        public ConverterParameterDefinition(string name, string value)
        {
            this.Name = name;
            this.Value = value;
        }

        /// <summary>
        /// Get or set the name of the parameter.
        /// </summary>
        [XmlAttribute("Name")]
        public string Name { get; set; }

        /// <summary>
        /// Get or set the value of the parameter.
        /// </summary>
        [XmlAttribute("Value")]
        public string Value { get; set; }
    }

    /// <summary>
    /// バイナリフィールドを定義するクラスです.
    /// </summary>
    [XmlType("BinaryField")]
    public class BinaryFieldDefinition : DefinitionBase
    {
        /// <summary>The binary field group definition.</summary>
        private BinaryFieldGroupDefinition groupDef = null;

        /// <summary>
        /// フィルドのGuidです。
        /// </summary>
        public BinaryFieldDefinition()
        {
            this.OutputFieldGuid = Guid.Empty;
            this.FieldType = BinaryFieldTypes.Normal;
            this.FieldSize = 0;
            this.SendModificationType = SendModificationTypes.ModifiedDataOnly;
            this.Converter = "DefaultConverter";
            this.ConverterParameters = new List<ConverterParameterDefinition>();
            this.InputPropertyGuidList = new List<Guid>();
            this.Tag = string.Empty;
        }

        /// <summary>
        /// データモデルプロパーティのGuidです。
        /// </summary>
        [XmlElement("OutputFieldGuid")]
        public Guid OutputFieldGuid { get; set; }

        /// <summary>
        /// Get or set the binary field type.
        /// </summary>
        [XmlAttribute("FieldType")]
        public BinaryFieldTypes FieldType { get; set; }

        /// <summary>
        /// Get or set the binary field size.
        /// </summary>
        [XmlAttribute("FieldSize")]
        public int FieldSize { get; set; }

        /// <summary>
        /// Get or set the type of message to send when the data is modified.
        /// </summary>
        [XmlElement("SendModificationType")]
        public SendModificationTypes SendModificationType { get; set; }

        /// <summary>
        /// コンバーター名を設定/取得します.
        /// </summary>
        [XmlElement("Converter")]
        public string Converter { get; set; }

        /// <summary>
        /// Get or set the parameters for the converter.
        /// </summary>
        [XmlElement("ConverterParam")]
        public List<ConverterParameterDefinition> ConverterParameters { get; set; }

        /// <summary>
        /// 入力プロパティのGUIDを設定/取得します.
        /// </summary>
        [XmlElement("InputPropertyGuid")]
        public List<Guid> InputPropertyGuidList { get; set; }

        /// <summary>
        /// フィルドグループです。
        /// </summary>
        [XmlElement("BinaryFieldGroup")]
        public BinaryFieldGroupDefinition GroupDefinition
        {
            get { return this.groupDef; }
            set { this.SetChildDefinition(ref this.groupDef, value); }
        }

        /// <summary>
        /// バイナリタグです.
        /// </summary>
        [XmlElement("Tag")]
        public string Tag { get; set; }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChild(this.GroupDefinition);
            base.Dispose();
        }
    }

    /// <summary>
    /// The definition of the binary conversion source data model instance.
    /// </summary>
    [XmlType("SourceDataModelInstance")]
    public class SourceDataModelInstanceDefinition : DefinitionBase
    {
        /// <summary>The source data model property instances.</summary>
        private DefinitionList<SourcePropertyInstanceDefinition> properties = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public SourceDataModelInstanceDefinition()
        {
            this.properties = new DefinitionList<SourcePropertyInstanceDefinition>(this);
            this.DataModelDefinitionGuid = Guid.Empty;
            this.IsCollapsed = false;
        }

        /// <summary>
        /// Get the flag indicating whether any of the parents are collapsed.
        /// This property also returns true if this data model instance itself is
        /// collapsed.
        /// </summary>
        [XmlIgnore]
        public bool IsAnyParentCollapsed
        {
            get
            {
                DefinitionBase def = this;
                while (def != null)
                {
                    var instance = def.Parent as SourceDataModelInstanceDefinition;
                    if (instance != null && instance.IsCollapsed == true)
                    {
                        return true;
                    }

                    def = def.Parent;
                }

                return false;
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether this data model instance is collapsed on UI.
        /// </summary>
        public bool IsCollapsed { get; set; }

        /// <summary>
        /// The Guid of the data model definition.
        /// </summary>
        [XmlElement("DataModelDefinitionGuid")]
        public Guid DataModelDefinitionGuid { get; set; }

        /// <summary>
        /// The property instances.
        /// </summary>
        [XmlElement("SourcePropertyInstance")]
        public DefinitionList<SourcePropertyInstanceDefinition> Properties
        {
            get { return this.properties; }
            set { this.SetChildDefinitionList(ref this.properties, value); }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChildren(this.Properties);
            base.Dispose();
        }
    }

    /// <summary>
    /// The definition of the binary conversion source data model instance.
    /// </summary>
    [XmlType("SourcePropertyInstance")]
    public class SourcePropertyInstanceDefinition : DefinitionBase
    {
        /// <summary>The underlying data model instance of the property.</summary>
        private SourceDataModelInstanceDefinition underlyingDataModel = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public SourcePropertyInstanceDefinition()
        {
            this.PropertyDefinitionGuid = Guid.Empty;
        }

        /// <summary>
        /// The Guid of the data model property definition.
        /// </summary>
        [XmlElement("PropertyDefinitionGuid")]
        public Guid PropertyDefinitionGuid { get; set; }

        /// <summary>
        /// The underlying data model instance.
        /// </summary>
        [XmlElement("SourceDataModelInstance")]
        public SourceDataModelInstanceDefinition DataModelInstance
        {
            get { return this.underlyingDataModel; }
            set { this.SetChildDefinition(ref this.underlyingDataModel, value); }
        }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public override void Dispose()
        {
            this.DisposeChild(this.DataModelInstance);
            base.Dispose();
        }
    }

    /// <summary>
    /// ソースコード生成のための一時的に利用する定義クラスです.
    /// Temporary class for generation of source code.
    /// </summary>
    public class RuntimeSuperClassDefinition
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="guid">Guidです.</param>
        public RuntimeSuperClassDefinition(Guid guid)
        {
            this.AccessModifier = string.Empty;
            this.Name = string.Empty;

            // ワークスペースマネージャーから定義を引っ張る.
            var def = WorkspaceManager.FindDefinition(guid) as RuntimeDataModelDefinition;
            if (def != null)
            {
                // 見つかったら設定.
                this.AccessModifier = "public";
                this.Name = string.Copy(def.Name);
            }
        }

        /// <summary>
        /// アクセス修飾子です.
        /// </summary>
        public string AccessModifier { get; set; }

        /// <summary>
        /// 名前を設定/取得します.
        /// </summary>
        public string Name { get; set; }
    }

    /// <summary>
    /// ソースコード生成のために一時利用する定義クラスです.
    /// Temporary class for generation of source code.
    /// </summary>
    public class RuntimeFileDefinition
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RuntimeFileDefinition()
        {
            this.FileName = string.Empty;
            this.Namespace = string.Empty;
            this.Define = string.Empty;
            this.Includes = new List<RuntimeDataModelIncludeDefinition>();
            this.Structures = new List<RuntimeDataModelDefinition>();
            this.Namespaces = new List<RuntimeDataModelNamespaceDefinition>();
        }

        /// <summary>
        /// ファイル名です.
        /// </summary>
        public string FileName { get; set; }

        /// <summary>
        /// 名前空間です.
        /// </summary>
        public string Namespace { get; set; }

        /// <summary>
        /// デファイン名です.
        /// </summary>
        public string Define { get; set; }

        /// <summary>
        /// インクルードファイルです.
        /// </summary>
        public List<RuntimeDataModelIncludeDefinition> Includes { get; set; }

        /// <summary>
        /// 構造体です.
        /// </summary>
        public List<RuntimeDataModelDefinition> Structures { get; set; }

        /// <summary>
        /// 名前空間です.
        /// </summary>
        public List<RuntimeDataModelNamespaceDefinition> Namespaces { get; set; }
    }
}
