﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.DataModelLogic.BinaryConverters;

using EffectMaker.Foundation.Attributes;

namespace EffectMaker.DataModelMaker.Core.Converters
{
    /// <summary>
    /// Manager class that manages available converters.
    /// </summary>
    public static class ConverterManager
    {
        /// <summary>The dictionary of the converters, with the full name as the key.</summary>
        private static Dictionary<string, ConverterInfo> converters =
            new Dictionary<string, ConverterInfo>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static ConverterManager()
        {
            CollectConverters("EffectMaker.DataModelLogic");
        }

        /// <summary>
        /// Enumerate all the converters.
        /// </summary>
        public static IEnumerable<ConverterInfo> Converters
        {
            get { return converters.Select(item => item.Value); }
        }

        /// <summary>
        /// Get the default converter.
        /// </summary>
        public static ConverterInfo DefaultConverter { get; private set; }

        /// <summary>
        /// Find converter with it's full name.
        /// </summary>
        /// <param name="fullName">The name of the converter, including the namespace.</param>
        /// <returns>The converter info.</returns>
        public static ConverterInfo FindConverter(string fullName)
        {
            ConverterInfo info;
            if (converters.TryGetValue(fullName, out info) == false)
            {
                return DefaultConverter;
            }

            return info;
        }

        /// <summary>
        /// Collect all the available converters.
        /// </summary>
        /// <param name="sourceAssemblyName">The name of the assembly to search for converters.</param>
        private static void CollectConverters(string sourceAssemblyName)
        {
            Assembly refAsm = null;

            // Load the specified assembly from the referenced assemblies.
            AssemblyName[] refAsmNames =
                Assembly.GetExecutingAssembly().GetReferencedAssemblies();
            foreach (AssemblyName refAsmName in refAsmNames)
            {
                if (refAsmName.Name == sourceAssemblyName)
                {
                    refAsm = Assembly.Load(refAsmName);
                    break;
                }
            }

            if (refAsm == null)
            {
                return;
            }

            // First clear saved converters.
            converters.Clear();
            DefaultConverter = null;

            // Get the type of the base converter.
            Type baseConverterType =
                typeof(EffectMaker.DataModelLogic.BinaryConverters.ConverterBase);

            // Get the types of converters in the specified assembly.
            IEnumerable<Type> converterTypes = null;
            try
            {
                converterTypes =
                    refAsm.GetTypes().Where(tp => tp.IsSubclassOf(baseConverterType));
            }
            catch (ReflectionTypeLoadException ex)
            {
                converterTypes = ex.Types.Where(tp => tp != null && tp.IsSubclassOf(baseConverterType));
            }

            if (converterTypes == null)
            {
                return;
            }

            // Add all the converters to our dictionary.
            foreach (Type converterType in converterTypes)
            {
                var converterInstance = Activator.CreateInstance(converterType) as ConverterBase;
                if (converterInstance == null)
                {
                    continue;
                }

                ConverterInfo info = new ConverterInfo(converterInstance);

                Attribute attr = converterType.GetCustomAttribute(typeof(DefaultAttribute));
                if (attr != null && DefaultConverter == null)
                {
                    DefaultConverter = info;
                }

                converters.Add(converterType.FullName, info);
            }
        }
    }
}
