﻿// ========================================================================
// <copyright file="ExpressionUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Linq.Expressions;

    /// <summary>
    /// 式用のユーティリティです。
    /// </summary>
    public static class ExpressionUtility
    {
        /// <summary>
        /// 指定した２つのプロパティ式を連結します。
        /// </summary>
        /// <typeparam name="TTarget">
        /// 対象となるオブジェクト（式のパラメータ）の型を指定します。
        /// </typeparam>
        /// <typeparam name="TResult1">１つ目の式の戻り値の型を指定します。</typeparam>
        /// <typeparam name="TResult2">２つ目の式の戻り値の型を指定します。</typeparam>
        /// <param name="expression1">１つ目の式を指定します。</param>
        /// <param name="expression2">２つ目の式を指定します。</param>
        /// <returns>２つのプロパティ式を連結した結果を返します。</returns>
        public static Expression<Func<TTarget, TResult2>> Concat<TTarget, TResult1, TResult2>(
            this Expression<Func<TTarget, TResult1>> expression1,
            Expression<Func<TResult1, TResult2>> expression2)
        {
            Ensure.Argument.NotNull(expression1);
            Ensure.Argument.NotNull(expression2);

            var memberExpression = expression2.Body.GetMemberExpression();
            Ensure.Operation.ObjectNotNull(memberExpression);

            Expression newBody = Concat(expression1.Body, expression2.Body);

            if (memberExpression.Type != typeof(TResult2))
            {
                newBody = Expression.TypeAs(newBody, typeof(TResult2));
            }

            return Expression.Lambda<Func<TTarget, TResult2>>(
                newBody,
                expression1.Parameters);
        }

        /// <summary>
        /// 指定した２つのプロパティ式を連結します。
        /// </summary>
        /// <typeparam name="TTarget">
        /// 対象となるオブジェクト（式のパラメータ）の型を指定します。
        /// </typeparam>
        /// <typeparam name="TResult1">１つ目の式の戻り値の型を指定します。</typeparam>
        /// <typeparam name="TResult2">２つ目の式の戻り値の型を指定します。</typeparam>
        /// <param name="expression1">１つ目の式を指定します。</param>
        /// <param name="expression2">２つ目の式を指定します。</param>
        /// <returns>２つのプロパティ式を連結した結果を返します。</returns>
        public static Expression<Func<TTarget, TResult2>> Concat<TTarget, TResult1, TResult2>(
            this Expression<Func<TTarget, TResult1>> expression1,
            LambdaExpression expression2)
        {
            Ensure.Argument.NotNull(expression1);
            Ensure.Argument.NotNull(expression2);

            var memberExpression = expression2.Body.GetMemberExpression();
            Ensure.Operation.ObjectNotNull(memberExpression);

            Expression newBody = Concat(expression1.Body, expression2.Body);

            if (memberExpression.Type != typeof(TResult2))
            {
                newBody = Expression.TypeAs(newBody, typeof(TResult2));
            }

            return Expression.Lambda<Func<TTarget, TResult2>>(
                newBody,
                expression1.Parameters);
        }

        /// <summary>
        /// 指定した２つのプロパティ式を連結します。
        /// </summary>
        /// <param name="expression1">１つ目の式を指定します。</param>
        /// <param name="expression2">２つ目の式を指定します。</param>
        /// <returns>２つのプロパティ式を連結した結果を返します。</returns>
        public static MemberExpression Concat(
            this Expression expression1,
            Expression expression2)
        {
            Ensure.Argument.NotNull(expression1);
            Ensure.Argument.NotNull(expression2);

            var memberExpression = expression2.GetMemberExpression();
            Ensure.Operation.ObjectNotNull(memberExpression);

            if (memberExpression.Expression is ParameterExpression)
            {
                return Expression.MakeMemberAccess(expression1, memberExpression.Member);
            }

            return Expression.MakeMemberAccess(
                Concat(expression1, memberExpression.Expression as MemberExpression),
                memberExpression.Member);
        }

        /// <summary>
        /// メンバーを示す式を取得します。
        /// </summary>
        /// <param name="expression">対象の式を指定します。</param>
        /// <returns>メンバーを示す式を返します。</returns>
        public static MemberExpression GetMemberExpression(this Expression expression)
        {
            Assertion.Argument.NotNull(expression);

            var memberExpression = expression as MemberExpression;

            if (memberExpression != null)
            {
                return memberExpression;
            }

            var unaryExpression = expression as UnaryExpression;

            if (unaryExpression != null)
            {
                return unaryExpression.Operand as MemberExpression;
            }

            return null;
        }
    }
}
