﻿// ========================================================================
// <copyright file="OrientedBoundingBox.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// OBB(境界矩形)クラスです。
    /// </summary>
    public class OrientedBoundingBox : BoundingVolume
    {
        /// <summary>
        /// キューブのデフォルトサイズです。
        /// </summary>
        public const float DefaultSize = 1.0f;

        /// <summary>
        /// 頂点の数です。
        /// </summary>
        public static readonly int VertexCount = 8;

        private Vector3 centerPosition = new Vector3();
        private Matrix33 orientationMatrix = new Matrix33();
        private Vector3 size = new Vector3(DefaultSize, DefaultSize, DefaultSize);

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OrientedBoundingBox()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のデータです。</param>
        public OrientedBoundingBox(OrientedBoundingBox source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 中心位置を取得します。
        /// </summary>
        public Vector3 CenterPosition
        {
            get
            {
                return this.centerPosition;
            }
        }

        /// <summary>
        /// 姿勢行列を取得します。
        /// <remarks>
        /// TODO:CreativeStudio内で生成されたデータについて、
        /// BoundingVolumeを更新するメソッドを準備する必要があります。
        /// </remarks>
        /// </summary>
        public Matrix33 OrientationMatrix
        {
            get { return this.orientationMatrix; }
        }

        /// <summary>
        /// 姿勢行列座標空間におけるサイズを取得します。
        /// </summary>
        public Vector3 Size
        {
            get { return this.size; }
        }

        /// <summary>
        /// 各頂点位置を取得します。
        /// </summary>
        public Vector3[] VertexPositions
        {
            get
            {
                float half_x = this.Size.X * 0.5f;
                float half_y = this.Size.Y * 0.5f;
                float half_z = this.Size.Z * 0.5f;

                Vector3[] posArray = new[]
                {
                    new Vector3(-half_x, -half_y, -half_z),
                    new Vector3(-half_x, -half_y, +half_z),
                    new Vector3(-half_x, +half_y, -half_z),
                    new Vector3(-half_x, +half_y, +half_z),
                    new Vector3(+half_x, -half_y, -half_z),
                    new Vector3(+half_x, -half_y, +half_z),
                    new Vector3(+half_x, +half_y, -half_z),
                    new Vector3(+half_x, +half_y, +half_z),
                };

                for (int i = 0; i != VertexCount; ++i)
                {
                    posArray[i] = (this.OrientationMatrix * posArray[i]) + this.CenterPosition;
                }

                return posArray;
            }
        }

        /// <summary>
        /// 主成分分析によりOBB を計算します。
        /// </summary>
        /// <remarks>C3T_Common.cpp から移植</remarks>
        /// <param name="poss">点列です。</param>
        public void CalculateByPCA(IList<Vector3> poss)
        {
            // 座標の数をチェックします。
            if (poss.Count == 0)
            {
                // 初期化します。
                this.centerPosition = new Vector3();
                this.orientationMatrix = new Matrix33();
                this.size = new Vector3(DefaultSize, DefaultSize, DefaultSize);
                return;
            }

            // 平均位置を計算します。
            Vector3 meanPosition = new Vector3();
            foreach (Vector3 pos in poss)
            {
                meanPosition += pos;
            }

            meanPosition /= poss.Count;

            // 共分散行列を作成します。
            float[,] covarianceMtx = new float[3, 3];
            foreach (Vector3 pos in poss)
            {
                for (int i = 0; i < 3; i++)
                {
                    for (int j = i; j < 3; j++)
                    {
                        covarianceMtx[i, j] +=
                            (pos[i] - meanPosition[i]) * (pos[j] - meanPosition[j]);
                    }
                }
            }

            for (int i = 0; i < 3; i++)
            {
                for (int j = i; j < 3; j++)
                {
                    covarianceMtx[i, j] /= poss.Count;
                    if (i != j)
                    {
                        covarianceMtx[j, i] = covarianceMtx[i, j];
                    }
                }
            }

            // jacobi 法で固有値 & 固有ベクトルを算出します。
            float[,] eigenVectors;
            float[] eigenValues;
            bool solved = JacobiCalc(covarianceMtx, out eigenVectors, out eigenValues);

            if (solved)
            {
                // 固有値の降順でソートします。
                List<KeyValuePair<float, int>> order = new List<KeyValuePair<float, int>>();
                for (int i = 0; i < 3; i++)
                {
                    order.Add(new KeyValuePair<float, int>(eigenValues[i], i));
                }

                order.Sort((x, y) => y.Key.CompareTo(x.Key));
                for (int i = 0; i < 3; i++)
                {
                    for (int j = 0; j < 3; j++)
                    {
                        this.orientationMatrix[i, j] = eigenVectors[j, order[i].Value];
                    }
                }
            }
            else
            {
                // 解がなければ AABB にします。
                this.orientationMatrix.SetIdentity();
            }

            // 境界ボックスを算出します。
            float[] min = new float[3];
            float[] max = new float[3];

            // 固有ベクトルとの内積
            Vector3 innerProducts = this.orientationMatrix * poss[0];
            for (int i = 0; i < 3; i++)
            {
                max[i] = min[i] = innerProducts[i];
            }

            for (int i = 1; i < poss.Count; i++)
            {
                innerProducts = this.orientationMatrix * poss[i];
                for (int j = 0; j < 3; j++)
                {
                    max[j] = Math.Max(innerProducts[j], max[j]);
                    min[j] = Math.Min(innerProducts[j], min[j]);
                }
            }

            this.centerPosition = new Vector3();
            for (int i = 0; i < 3; i++)
            {
                // 辺の長さ
                this.size[i] = max[i] - min[i];

                // 中心の計算
                for (int j = 0; j < 3; j++)
                {
                    this.centerPosition[j] += ((max[i] + min[i]) / 2) * this.orientationMatrix[i, j];
                }
            }

            // ゼロへの丸めこみを行います。
            for (int i = 0; i < 3; i++)
            {
                this.centerPosition[i] = FloatUtility.RoundToZero(this.centerPosition[i]);
                this.size[i] = FloatUtility.RoundToZero(this.size[i]);
                for (int j = 0; j < 3; j++)
                {
                    this.orientationMatrix[i, j] =
                        FloatUtility.RoundToZero(this.orientationMatrix[i, j]);
                }
            }
        }

        /// <summary>
        /// OBB の全領域を含む AABB を生成します。
        /// </summary>
        /// <returns>OBB の全領域を含む AABB です。</returns>
        public AxisAlignedBoundingBox MakeAABB()
        {
            Vector3 max = new Vector3();
            Vector3 min = new Vector3();

            int index = 0;
            foreach (Vector3 veretexPosition in this.VertexPositions)
            {
                if (index == 0)
                {
                    max.Set(veretexPosition);
                    min.Set(veretexPosition);
                }
                else
                {
                    if (veretexPosition.X > max.X)
                    {
                        max.X = veretexPosition.X;
                    }

                    if (veretexPosition.Y > max.Y)
                    {
                        max.Y = veretexPosition.Y;
                    }

                    if (veretexPosition.Z > max.Z)
                    {
                        max.Z = veretexPosition.Z;
                    }

                    if (veretexPosition.X < min.X)
                    {
                        min.X = veretexPosition.X;
                    }

                    if (veretexPosition.Y < min.Y)
                    {
                        min.Y = veretexPosition.Y;
                    }

                    if (veretexPosition.Z < min.Z)
                    {
                        min.Z = veretexPosition.Z;
                    }
                }

                ++index;
            }

            AxisAlignedBoundingBox aabb = new AxisAlignedBoundingBox();
            aabb.CenterPosition = (max + min) / 2;
            aabb.Size = max - min;
            return aabb;
        }

        /// <summary>
        /// OBB の全領域を含む バウンディングスフィア を生成します。
        /// </summary>
        /// <returns> OBB の全領域を含む バウンディングスフィアです。 </returns>
        public BoundingSphere MakeBoundingSphere()
        {
            BoundingSphere boundingSphere = new BoundingSphere();

            boundingSphere.Set(this.VertexPositions);

            return boundingSphere;
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>このインスタンスのコピーである新しいオブジェクトです。</returns>
        public override object Clone()
        {
            return new OrientedBoundingBox(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public override void Set(object source)
        {
            this.Set(source as OrientedBoundingBox);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        protected void Set(OrientedBoundingBox source)
        {
            Ensure.Argument.NotNull(source);

            this.CenterPosition.Set(source.CenterPosition);
            this.OrientationMatrix.Set(source.OrientationMatrix);
            this.Size.Set(source.Size);
        }

        /// <summary>
        /// Jacobi 法により固有値と固有ベクトルを求めます。
        /// </summary>
        /// <remarks>C3T_Common.cpp より移植、NUMERICAL RECIPES in C より</remarks>
        /// <param name="a">３次行列です。</param>
        /// <param name="v">固有ベクトルです。</param>
        /// <param name="d">固有値です。</param>
        /// <returns>解が求まればtrue を返します。</returns>
        private static bool JacobiCalc(float[,] a, out float[,] v, out float[] d)
        {
            int n = 3;
            int i, j, iq, ip;
            float tresh, theta, tau, t, sm, s, h, g, c;
            float[] b = new float[3];
            float[] z = new float[3];
            v = new float[3, 3];
            d = new float[3];

            for (ip = 0; ip < n; ip++)
            {
                for (iq = 0; iq < n; iq++)
                {
                    v[ip, iq] = 0.0f;
                }

                v[ip, ip] = 1.0f;
            }

            for (ip = 0; ip < n; ip++)
            {
                b[ip] = d[ip] = a[ip, ip];
                z[ip] = 0.0f;
            }

            for (i = 0; i < 50; i++)
            {
                sm = 0.0f;
                for (ip = 0; ip < n - 1; ip++)
                {
                    for (iq = ip + 1; iq < n; iq++)
                    {
                        sm += Math.Abs(a[ip, iq]);
                    }
                }

                if (sm == 0.0f)
                {
                    return true;
                }

                if (i < 3)
                {
                    tresh = 0.2f * sm / (n * n);
                }
                else
                {
                    tresh = 0.0f;
                }

                for (ip = 0; ip < n - 1; ip++)
                {
                    for (iq = ip + 1; iq < n; iq++)
                    {
                        g = 100.0f * Math.Abs(a[ip, iq]);
                        if (i > 3 && (Math.Abs(d[ip]) + g) == Math.Abs(d[ip])
                            && (Math.Abs(d[iq]) + g) == Math.Abs(d[iq]))
                        {
                            a[ip, iq] = 0.0f;
                        }
                        else if (Math.Abs(a[ip, iq]) > tresh)
                        {
                            h = d[iq] - d[ip];
                            if ((Math.Abs(h) + g) == Math.Abs(h))
                            {
                                t = a[ip, iq] / h;
                            }
                            else
                            {
                                theta = 0.5f * h / a[ip, iq];
                                t = 1 / (Math.Abs(theta) + (float)Math.Sqrt(1.0f + (theta * theta)));
                                if (theta < 0.0f)
                                {
                                    t = -t;
                                }
                            }

                            c = 1.0f / (float)Math.Sqrt(1 + (t * t));
                            s = t * c;
                            tau = s / (1.0f + c);
                            h = t * a[ip, iq];
                            z[ip] -= h;
                            z[iq] += h;
                            d[ip] -= h;
                            d[iq] += h;
                            a[ip, iq] = 0.0f;

                            for (j = 0; j < ip; j++)
                            {
                                JacobiRotate(a, s, tau, j, ip, j, iq);
                            }

                            for (j = ip + 1; j < iq; j++)
                            {
                                JacobiRotate(a, s, tau, ip, j, j, iq);
                            }

                            for (j = iq + 1; j < n; j++)
                            {
                                JacobiRotate(a, s, tau, ip, j, iq, j);
                            }

                            for (j = 0; j < n; j++)
                            {
                                JacobiRotate(v, s, tau, j, ip, j, iq);
                            }
                        }
                    }
                }

                for (ip = 0; ip < n; ip++)
                {
                    b[ip] += z[ip];
                    d[ip] = b[ip];
                    z[ip] = 0.0f;
                }
            }

            return false;
        }

        private static void JacobiRotate(float[,] a, float s, float tau, int i, int j, int k, int l)
        {
            float h, g;
            g = a[i, j];
            h = a[k, l];
            a[i, j] = g - (s * (h + (g * tau)));
            a[k, l] = h + (s * (g - (h * tau)));
        }
    }
}
