﻿// ========================================================================
// <copyright file="DefaultConverters.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Conversion
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Linq;
    using System.Reflection;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// 既定のコンバータです。
    /// </summary>
    public class DefaultConverters
    {
        /// <summary>
        /// プリミティブ型を変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        public static object ConvertPrimitive(object src, Type dstType, Converter converter)
        {
            object dstModel = src;
            return dstModel;
        }

        /// <summary>
        /// リストを変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        public static object ConvertList(object src, Type dstType, Converter converter)
        {
            Ensure.Argument.NotNull(src);

            IEnumerable srcList = src as IEnumerable;

            // リスト要素の型を得ます。
            Type serializableType = ConverterUtil.GetIListElementType(dstType);
            Ensure.Operation.ObjectNotNull(serializableType);
            Type listType = typeof(List<>).MakeGenericType(serializableType);
            IList dstArray = Activator.CreateInstance(listType) as IList;

            // すべての要素について、再帰呼び出しをします。
            foreach (object elementObject in srcList)
            {
                object dstObject = converter.Convert(elementObject, serializableType);
                dstArray.Add(dstObject);
            }

            if (src.GetType().IsArray)
            {
                ArrayList array = new ArrayList(dstArray);
                return array.ToArray(serializableType);
            }
            else
            {
                return dstArray;
            }
        }

        //-----------------------------------------------------------------
        // ConvertClass
        //-----------------------------------------------------------------

        /// <summary>
        /// クラスを変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        public static object ConvertClass(object src, Type dstType, Converter converter)
        {
            // 出力がインタフェース型の場合、インスタンスが生成出来ません。
            if (dstType.IsInterface)
            {
                string message = string.Format(
                    "Invalid interface type is specified in converting from[{0}] to [{1}].",
                    src.GetType().Name,
                    dstType.Name);
                Ensure.Operation.True(false, message);
            }

            // 同じ型でクローン可能ならクローンした結果を返します。
            if (src.GetType() == dstType && src is ICloneable)
            {
                return (src as ICloneable).Clone();
            }

            // 実体を生成します。（引数をとらないコンストラクタが必須である点に注意！）
            object dstModel = converter.CreateInstance(dstType);

            converter.TouchToConvertedCache(src, dstModel);

            // すべてのプロパティについて...
            // 名前付きプロパティを優先して処理するようにします。
            PropertyInfo[] properties = GetObjectProperties(src);
            foreach (PropertyInfo propInfo in properties)
            {
                // 無視属性が付いていたら中断します。
                if (ShouldPropertyBeIgnored(propInfo))
                {
                    continue;
                }

                // 読み込み元プロパティを取得します。
                PropertyInfo srcPropInfo = GetValidSrcPropInfo(propInfo);
                if (srcPropInfo == null)
                {
                    // 発見できなかったら中断します。
                    continue;
                }

                // 変換元インスタンスを取得します。
                object srcObject = srcPropInfo.GetValue(src, null);
                if (srcObject == null)
                {
                    // 発見できなかったら中断します。
                    continue;
                }

                // 書き込み先プロパティを取得します。
                PropertyInfo dstPropInfo =
                    FindDstPropInfo(dstType, converter, srcPropInfo, ref srcObject);
                if (dstPropInfo == null)
                {
                    // 発見できなかったら中断します。
                    continue;
                }

                // 得られたプロパティに無視属性はついていないはず。
                Ensure.Operation.ObjectNotNull(!ShouldPropertyBeIgnored(dstPropInfo));

                // 上書き設定アトリビュートを解釈し、必要に応じて上書き処理をします。
                OverwriteAttribute.KindType overwriteType =
                    ReadOverwriteType(srcPropInfo, dstPropInfo);

                if (overwriteType == OverwriteAttribute.KindType.None)
                {
                    // プロパティに値を設定します。
                    PropertyInfo writablePropInfo = ConverterUtil.GetWritableProperty(dstPropInfo);
                    if (writablePropInfo == null)
                    {
                        string format =
                            "DefaultConverters : [{0}.{1}] must be a writable property. " +
                            "Use OverwriteAttribute instead.";
                        string msg = string.Format(
                            format, dstPropInfo.DeclaringType.Name, dstPropInfo.Name);

                        Ensure.Operation.True(false, msg);
                    }

                    writablePropInfo.SetValue(dstModel, srcObject, null);
                }
                else
                {
                    try
                    {
                        // 上書き設定をします。
                        OverwritePropertyValue(srcObject, dstModel, dstPropInfo, overwriteType);
                    }
                    catch (Exception e)
                    {
                        string msg = string.Format(
                            "DefaultConverters : Error in OverwritePropertyValue [{0}.{1}]. ",
                            dstPropInfo.DeclaringType.Name,
                            dstPropInfo.Name);
                        msg += Environment.NewLine + e.Message;

                        Ensure.Operation.True(false, msg);
                    }
                }
            }

            return dstModel;
        }

        /// <summary>
        /// プロパティ配列を列挙します。
        /// </summary>
        /// <param name="target">対象です。</param>
        /// <returns>プロパティ配列です。</returns>
        private static PropertyInfo[] GetObjectProperties(object target)
        {
            // すべてのプロパティについて...
            // 名前付きプロパティを優先して処理するようにします。
            PropertyInfo[] properties = ConverterUtil.GetProperties(target.GetType());
            PropertyInfo[] sortedProperties = new PropertyInfo[properties.Length];

            int count = 0;
            for (int i = 0; i < properties.Length; i++)
            {
                if (ConverterUtil.GetNamedPropaertyName(properties[i]) != null)
                {
                    sortedProperties[count] = properties[i];
                    count++;
                    properties[i] = null;
                }
            }

            for (int i = 0; i < properties.Length; i++)
            {
                if (properties[i] != null)
                {
                    sortedProperties[count] = properties[i];
                    count++;
                }
            }

            return sortedProperties;
        }

        /// <summary>
        /// 正しい設定元プロパティを取得します。
        /// </summary>
        /// <param name="srcPropInfo">設定元プロパティです。</param>
        /// <returns>正しい設定元プロパティです。</returns>
        private static PropertyInfo GetValidSrcPropInfo(PropertyInfo srcPropInfo)
        {
            // 読み込み不可能なプロパティには対応しません。
            srcPropInfo = ConverterUtil.GetReadableProperty(srcPropInfo);
            if (srcPropInfo == null)
            {
                return null;
            }

            // パラメータつきのプロパティには対応しません。
            if (srcPropInfo.GetIndexParameters().Length > 0)
            {
                return null;
            }

            return srcPropInfo;
        }

        /// <summary>
        /// 設定先プロパティを探します。
        /// </summary>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">コンバータです。</param>
        /// <param name="srcPropInfo">設定元プロパティです。</param>
        /// <param name="srcObject">設定元オブジェクトです。</param>
        /// <returns>設定先プロパティです。</returns>
        private static PropertyInfo FindDstPropInfo(
            Type dstType,
            Converter converter,
            PropertyInfo srcPropInfo,
            ref object srcObject)
        {
            PropertyInfo dstPropInfo;

            // 名前と型が一致する対象プロパティを検索します。
            dstPropInfo = FindDstPropInfoByName(dstType, srcPropInfo);

            // オブジェクト型のプロパティが発見された場合は、
            // 型変換ができるプロパティを探し、発見できればそちらを採用します。
            if (dstPropInfo != null && dstPropInfo.PropertyType == typeof(object))
            {
                PropertyInfo foundProperty = FindDstPropInfoByConvertionResultType(
                    dstType, converter, srcPropInfo, ref srcObject);
                dstPropInfo = foundProperty != null ? foundProperty : dstPropInfo;
            }

            // 発見できない場合は、型変換を試みます。
            // 変換後の型にマッチするプロパティを探します。
            if (dstPropInfo == null)
            {
                dstPropInfo = FindDstPropInfoByConvertionResultType(
                    dstType, converter, srcPropInfo, ref srcObject);
            }

            // それでもプロパティが発見できない場合には、名前の一致するプロパティを
            // 探し、コンバートができないか試みます。
            if (dstPropInfo == null)
            {
                dstPropInfo = FindDstPropInfoByNameMatch(
                    dstType, converter, srcPropInfo, ref srcObject);
            }

            return dstPropInfo;
        }

        /// <summary>
        /// 名前と型が一致する(無視指定が無い)対象プロパティを検索します。
        /// </summary>
        /// <param name="dstType">出力型です。</param>
        /// <param name="srcPropInfo">設定元プロパティです。</param>
        /// <returns>設定先プロパティです。</returns>
        private static PropertyInfo FindDstPropInfoByName(
            Type dstType,
            PropertyInfo srcPropInfo)
        {
            PropertyInfo dstPropInfo = ConverterUtil.FindSetPropertyFromType(
                dstType, srcPropInfo.PropertyType, srcPropInfo.Name);

            if (dstPropInfo != null && ShouldPropertyBeIgnored(dstPropInfo))
            {
                return null;
            }

            return dstPropInfo;
        }

        /// <summary>
        /// 設定先プロパティを変換後の型から探します。
        /// </summary>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">コンバータです。</param>
        /// <param name="srcPropInfo">設定元プロパティです。</param>
        /// <param name="srcObject">設定元オブジェクトです。</param>
        /// <returns>設定先プロパティです。</returns>
        private static PropertyInfo FindDstPropInfoByConvertionResultType(
            Type dstType,
            Converter converter,
            PropertyInfo srcPropInfo,
            ref object srcObject)
        {
            PropertyInfo dstPropInfo = null;

            // 変換できる型でプロパティを探します。
            Type[] resultTypes = converter.FindConvertionResultTypes(srcObject.GetType());
            foreach (Type resultType in resultTypes)
            {
                dstPropInfo = ConverterUtil.FindSetPropertyFromType(
                    dstType, resultType, srcPropInfo.Name);

                // プロパティを発見
                if (dstPropInfo != null)
                {
                    // 発見したが、無視属性付きならパス
                    if (ShouldPropertyBeIgnored(dstPropInfo))
                    {
                        dstPropInfo = null;
                        continue;
                    }

                    // 変換して、変換結果を設定元とする。
                    srcObject = converter.Convert(srcObject, resultType);
                    break;
                }
            }

            return dstPropInfo;
        }

        /// <summary>
        /// 設定先プロパティを名前の一致で探します。
        /// </summary>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">コンバータです。</param>
        /// <param name="srcPropInfo">設定元プロパティです。</param>
        /// <param name="srcObject">設定元オブジェクトです。</param>
        /// <returns>設定先プロパティです。</returns>
        private static PropertyInfo FindDstPropInfoByNameMatch(
            Type dstType,
            Converter converter,
            PropertyInfo srcPropInfo,
            ref object srcObject)
        {
            Predicate<PropertyInfo> match = (prop) => prop.Name == srcPropInfo.Name;
            PropertyInfo foundProp = Array.Find(
                dstType.GetProperties(ConverterUtil.PropartyBindingFlags), match);

            // 無視属性の無いプロパティを発見
            PropertyInfo dstPropInfo = null;
            if (foundProp != null)
            {
                // 無視属性が無ければ採用
                if (!ShouldPropertyBeIgnored(foundProp))
                {
                    srcObject = converter.Convert(srcObject, foundProp.PropertyType);
                    if (srcObject != null)
                    {
                        // 変換が成功すれば採用
                        dstPropInfo = foundProp;
                    }
                }
            }

            return dstPropInfo;
        }

        /// <summary>
        /// 上書き処理の種類を読み取ります。
        /// </summary>
        /// <param name="srcPropInfo">設定元プロパティです。</param>
        /// <param name="dstPropInfo">設定先プロパティです。</param>
        /// <returns>上書き処理の種類です。</returns>
        private static OverwriteAttribute.KindType ReadOverwriteType(
            PropertyInfo srcPropInfo,
            PropertyInfo dstPropInfo)
        {
            Ensure.Argument.NotNull(srcPropInfo);
            Ensure.Argument.NotNull(dstPropInfo);

            OverwriteAttribute.KindType overwriteType;
            overwriteType = ReadOverwriteAttribute(dstPropInfo);

            if (overwriteType == OverwriteAttribute.KindType.None)
            {
                overwriteType = ReadOverwriteAttribute(srcPropInfo);
            }

            // プリミティブ型なら、KindType.Set は行わない。
            if (overwriteType == OverwriteAttribute.KindType.Set)
            {
                if (srcPropInfo.PropertyType.IsPrimitive || dstPropInfo.PropertyType.IsPrimitive)
                {
                    overwriteType = OverwriteAttribute.KindType.None;
                }
            }

            return overwriteType;
        }

        /// <summary>
        /// チャイルドオブジェクトコレクションか判定します。
        /// </summary>
        /// <param name="srcObject">コレクションです。</param>
        /// <returns>チャイルドオブジェクトのコレクションなら true を返します。</returns>
        private static bool IsChildObjectList(object srcObject)
        {
            IEnumerable srcCollection = srcObject as IEnumerable;
            IEnumerator enumartor = srcCollection.GetEnumerator();
            enumartor.Reset();
            if (enumartor.MoveNext())
            {
                IChildObject topItem = enumartor.Current as IChildObject;
                return topItem != null;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// プロパティに上書き設定をします。
        /// </summary>
        /// <param name="srcObject">設定元です。</param>
        /// <param name="dstObject">設定先です。</param>
        /// <param name="dstPropInfo">設定先プロパティです。</param>
        /// <param name="overwriteType">上書き操作の種類です。</param>
        private static void OverwritePropertyValue(
            object srcObject,
            object dstObject,
            PropertyInfo dstPropInfo,
            OverwriteAttribute.KindType overwriteType)
        {
            // リストへの設定を行います。
            if (overwriteType == OverwriteAttribute.KindType.ListSet ||
                overwriteType == OverwriteAttribute.KindType.ListSetOneByOne ||
                overwriteType == OverwriteAttribute.KindType.ListAppend)
            {
                object list = dstPropInfo.GetValue(dstObject, null);
                Ensure.Operation.True(
                    list != null,
                    "DefaultConverters : Target must not be null.");

                // リセットが必要なら設定までにリセットします。
                if (overwriteType == OverwriteAttribute.KindType.ListSet)
                {
                    MethodInfo clearMethod = list.GetType().GetMethod("Clear");
                    Ensure.Operation.ObjectNotNull(clearMethod);

                    clearMethod.Invoke(list, new object[0]);
                }

                // AddRangeが使えないか試みます。
                MethodInfo addRangeMethod = list.GetType().GetMethod("AddRange");
                if (addRangeMethod != null && !IsChildObjectList(srcObject) &&
                    overwriteType != OverwriteAttribute.KindType.ListSetOneByOne)
                {
                    addRangeMethod.Invoke(list, new object[] { srcObject });
                }
                else
                {
                    // 要素を追加します。
                    MethodInfo addMethod = list.GetType().GetMethod("Add");
                    Ensure.Operation.ObjectNotNull(addMethod);

                    object[] parameters = new object[1];
                    foreach (object srcItem in srcObject as IEnumerable)
                    {
                        if (srcItem is IConditionalConvertedListItem &&
                            !(srcItem as IConditionalConvertedListItem).NeedToConverted)
                        {
                            continue;
                        }

                        parameters[0] = srcItem;
                        addMethod.Invoke(list, parameters);
                    }
                }
            }
            else if (overwriteType == OverwriteAttribute.KindType.Set)
            {
                // ISettableインタフェースを持つ、オブジェクトへの設定を行います。
                object dstPropertyObj = dstPropInfo.GetValue(dstObject, null);

                Ensure.Operation.True(
                    dstPropertyObj != null,
                    "DefaultConverters : Target must not be null.");

                if (!(dstPropertyObj is ISettable))
                {
                    string errorMsg = string.Format(
                        "DefaultConverters : [{0}] must implement ISettable.",
                        dstPropertyObj.GetType().Name);

                    Ensure.Operation.True(false, errorMsg);
                }

                ISettable dstSettble = dstPropertyObj as ISettable;
                dstSettble.Set(srcObject);
            }
        }

        /// <summary>
        /// プロパティに無視属性が付いているか調べます。
        /// </summary>
        /// <param name="propInfo">プロパティです。</param>
        /// <returns>無視属性が付いていれば true を返します。</returns>
        private static bool ShouldPropertyBeIgnored(PropertyInfo propInfo)
        {
            Ensure.Argument.NotNull(propInfo);
            return ConverterUtil.FindPropertyAttribute<ConversionIgnoreAttribute>(propInfo) != null;
        }

        /// <summary>
        /// 上書きアトリビュート値を読みます。
        /// </summary>
        /// <param name="propInfo">プロパティです。</param>
        /// <returns>上書きアトリビュートの種類です。</returns>
        private static OverwriteAttribute.KindType ReadOverwriteAttribute(PropertyInfo propInfo)
        {
            Ensure.Argument.NotNull(propInfo);

            // 基底プロパティをさかのぼってアトリビュートを調査します。
            OverwriteAttribute attribute = null;
            while (propInfo != null && attribute == null)
            {
                attribute = ConverterUtil.FindPropertyAttribute<OverwriteAttribute>(propInfo);
                propInfo = ConverterUtil.GetBasePropertyInfo(propInfo);
            }

            return (attribute != null) ? attribute.Kind : OverwriteAttribute.KindType.None;
        }
    }
}
