﻿// ========================================================================
// <copyright file="MessageQueue.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;
using NWCore.Protocols;

namespace NWCore.Viewer
{
    /// <summary>
    /// メッセージキュー
    /// </summary>
    public sealed class MessageQueue
    {
        private List<BaseMessage> m_queue = new List<BaseMessage>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public MessageQueue()
        {
        }


        /// <summary>
        /// Get the number of messages in the queue.
        /// </summary>
        public int Count
        {
            get { return m_queue.Count; }
        }


        /// <summary>
        /// Check if there is any queued message.
        /// </summary>
        /// <returns>True if there are queued messages.</returns>
        public bool HasQueuedMessage()
        {
            return ( m_queue.Count>0 );
        }


        /// <summary>
        /// Push a message to the queue.
        /// </summary>
        public void PushMessage(BaseMessage message)
        {
            lock ( typeof(MessageQueue) )
            {
                // Remove same messages those already queued.
                for ( int i=m_queue.Count-1;i>=0;--i )
                {
                    BaseMessage msg = m_queue[i];
                    if ( msg!=null &&
                         msg.IsOverrideOnPush==true &&
                         msg.GetType()==message.GetType() )
                    {
                        m_queue.RemoveAt( i );
                    }
                }

                // Push the message to the end.
                m_queue.Add( message );
            }
        }


        /// <summary>
        /// Pop out the oldest message from the queue.
        /// </summary>
        public BaseMessage PopMessage()
        {
            if ( m_queue.Count>0 )
            {
                lock ( typeof(MessageQueue) )
                {
                    if ( m_queue.Count>0 )
                    {
                        BaseMessage msg = m_queue[0];
                        m_queue.RemoveAt(0);
                        return msg;
                    }
                }
            }

            return null;
        }


        /// <summary>
        /// Pop out all the messages from the queue.
        /// </summary>
        public BaseMessage[] PopMessages()
        {
            if ( m_queue.Count>0 )
            {
                lock ( typeof(MessageQueue) )
                {
                    BaseMessage[] messages = m_queue.ToArray();

                    m_queue.Clear();

                    return messages;
                }
            }

            return new BaseMessage[0];
        }
    }
}
