﻿// ========================================================================
// <copyright file="Vector3i.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NintendoWare.ToolDevelopmentKit.ComponentModel;

namespace NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0
{
    /// <summary>
    /// 3 次元のベクトルを表します
    /// </summary>
    public sealed class Vector3i : IVectori, ICloneable, ISettable, IEquatable<Vector3i>
    {
        #region Member Variables

        private int x;
        private int y;
        private int z;

        #endregion

        #region Constructors

        /// <summary>
        /// デフォルトコンストラクタです
        /// </summary>
        public Vector3i()
        {
        }

        /// <summary>
        /// コピーコンストラクタです
        /// </summary>
        /// <param name="source">コピー元のベクトルです</param>
        public Vector3i(Vector3i source)
        {
            this.Set(source);
        }

        /// <summary>
        /// X,Y,Z座標を指定の値に設定するコンストラクタです
        /// </summary>
        /// <param name="x">X座標です</param>
        /// <param name="y">Y座標です</param>
        /// <param name="z">Z座標です</param>
        public Vector3i(int x, int y, int z)
        {
            this.Set(x, y, z);
        }

        /// <summary>
        /// X,Y,Z座標を指定の値に設定するコンストラクタです
        /// </summary>
        /// <param name="source">X,Y,Z座標の配列です</param>
        public Vector3i(int[] source)
        {
            this.Set(source);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 次元数を取得します
        /// </summary>
        [XmlIgnore]
        public int Dimension
        {
            get
            {
                return 3;
            }
        }

        /// <summary>
        /// X 成分を取得または設定します
        /// </summary>
        [NotifyParentProperty(true)]
        [XmlElement("x")]
        public int X
        {
            get
            {
                return this.x;
            }

            set
            {
                if (this.x == value)
                {
                    return;
                }

                this.x = value;
            }
        }

        /// <summary>
        /// Y 成分を取得または設定します
        /// </summary>
        [NotifyParentProperty(true)]
        [XmlElement("y")]
        public int Y
        {
            get
            {
                return this.y;
            }

            set
            {
                if (this.y == value)
                {
                    return;
                }

                this.y = value;
            }
        }

        /// <summary>
        /// Z 成分を取得または設定します
        /// </summary>
        [NotifyParentProperty(true)]
        [XmlElement("z")]
        public int Z
        {
            get
            {
                return this.z;
            }

            set
            {
                if (this.z == value)
                {
                    return;
                }

                this.z = value;
            }
        }

        /// <summary>
        /// ベクトルの長さを取得します
        /// </summary>
        /// <returns>ベクトルの長さです</returns>
        [XmlIgnore]
        public int Length
        {
            get { return (int)Math.Sqrt(this.LengthSquared); }
        }

        /// <summary>
        /// 長さの2乗を取得します
        /// </summary>
        /// <returns>ベクトルの長さの２乗です</returns>
        [XmlIgnore]
        public int LengthSquared
        {
            get { return (this.X * this.X) + (this.Y * this.Y) + (this.Z * this.Z); }
        }

        /// <summary>
        /// 指定したインデックス位置の成分を取得、または設定します
        /// </summary>
        /// <param name="i">インデックスです</param>
        /// <returns>成分です</returns>
        [XmlIgnore]
        public int this[int i]
        {
            get
            {
                switch (i)
                {
                    case 0:
                        return this.X;
                    case 1:
                        return this.Y;
                    case 2:
                        return this.Z;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            set
            {
                switch (i)
                {
                    case 0:
                        this.X = value;
                        break;
                    case 1:
                        this.Y = value;
                        break;
                    case 2:
                        this.Z = value;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }

        #endregion

        #region Operators

        /// <summary>
        /// 2 つのベクトルを加算します
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです</param>
        /// <param name="rhs">演算の右のベクトルです</param>
        /// <returns>lhsとrhsを加算した結果のベクトルです</returns>
        public static Vector3i operator +(Vector3i lhs, Vector3i rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector3i(lhs.X + rhs.X, lhs.Y + rhs.Y, lhs.Z + rhs.Z);
        }

        /// <summary>
        /// 2 つのベクトルを減算します
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです</param>
        /// <param name="rhs">演算の右のベクトルです</param>
        /// <returns>lhsとrhsを減算した結果のベクトルです</returns>
        public static Vector3i operator -(Vector3i lhs, Vector3i rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector3i(lhs.X - rhs.X, lhs.Y - rhs.Y, lhs.Z - rhs.Z);
        }

        /// <summary>
        /// ベクトルにスカラーを乗算します
        /// </summary>
        /// <param name="vec">演算の左のベクトルです</param>
        /// <param name="scalar">演算の右のスカラーです</param>
        /// <returns>vecとscalarを乗算した結果のベクトルです</returns>
        public static Vector3i operator *(Vector3i vec, int scalar)
        {
            Ensure.Argument.NotNull(vec);

            return new Vector3i(vec.X * scalar, vec.Y * scalar, vec.Z * scalar);
        }

        /// <summary>
        /// ベクトルをスカラーで除算します
        /// </summary>
        /// <param name="vec">演算の左のベクトルです</param>
        /// <param name="scalar">演算の右のスカラーです</param>
        /// <returns>vecとscalarを除算した結果のベクトルです</returns>
        public static Vector3i operator /(Vector3i vec, int scalar)
        {
            Ensure.Argument.NotNull(vec);
            Ensure.Operation.DividerNotZero(scalar);

            int invScale = 1 / scalar;
            return new Vector3i(vec.X * invScale, vec.Y * invScale, vec.Z * invScale);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 成分が全て 0 のベクトルを作成します
        /// </summary>
        /// <returns>成分が全て 0 のベクトルです</returns>
        [XmlIgnore]
        public static Vector3i Zero
        {
            get
            {
                Vector3i result = new Vector3i();
                result.SetZero();
                return result;
            }
        }

        /// <summary>
        /// 成分が全て 1 のベクトルを作成します
        /// </summary>
        /// <returns>成分が全て 1 のベクトルです</returns>
        [XmlIgnore]
        public static Vector3i One
        {
            get
            {
                Vector3i result = new Vector3i();
                result.SetOne();
                return result;
            }
        }

        /// <summary>
        /// 2 つのベクトルの内積を計算します
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです</param>
        /// <param name="rhs">演算の右のベクトルです</param>
        /// <returns>lhsとrhsの外積を計算した結果のスカラーです</returns>
        public static int Dot(Vector3i lhs, Vector3i rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return (lhs.X * rhs.X) + (lhs.Y * rhs.Y) + (lhs.Z * rhs.Z);
        }

        /// <summary>
        /// 2 つのベクトルの外積を計算します
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです</param>
        /// <param name="rhs">演算の右のベクトルです</param>
        /// <returns>lhsとrhsの外積を計算した結果のベクトルです</returns>
        public static Vector3i Cross(Vector3i lhs, Vector3i rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector3i(
                (lhs.Y * rhs.Z) - (lhs.Z * rhs.Y),
                (lhs.Z * rhs.X) - (lhs.X * rhs.Z),
                (lhs.X * rhs.Y) - (lhs.Y * rhs.X));
        }

        /// <summary>
        /// 指定したベクトルを正規化したベクトルを作成します
        /// </summary>
        /// <remarks>
        /// ベクトルの長さが FloatUtility.Epsilon 以下の場合は
        /// DivideByZeroException　例外を投げます
        /// </remarks>
        /// <param name="vec">指定のベクトルです</param>
        /// <returns>正規化したベクトルです</returns>
        public static Vector3i Normalize(Vector3i vec)
        {
            Vector3i result = new Vector3i(vec);
            result.Normalize();
            return result;
        }

        /// <summary>
        /// 正規化したベクトルを作成します
        /// ベクトル vec の長さが FloatUtility.Epsilon 以下の場合は
        /// 指定したベクトル substitute に設定します
        /// </summary>
        /// <param name="vec">元のベクトルです</param>
        /// <param name="substitute">ベクトルの長さが FloatUtility.Epsilon 以下の場合に
        /// 代わりに設定するベクトルです</param>
        /// <returns>ベクトルの長さを正規化できた場合はtrueを返します
        /// それ以外の場合は false を返します</returns>
        public static bool SafeNormalize(Vector3i vec, Vector3i substitute)
        {
            Ensure.Argument.NotNull(vec);
            Ensure.Argument.NotNull(substitute);

            int lengthSquared = vec.LengthSquared;
            if (lengthSquared == 0)
            {
                vec.Set(substitute);
                return false;
            }

            vec.Normalize(lengthSquared);
            return true;
        }

        /// <summary>
        /// X,Y,Z座標を格納した配列を返します
        /// </summary>
        /// <returns>X,Y,Z座標の配列です</returns>
        public int[] ToArray()
        {
            return new int[] { this.X, this.Y, this.Z };
        }

        /// <summary>
        /// オブジェクトを設定します
        /// </summary>
        /// <param name="source">設定するオブジェクトです</param>
        public void Set(object source)
        {
            Ensure.Argument.True(source is Vector3i);
            this.Set(source as Vector3i);
        }

        /// <summary>
        /// ベクトルを設定します
        /// </summary>
        /// <param name="source">設定するベクトルです</param>
        public void Set(Vector3i source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.X, source.Y, source.Z);
        }

        /// <summary>
        /// X,Y,Z座標を指定の値に設定します
        /// </summary>
        /// <param name="x">X座標です</param>
        /// <param name="y">Y座標です</param>
        /// <param name="z">Z座標です</param>
        public void Set(int x, int y, int z)
        {
            this.X = x;
            this.Y = y;
            this.Z = z;
        }

        /// <summary>
        /// X,Y,Z座標を指定の値に設定します
        /// </summary>
        /// <param name="source">X,Y,Z座標の配列です</param>
        public void Set(int[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(this.Dimension == source.Length);

            this.Set(source[0], source[1], source[2]);
        }

        /// <summary>
        /// 成分を0に設定します
        /// </summary>
        public void SetZero()
        {
            this.Set(0, 0, 0);
        }

        /// <summary>
        /// 成分を1に設定します
        /// </summary>
        public void SetOne()
        {
            this.Set(1, 1, 1);
        }

        /// <summary>
        /// オブジェクトを複製します
        /// </summary>
        /// <returns>複製したオブジェクトです</returns>
        public object Clone()
        {
            return new Vector3i(this);
        }

        /// <summary>
        /// 指定したベクトルとの内積を計算します
        /// </summary>
        /// <param name="vec">掛けるベクトルです</param>
        /// <returns>vecとの内積を計算した結果のスカラーです</returns>
        public int Dot(Vector3i vec)
        {
            return Dot(this, vec);
        }

        /// <summary>
        /// 指定したベクトルとの外積を計算します
        /// </summary>
        /// <param name="vec">掛けるベクトルです</param>
        /// <returns>vecとの外積を計算した結果のベクトルです</returns>
        public Vector3i Cross(Vector3i vec)
        {
            return Cross(this, vec);
        }

        /// <summary>
        /// 正規化したベクトルに設定します
        /// </summary>
        /// <remarks>
        /// ベクトルの長さが FloatUtility.Epsilon 以下の場合は
        /// DivideByZeroException　例外を投げます
        /// </remarks>
        public void Normalize()
        {
            int lengthSquared = this.LengthSquared;
            if (lengthSquared == 0)
            {
                throw new DivideByZeroException();
            }

            Normalize(lengthSquared);
        }

        /// <summary>
        /// 正規化したベクトルに設定します
        /// ベクトルの長さが FloatUtility.Epsilon 以下の場合は
        /// 指定したベクトルに設定します
        /// </summary>
        /// <param name="substitute">ベクトルの長さが FloatUtility.Epsilon 以下の場合に
        /// 代わりに設定するベクトルです</param>
        /// <returns>ベクトルの長さを正規化できた場合はtrueを返します
        /// それ以外の場合は false を返します</returns>
        public bool SafeNormalize(Vector3i substitute)
        {
            return SafeNormalize(this, substitute);
        }

        /// <summary>
        /// 等値であるかどうか比較します
        /// </summary>
        /// <param name="other">比較対象です</param>
        /// <returns>等値であればtrueを返します</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Vector3i);
        }

        /// <summary>
        /// 等値であるかどうか比較します
        /// </summary>
        /// <param name="other">比較対象です</param>
        /// <returns>等値であれば true を返します</returns>
        public bool Equals(Vector3i other)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                (this.X == other.X) &&
                (this.Y == other.Y) &&
                (this.Z == other.Z);
        }

        /// <summary>
        /// 等値であるかどうか比較します
        /// </summary>
        /// <param name="other">比較対象です</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です</param>
        /// <returns>等値であればtrueを返します</returns>
        public bool Equals(Vector3i other, int tolerance)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                FloatUtility.NearlyEqual(this.X, other.X, tolerance) &&
                FloatUtility.NearlyEqual(this.Y, other.Y, tolerance) &&
                FloatUtility.NearlyEqual(this.Z, other.Z, tolerance);
        }

        /// <summary>
        /// ハッシュ値を取得します
        /// </summary>
        /// <returns>ハッシュ値です</returns>
        public override int GetHashCode()
        {
            return
                this.X.GetHashCode() ^
                this.Y.GetHashCode() ^
                this.Z.GetHashCode();
        }

        /// 現在のオブジェクトを表す文字列を返します
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です</returns>
        public override string ToString()
        {
            return string.Format("[{0} {1} {2}]", this.X, this.Y, this.Z);
        }

        // 長さの2乗を受け取り長さを１に正規化
        private void Normalize(int lengthSquared)
        {
            if (lengthSquared == 1)
            {
                return;
            }

            Debug.Assert(lengthSquared == this.LengthSquared, "Unexpected case!");
            int inverseLength = 1 / (int)Math.Sqrt(lengthSquared);
            this.Set(this * inverseLength);
        }

        #endregion
    }
}
