﻿// ========================================================================
// <copyright file="EmitterGameData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0
{
    /// <summary>
    /// エミッタ用ゲーム設定データ
    /// </summary>
    [Serializable]
    public class EmitterGameData : BaseDataModel<EmitterGameData>
    {
        #region Properties

        /// <summary>
        /// エミッタ名
        /// </summary>
        [XmlAttribute]
        public string Name { get; set; }

        /// <summary>
        /// エミッタインデックス
        /// </summary>
        [XmlAttribute]
        public int Index { get; set; }

        /// <summary>
        /// このエミッタは有効か？
        /// </summary>
        [XmlAttribute]
        public bool DispEmitter { get; set; }

        #endregion

        #region コンストラクタ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EmitterGameData()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EmitterGameData(EmitterGameData source)
        {
            Ensure.Argument.NotNull(source);
            this.Set(source);
        }

        /// <summary>
        /// シリアライズのための設定
        /// </summary>
        public void SetForSerialize(EmitterData emitter, int index)
        {
            this.Name = emitter.Name;
            this.Index = index;
            this.DispEmitter = emitter.EmitterGameData.DispEmitter;
        }

        /// <summary>
        /// デシリアライズのための設定
        /// </summary>
        public void SetForDeserialize(EmitterData emitter)
        {
            emitter.EmitterGameData.DispEmitter = this.DispEmitter;
        }

        #endregion

        #region Utility

        /// <summary>
        /// クローン
        /// </summary>
        public override EmitterGameData Clone()
        {
            return new EmitterGameData(this);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.Name = String.Empty;
            this.Index = -1;
            this.DispEmitter = true;
        }

        /// <summary>
        /// 設定します。
        /// </summary>
        public override void Set(EmitterGameData source)
        {
            Ensure.Argument.NotNull(source);

            this.Name = source.Name;
            this.Index = source.Index;
            this.DispEmitter = source.DispEmitter;
        }

        #endregion
    }


    /// <summary>
    /// プレビュー用ゲーム設定データ
    /// </summary>
    [Serializable]
    public class PreviewGameData : BaseDataModel<PreviewGameData>
    {
        #region Properties

        /// <summary>
        /// エミッタ名
        /// </summary>
        [XmlAttribute]
        public string Name { get; set; }

        /// <summary>
        /// エミッタインデックス
        /// </summary>
        [XmlAttribute]
        public int Index { get; set; }

        /// <summary>
        /// このエミッタは有効か？
        /// </summary>
        [XmlAttribute]
        public bool IsShow { get; set; }

        #endregion

        #region コンストラクタ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PreviewGameData()
        {
        }

        public PreviewGameData(PreviewGameData source)
        {
            Set(source);
        }

        /// <summary>
        /// シリアライズのための設定
        /// </summary>
        public void SetForSerialize(IGameSettingsDocument preview, int index)
        {
            this.Name = preview.Name;
            this.Index = index;
            this.IsShow = preview.GameConfigData.DispPreviewNode;
        }

        /// <summary>
        /// デシリアライズのための設定
        /// </summary>
        public void SetForDeserialize(IGameSettingsDocument preview)
        {
            preview.GameConfigData.DispPreviewNode = this.IsShow;
        }

        #endregion

        #region Utility

        /// <summary>
        /// クローン
        /// </summary>
        public override PreviewGameData Clone()
        {
            return new PreviewGameData(this);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.Name = String.Empty;
            this.IsShow = true;
        }

        /// <summary>
        /// 設定します。
        /// </summary>
        public override void Set(PreviewGameData source)
        {
            Ensure.Argument.NotNull(source);

            this.Name = source.Name;
            this.Index = source.Index;
            this.IsShow = source.IsShow;
        }

        #endregion
    }
}
