﻿// ========================================================================
// <copyright file="AnimTableList.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System.Collections.Generic;
using NintendoWare.ToolDevelopmentKit;
using NWCore.Utility;

namespace NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0
{
    /// <summary>
    /// Class for the data model for the animation table list.
    /// </summary>
    public abstract class AnimTableList : BaseDataModel<AnimTableList>
    {
        #region Class for the registered animation table information

        /// <summary>
        /// Class for the registered animation table information.
        /// </summary>
        protected class RegisteredAnimTableInfo
        {
            #region Constructor

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="target">The applied target type of the animation table.</param>
            /// <param name="templateKeyFrame">The template key frame data.</param>
            /// <param name="bCreateDefaultKeyFrame">True to create the first key frame with default values on table creation.</param>
            public RegisteredAnimTableInfo( AnimTableTargetTypes target,
                                            AnimTableKeyFrameData templateKeyFrame,
                                            bool bCreateDefaultKeyFrame )
            {
                this.Target                = target;
                this.TemplateKeyFrame      = templateKeyFrame;
                this.CreateDefaultKeyFrame = bCreateDefaultKeyFrame;
            }

            #endregion

            #region Properties

            /// <summary>
            /// Get or set the applied target type of the animation table.
            /// </summary>
            public AnimTableTargetTypes Target { get; set; }


            /// <summary>
            /// Get or set key frame data template for create key frames.
            /// </summary>
            public AnimTableKeyFrameData TemplateKeyFrame { get; set; }


            /// <summary>
            /// Get or set the flag indicating whether to create the first key frame on table creation.
            /// </summary>
            public bool CreateDefaultKeyFrame { get; set; }

            #endregion

            #region Utility

            /// <summary>
            /// Clone the instance.
            /// </summary>
            /// <returns>The created instance.</returns>
            public RegisteredAnimTableInfo Clone()
            {
                RegisteredAnimTableInfo dst =
                    new RegisteredAnimTableInfo( this.Target,
                                                 this.TemplateKeyFrame.Clone(),
                                                 this.CreateDefaultKeyFrame );

                return dst;
            }

            #endregion
        }

        #endregion

        #region Initialization

        public override void Initialize()
        {
            if ( this.Animations!=null )
            {
                foreach ( AnimTableData data in this.Animations )
                {
                    data.Initialize();
                }
            }
        }

        #endregion

        #region Properties

        /// <summary>
        /// The list of the animations.
        /// </summary>
        public List<AnimTableData> Animations { get; set; }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableList source )
        {
            Ensure.Argument.NotNull(source);

            this.Animations.Clear();
            foreach ( AnimTableData data in source.Animations )
            {
                this.Animations.Add( data.Clone() );
            }
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableList Clone()
        {
            return null;
        }

        #endregion

        #region Animation tables

        /// <summary>
        /// Register animation tables with the animation table name
        /// and an empty key frame data of the desired key type as
        /// a template for creating new key frames for deserilization.
        /// The given key frame data can be of one of the derived
        /// class of AnimTableKeyFrameData, eq. IntAnimTableKeyFrameData
        /// or FloatAnimTableKeyFrameData.
        /// </summary>
        /// <param name="target">Applied target type of the animation table.</param>
        /// <param name="templateKeyFrame">The template key frame data.</param>
        /// <param name="bCreateDefaultKeyFrame">True to create default key frame on table creation.</param>
        /// <returns>True on success.</returns>
        public bool RegisterAnimationTable( AnimTableTargetTypes target,
                                            AnimTableKeyFrameData templateKeyFrame,
                                            bool bCreateDefaultKeyFrame = false )
        {
            if ( templateKeyFrame==null )
                return false;

            RegisteredAnimTableInfo registeredInfo;
            RegisteredAnimTableInfo newInfo = new RegisteredAnimTableInfo( target,
                                                                           templateKeyFrame,
                                                                           bCreateDefaultKeyFrame );
            if ( m_registeredAnimTableInfo.TryGetValue(target,
                                                       out registeredInfo)==true )
            {
                // This animation table name is already registered. Assert and bale out.
                System.Diagnostics.Debug.Assert( false, "The animation table type \"" + target.ToString() + "\" is already registered." );
                return false;
            }

            // Register the animation table and the key frame data template.
            m_registeredAnimTableInfo.Add( target, newInfo );

            return true;
        }


        /// <summary>
        /// Create the animation table key frame with the template registered
        /// with the animation table target.
        /// </summary>
        /// <param name="target">Applied target type of the animation table.</param>
        /// <returns>The animation table key frame data.</returns>
        public AnimTableKeyFrameData CreateAnimationTableKeyFrame( AnimTableTargetTypes target )
        {
            RegisteredAnimTableInfo info;
            if ( m_registeredAnimTableInfo.TryGetValue(target,
                                                       out info)==false )
            {
                System.Diagnostics.Debug.Assert( false, "The requested animation table with the given target is not registered." );
                return null;
            }

            if ( info==null || info.TemplateKeyFrame==null )
                return null;

            return info.TemplateKeyFrame.Clone();
        }


        /// <summary>
        /// Create and add all the registered animation tables to the list.
        /// </summary>
        public void AddRegisteredAnimationTables()
        {
            foreach ( KeyValuePair<AnimTableTargetTypes, RegisteredAnimTableInfo> pair in m_registeredAnimTableInfo )
            {
                // Check if the animation table already exists.
                bool bFound = false;
                foreach ( AnimTableData existingAnimTable in this.Animations )
                {
                    if ( existingAnimTable.Name==pair.Key )
                    {
                        bFound = true;
                        break;
                    }
                }

                // The animation table was not found, create it.
                if ( bFound==false )
                {
                    AnimTableData animTable = new AnimTableData();
                    animTable.Initialize();
                    animTable.Name = pair.Value.Target;

                    if ( pair.Value.CreateDefaultKeyFrame==true )
                    {
                        animTable.KeyFrames.Add( CreateAnimationTableKeyFrame(pair.Value.Target) );
                    }

                    this.Animations.Add( animTable );
                }
            }
        }


        /// <summary>
        /// Find the animation table data with the given target type.
        /// </summary>
        /// <param name="target">Applied target type of the animation table.</param>
        /// <returns>The animation table data or null if not found.</returns>
        public AnimTableData FindAnimationTableData( AnimTableTargetTypes target )
        {
            foreach ( AnimTableData data in this.Animations )
            {
                if ( data!=null && data.Name==target )
                    return data;
            }

            return null;
        }

        #endregion

        #region Member variables

        private Dictionary<AnimTableTargetTypes, RegisteredAnimTableInfo> m_registeredAnimTableInfo =
            new Dictionary<AnimTableTargetTypes, RegisteredAnimTableInfo>();

        #endregion
    }

    #region The animation table list for the emitters

    public class EmitterAnimTableList : AnimTableList
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public EmitterAnimTableList()
        {
            RegisterAnimationTable( AnimTableTargetTypes.emit_rate,            new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.all_directional_vel,  new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.directional_vel,      new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.ptcl_life,            new IntAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.emitter_scale_x,      new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_scale_y,      new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_scale_z,      new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.emitter_rot_x,        new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_rot_y,        new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_rot_z,        new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.emitter_trans_x,      new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_trans_y,      new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_trans_z,      new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.color0_r,             new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.color0_g,             new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.color0_b,             new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.color1_r,             new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.color1_g,             new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.color1_b,             new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.alpha,                new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.emitter_form_scale_x, new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_form_scale_y, new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.emitter_form_scale_z, new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.ptcl_scale_x,         new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.ptcl_scale_y,         new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.gravity,              new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.gravity_child,        new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.field_random_x,       new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_random_y,       new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_random_z,       new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_magnet,         new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_span_rot,       new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_span_outer,     new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_convergence,    new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_pos_add_x,      new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_pos_add_y,      new FloatAnimTableKeyFrameData() );
            RegisterAnimationTable( AnimTableTargetTypes.field_pos_add_z,      new FloatAnimTableKeyFrameData() );

            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_color0_r,   new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_color0_g,   new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_color0_b,   new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_color1_r,   new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_color1_g,   new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_color1_b,   new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_alpha,      new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_scale_x,    new FloatAnimTableKeyFrameData(1.0f), true );
            RegisterAnimationTable( AnimTableTargetTypes.anim_8key_scale_y,    new FloatAnimTableKeyFrameData(1.0f), true );

            this.Animations = new List<AnimTableData>();
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">The source to copy from.</param>
        public EmitterAnimTableList( AnimTableList source ) :
            this()
        {
            this.Set(source);
        }

        #endregion

        #region Utility

        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableList Clone()
        {
            return new EmitterAnimTableList( this );
        }

        #endregion
    }

    #endregion
}
