﻿// ========================================================================
// <copyright file="ChildData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_2.Build_0.Revision_0
{
    /// <summary>
    /// チャイルド
    /// </summary>
    public sealed class ChildData : BaseDataModel<ChildData>
    {
        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ChildData()
        {
            this.Color = new RgbaColor();        // 色
            this.TextureRes = new TextureRes();  // チャイルド用テクスチャ
            this.Scale = new Vector2(1, 1); ;    // スケール値
            this.RandVel = new Vector3();        // ランダム速度
            this.UserShaderParams = new float[32];
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public ChildData(ChildData source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.AirRegist = 1.0f;
            this.Alpha = 1.0f;
            this.MeshType = MeshType.Particle;
            this.BillboardType = BillboardType.Billboard;
            this.BillboardPrimitiveFileSource = string.Empty;
            this.EmitRate = 1;
            this.EmitStep = 10;
            this.EmitTiming = 60;
            this.ScaleInheritRate = 1.0f;
            this.VelInheritRate = 0.0f;
            this.VelInheritRate = 1.0f;
            this.UseSoftParticle = false;
            this.SoftParticleFadeDistance = 0.0f;
            this.SoftParticleVolumeParam = 0.0f;
            this.ShaderType = ShaderType.Particle;
            this.ShaderParam0 = 0.0f;
            this.ShaderParam1 = 0.0f;

            this.UserShaderSwitch = 0x01010101;
            this.UserShaderFlags = 0;
            for (int i = 0; i < 32; ++i)
            {
                this.UserShaderParams[i] = 0.0f;
            }

            this.RandomScale = 0.0f;

            float worldScale = s_WorldScaleRange / 100.0f;
            {
                this.FigureVelocity = 0.1f * worldScale;
                this.Scale.Set(1*worldScale, 1*worldScale);
            }

            this.ZBufATestType = ZBufATestType.Normal;
            this.DisplaySide = DisplaySideTypes.Both;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(ChildData source)
        {
            Ensure.Argument.NotNull(source);

            this.Color.Set(source.Color);
            this.TextureRes.Set(source.TextureRes);
            this.Scale.Set(source.Scale);
            this.RandVel.Set(source.RandVel);

            this.AirRegist = source.AirRegist;
            this.Alpha = source.Alpha;
            this.AlphaInit = source.AlphaInit;
            this.AlphaTarget = source.AlphaTarget;
            this.MeshType = source.MeshType;
            this.BillboardType = source.BillboardType;
            this.BillboardPrimitiveFileSource = source.BillboardPrimitiveFileSource;
            this.CombinerType = source.CombinerType;
            this.EmitRate = source.EmitRate;
            this.EmitStep = source.EmitStep;
            this.EmitTiming = source.EmitTiming;
            this.FigureVelocity = source.FigureVelocity;
            this.InitPosRand = source.InitPosRand;
            this.RandomScale = source.RandomScale;
            this.ScaleInheritRate = source.ScaleInheritRate;
            this.VelInheritRate = source.VelInheritRate;
            this.ZBufATestType = source.ZBufATestType;
            this.DisplaySide = source.DisplaySide;
            this.UseSoftParticle = source.UseSoftParticle;
            this.SoftParticleFadeDistance = source.SoftParticleFadeDistance;
            this.SoftParticleVolumeParam = source.SoftParticleVolumeParam;
            this.ShaderType = source.ShaderType;
            this.ShaderSetting = source.ShaderSetting;
            this.ShaderParam0 = source.ShaderParam0;
            this.ShaderParam1 = source.ShaderParam1;
            this.UserShaderSwitch = source.UserShaderSwitch;
            this.UserShaderFlags = source.UserShaderFlags;
            for (int i = 0; i < 32; ++i)
            {
                this.UserShaderParams[i] = source.UserShaderParams[i];
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ChildData Clone()
        {
            return new ChildData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 放出量
        /// </summary>
        public int EmitRate { get; set; }

        /// <summary>
        /// タイミング
        /// </summary>
        public int EmitTiming { get; set; }

        /// <summary>
        /// 放出間隔
        /// </summary>
        public int EmitStep { get; set; }

        /// <summary>
        /// 速度継承率
        /// </summary>
        public float VelInheritRate { get; set; }

        /// <summary>
        /// 形状速度
        /// </summary>
        public float FigureVelocity { get; set; }

        /// <summary>
        /// ランダム速度
        /// </summary>
        public Vector3 RandVel { get; set; }

        /// <summary>
        /// 初期位置のランダム量
        /// </summary>
        public float InitPosRand { get; set; }

        /// <summary>
        /// Mesh type
        /// </summary>
        public MeshType MeshType { get; set; }

        /// <summary>
        /// ビルボードタイプ
        /// </summary>
        public BillboardType BillboardType { get; set; }

        /// <summary>
        /// プリミティブファイルパス
        /// </summary>
        public string BillboardPrimitiveFileSource { get; set; }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType ZBufATestType { get; set; }

        /// <summary>
        /// カリング
        /// </summary>
        public DisplaySideTypes DisplaySide { get; set; }

        /// <summary>
        /// チャイルド用テクスチャ
        /// </summary>
        public TextureRes TextureRes { get; set; }

        /// <summary>
        /// 色
        /// </summary>
        public RgbaColor Color { get; set; }

        /// <summary>
        /// アルファ
        /// </summary>
        public float Alpha { get; set; }

        /// <summary>
        /// アルファアニメのターゲット
        /// </summary>
        public float AlphaTarget { get; set; }

        /// <summary>
        /// アルファアニメの初期値
        /// </summary>
        public float AlphaInit { get; set; }

        /// <summary>
        /// スケール継承率
        /// </summary>
        public float ScaleInheritRate { get; set; }

        /// <summary>
        /// スケール値
        /// </summary>
        public Vector2 Scale { get; set; }

        /// <summary>
        /// ランダム値
        /// </summary>
        public float RandomScale { get; set; }

        /// <summary>
        /// コンバイナタイプ
        /// </summary>
        public CombinerType CombinerType { get; set; }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float AirRegist { get; set; }

        /// <summary>
        /// シェーダータイプ：ソフトパーティクルを利用する
        /// </summary>
        public bool UseSoftParticle { get; set; }

        /// <summary>
        /// Fade distance for soft particle.
        /// </summary>
        public float SoftParticleFadeDistance { get; set; }

        /// <summary>
        /// Volume parameter for the soft particle
        /// </summary>
        public float SoftParticleVolumeParam { get; set; }

        /// <summary>
        /// シェーダータイプ
        /// </summary>
        public ShaderType ShaderType { get; set; }

        /// <summary>
        /// シェーダセッティング
        /// </summary>
        public uint ShaderSetting { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam0 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam1 { get; set; }

        /// <summary>
        /// ユーザーシェーダスイッチ
        /// </summary>
        public uint UserShaderSwitch { get; set; }

        #region Accessing each byte of UserShaderSwitch

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch0
        {
            get { return (UserShaderSwitch & ((uint)0x000000FF)); }
            set
            {
                uint iMask = (uint)0x000000FF;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch1
        {
            get { return (UserShaderSwitch & ((uint)0x0000FF00)); }
            set
            {
                uint iMask = (uint)0x0000FF00;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch2
        {
            get { return (UserShaderSwitch & ((uint)0x00FF0000)); }
            set
            {
                uint iMask = (uint)0x00FF0000;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch3
        {
            get { return (UserShaderSwitch & ((uint)0xFF000000)); }
            set
            {
                uint iMask = (uint)0xFF000000;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        #endregion

        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        public uint UserShaderFlags { get; set; }

        #region Accessing each bit of UserShaderFlags

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag0
        {
            get { return ((UserShaderFlags & ((uint)1 << 0)) != 0); }
            set
            {
                uint iMask = (uint)1 << 0;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag1
        {
            get { return ((UserShaderFlags & ((uint)1 << 1)) != 0); }
            set
            {
                uint iMask = (uint)1 << 1;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag2
        {
            get { return ((UserShaderFlags & ((uint)1 << 2)) != 0); }
            set
            {
                uint iMask = (uint)1 << 2;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag3
        {
            get { return ((UserShaderFlags & ((uint)1 << 3)) != 0); }
            set
            {
                uint iMask = (uint)1 << 3;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag4
        {
            get { return ((UserShaderFlags & ((uint)1 << 4)) != 0); }
            set
            {
                uint iMask = (uint)1 << 4;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag5
        {
            get { return ((UserShaderFlags & ((uint)1 << 5)) != 0); }
            set
            {
                uint iMask = (uint)1 << 5;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag6
        {
            get { return ((UserShaderFlags & ((uint)1 << 6)) != 0); }
            set
            {
                uint iMask = (uint)1 << 6;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag7
        {
            get { return ((UserShaderFlags & ((uint)1 << 7)) != 0); }
            set
            {
                uint iMask = (uint)1 << 7;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag8
        {
            get { return ((UserShaderFlags & ((uint)1 << 8)) != 0); }
            set
            {
                uint iMask = (uint)1 << 8;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag9
        {
            get { return ((UserShaderFlags & ((uint)1 << 9)) != 0); }
            set
            {
                uint iMask = (uint)1 << 9;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag10
        {
            get { return ((UserShaderFlags & ((uint)1 << 10)) != 0); }
            set
            {
                uint iMask = (uint)1 << 10;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag11
        {
            get { return ((UserShaderFlags & ((uint)1 << 11)) != 0); }
            set
            {
                uint iMask = (uint)1 << 11;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag12
        {
            get { return ((UserShaderFlags & ((uint)1 << 12)) != 0); }
            set
            {
                uint iMask = (uint)1 << 12;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag13
        {
            get { return ((UserShaderFlags & ((uint)1 << 13)) != 0); }
            set
            {
                uint iMask = (uint)1 << 13;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag14
        {
            get { return ((UserShaderFlags & ((uint)1 << 14)) != 0); }
            set
            {
                uint iMask = (uint)1 << 14;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag15
        {
            get { return ((UserShaderFlags & ((uint)1 << 15)) != 0); }
            set
            {
                uint iMask = (uint)1 << 15;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag16
        {
            get { return ((UserShaderFlags & ((uint)1 << 16)) != 0); }
            set
            {
                uint iMask = (uint)1 << 16;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag17
        {
            get { return ((UserShaderFlags & ((uint)1 << 17)) != 0); }
            set
            {
                uint iMask = (uint)1 << 17;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag18
        {
            get { return ((UserShaderFlags & ((uint)1 << 18)) != 0); }
            set
            {
                uint iMask = (uint)1 << 18;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag19
        {
            get { return ((UserShaderFlags & ((uint)1 << 19)) != 0); }
            set
            {
                uint iMask = (uint)1 << 19;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag20
        {
            get { return ((UserShaderFlags & ((uint)1 << 20)) != 0); }
            set
            {
                uint iMask = (uint)1 << 20;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag21
        {
            get { return ((UserShaderFlags & ((uint)1 << 21)) != 0); }
            set
            {
                uint iMask = (uint)1 << 21;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag22
        {
            get { return ((UserShaderFlags & ((uint)1 << 22)) != 0); }
            set
            {
                uint iMask = (uint)1 << 22;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag23
        {
            get { return ((UserShaderFlags & ((uint)1 << 23)) != 0); }
            set
            {
                uint iMask = (uint)1 << 23;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag24
        {
            get { return ((UserShaderFlags & ((uint)1 << 24)) != 0); }
            set
            {
                uint iMask = (uint)1 << 24;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag25
        {
            get { return ((UserShaderFlags & ((uint)1 << 25)) != 0); }
            set
            {
                uint iMask = (uint)1 << 25;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag26
        {
            get { return ((UserShaderFlags & ((uint)1 << 26)) != 0); }
            set
            {
                uint iMask = (uint)1 << 26;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag27
        {
            get { return ((UserShaderFlags & ((uint)1 << 27)) != 0); }
            set
            {
                uint iMask = (uint)1 << 27;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag28
        {
            get { return ((UserShaderFlags & ((uint)1 << 28)) != 0); }
            set
            {
                uint iMask = (uint)1 << 28;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag29
        {
            get { return ((UserShaderFlags & ((uint)1 << 29)) != 0); }
            set
            {
                uint iMask = (uint)1 << 29;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag30
        {
            get { return ((UserShaderFlags & ((uint)1 << 30)) != 0); }
            set
            {
                uint iMask = (uint)1 << 30;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag31
        {
            get { return ((UserShaderFlags & ((uint)1 << 31)) != 0); }
            set
            {
                uint iMask = (uint)1 << 31;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        #endregion

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public float[] UserShaderParams { get; set; }

        #endregion
    }
}
