﻿// ========================================================================
// <copyright file="MathUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;

namespace App.Utility
{
    /// <summary>
    /// スコープ用ストップウォッチ
    /// </summary>
    public class ScopedStopwatch : IDisposable
    {
        private string message = String.Empty;
        private Stopwatch stopwatch = new Stopwatch();
        private ConsoleTypes consoleType = ConsoleTypes.DEBUG_WINDOW;

        /// <summary>
        /// 出力先のタイプ
        /// </summary>
        public enum ConsoleTypes
        {
            /// <summary>
            /// デバッグコンソールウィンドウへ出力
            /// </summary>
            DEBUG_WINDOW,

            /// <summary>
            /// デバッグとして出力ウィンドウへ出力
            /// </summary>
            DEBUG,

            /// <summary>
            /// トレースとして出力ウィンドウへ出力
            /// </summary>
            TRACE,
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ScopedStopwatch(string message)
            : this(message, ConsoleTypes.DEBUG_WINDOW)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ScopedStopwatch(string message, ConsoleTypes type)
        {
            this.message = message;
            this.consoleType = type;
            this.stopwatch.Start();
        }

        /// <summary>
        /// 破棄
        /// </summary>
        public void Dispose()
        {
            this.stopwatch.Stop();

            string msg = String.Format(
                " - Time: {0} ms {1}", this.stopwatch.ElapsedMilliseconds, this.message);
            switch (this.consoleType)
            {
                default:
                case ConsoleTypes.DEBUG_WINDOW:
                    DebugConsole.WriteLine(msg);
                    break;
                case ConsoleTypes.DEBUG:
                    Debug.WriteLine(msg);
                    break;
                case ConsoleTypes.TRACE:
                    Trace.WriteLine(msg);
                    break;
            }
        }
    }
}
