﻿// ========================================================================
// <copyright file="DocumentIO.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

//#define LoadAsynchronously

// Turn this switch on to convert old eset files
// that texture pattern table was still an array.
#define CONVERT_OLD_ESET_FILE

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Xml.Serialization;
//using App.Command;
//using App.Controls;
using App.Data;
//using NintendoWare.Tga;
using NintendoWare.ToolDevelopmentKit.Xml;
//using nw.g3d.iflib;
//using nw.g3d.nw4f_3dif;
using NWCore.DataModel;
using NWCore.Serializer;
using NWCore.Viewer;
using App.Utility;
using NWCore;

namespace App.IO
{
    /// <summary>
    /// Command_FileNewProject用コマンド引数です。
    /// </summary>
    public class FileNewProjectArgs
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FileNewProjectArgs(bool isRequireInputName)
        {
            this.IsRequireInputName = isRequireInputName;
        }

        /// <summary>
        /// 名前入力を要求します。
        /// </summary>
        public bool IsRequireInputName { get; set; }
    }


    /// <summary>
    /// ドキュメント入出力。
    /// </summary>
    public static class DocumentIO
    {
        private static bool s_requireInputName = true;
        private const int DEF_MEMRY_BUFFER = (1024 * 1024 * 10);
        const string FILEEXT_FTX = ".ftx";
        const string FILEEXT_TGA = ".tga";
        public const int s_MaxFileNameLength = 128;

        /// <summary>
        /// クリップボードID
        /// </summary>
        public const string CLIPBOARDID_PROJECTNODE = "EM4F_ProjectTree_DocumentClipboardData";

        /// <summary>
        /// 名前入力を求めるようにしますか？
        /// </summary>
        /// <remarks>
        /// =true..新規生成で名前入力を求めます。
        /// =false..新規生成でデフォルト名のまま生成します。
        /// </remarks>
        public static bool IsRequireInputName
        {
            get { return s_requireInputName; }
            set { s_requireInputName = value; }
        }

        #region Name validators

        /// <summary>
        /// Validates or not an EmitterSet name
        /// </summary>
        /// <param name="dialog">Name input dialog</param>
        /// <param name="name">Given name</param>
        /// <param name="customValidationData">Project</param>
        /// <returns></returns>
        /*
        static bool EmitterSetNameValidator( NameInputDialog dialog,
                                             string name,
                                             object customValidationData )
        {
            string oldName = (customValidationData as object[])[0] as string;
            if ( oldName!=null && oldName==name )
            {
                return true;
            }

            EffectProjectDocument project = (customValidationData as object[])[1] as EffectProjectDocument;
            bool success = ( project.FindByName(name)==null );

            if (!success)
            {
                string message = string.Format(res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_ESET_NAME_ALREADY_USED, name);
                ThreadSafeMsgBox.Show(dialog,
                                      message,
                                      res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_INVALID_NAME,
                                      System.Windows.Forms.MessageBoxButtons.OK,
                                      System.Windows.Forms.MessageBoxIcon.Exclamation);
            }

            return success;
        }
        */

        /// <summary>
        /// Validates or not an Emitter name
        /// </summary>
        /// <param name="dialog">Name input dialog</param>
        /// <param name="name">Given name</param>
        /// <param name="customValidationData">Parent emitter set</param>
        /// <returns></returns>
        /*
        static bool EmitterNameValidator( NameInputDialog dialog,
                                          string name,
                                          object customValidationData )
        {
            string oldName = (customValidationData as object[])[0] as string;
            if ( oldName!=null && oldName==name )
            {
                return true;
            }

            EmitterSetDocument emitterSetDoc = (customValidationData as object[])[1] as EmitterSetDocument;
            bool success = ( emitterSetDoc.FindChild(name)==null );

            if (!success)
            {
                string message = string.Format(res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_E_NAME_ALREADY_USED, name);
                ThreadSafeMsgBox.Show(dialog,
                                      message,
                                      res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_INVALID_NAME,
                                      System.Windows.Forms.MessageBoxButtons.OK,
                                      System.Windows.Forms.MessageBoxIcon.Exclamation);
            }

            return success;
        }
        */

        /// <summary>
        /// Validates or not an Game Settings name
        /// </summary>
        /// <param name="dialog">Name input dialog</param>
        /// <param name="name">Given name</param>
        /// <param name="customValidationData">Parent emitter set</param>
        /// <returns></returns>
        /*
        static bool GameSettingsNameValidator( NameInputDialog dialog,
                                               string name,
                                               object customValidationData )
        {
            string oldName = (customValidationData as object[])[0] as string;
            if ( oldName!=null && oldName==name )
            {
                return true;
            }

            EmitterSetDocument emitterSetDoc = (customValidationData as object[])[1] as EmitterSetDocument;
            bool success = ( emitterSetDoc.FindGameSettingsDocument(name)==null );

            if (!success)
            {
                string message = string.Format(res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_GS_NAME_ALREADY_USED, name);
                ThreadSafeMsgBox.Show(dialog,
                                      message,
                                      res.Strings.MSGBOX_CAPTION_NAME_INPUT_DIALOG_INVALID_NAME,
                                      System.Windows.Forms.MessageBoxButtons.OK,
                                      System.Windows.Forms.MessageBoxIcon.Exclamation);
            }

            return success;
        }
        */

        #endregion

        #region 新規作成

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileNewProject(MenuCommandArgs args)
        {
            if (args.RequireExecute)
            {
                // 名前を入力します。
                string defName = DocumentCreator.CreateDefaultProjectName();

                bool require = IsRequireInputName;
                if (args.CommandData != null)
                {
                    FileNewProjectArgs prjArgs = args.CommandData as FileNewProjectArgs;
                    if (require == true)
                    {
                        require = prjArgs.IsRequireInputName;
                    }
                }

                if (require == true)
                {
                    using (NameInputDialog dialog =
                        new NameInputDialog(defName, res.Strings.NAMEINPUT_PROJECT))
                    {
                        dialog.ShowDialog();
                        if (dialog.DialogResult == DialogResult.OK)
                        {
                            defName = dialog.InputedName;
                        }
                        else
                        {
                            // そのままキャンセルします。
                            return;
                        }
                    }
                }

                // fire a command to create new emitter set document
                TheApp.CommandManager.Execute(new EffectProjectCreationCommand(defName));

                if (ProjectManager.ActiveProject != null)
                {
                    // ツリーを更新させます。
                    ProjectManager.ActiveProject.BuildTreeNodes();
                }

                MCSManager.WaitTransmission();
            }
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileNewEmitterSet(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled =
                    (ProjectManager.Projects.Count > 0 &&
                        ProjectManager.ActiveProject != null &&
                        ProjectManager.ActiveProject.Count < TheApp.MAX_EMITTERSET_COUNT);
                return;
            }

            // 追加先のプロジェクトです。
            IProjectDocument project = ProjectManager.ActiveProject;
            if (project == null)
            {
                return;
            }

            EmitterSetDocument emitterSetDoc = null;
            bool saveBuildFlag = project.EnableBuildTreeNodes;
            project.EnableBuildTreeNodes = false;
            {
                string esetName;
                if ( args.CommandData is string )
                {
                    // Use the assigned emitter set name.
                    esetName = args.CommandData as string;

                    if ( project.FindChild(esetName)!=null )
                    {
                        ThreadSafeMsgBox.Show( TheApp.MainFrame,
                                               res.Strings.WARNING_CREATE_ESET_NAME_CONFLICT,
                                               res.Strings.WARNING_CAPTION,
                                               MessageBoxButtons.OK,
                                               MessageBoxIcon.Exclamation );
                        return;
                    }
                }
                else
                {
                    // デフォルト名を決める
                    esetName = DocumentCreator.CreateDefaultEmitterSetName( project );

                    // 名前入力ダイアログで入力するようにします。
                    if (IsRequireInputName == true)
                    {
                        using (NameInputDialog dialog =
                            new NameInputDialog(esetName, res.Strings.NAMEINPUT_EMITTERSET))
                        {
                            dialog.ValidateNameWhenCustomToo = true;
                            dialog.CustomValidationData = new object[2] { null, project };
                            dialog.InputValidator = new NameInputDialog.NameValidator(EmitterSetNameValidator);

                            dialog.ShowDialog();
                            if (dialog.DialogResult == DialogResult.OK)
                            {
                                esetName = dialog.InputedName;
                            }
                            else
                            {
                                // そのままキャンセルします。
                                project.EnableBuildTreeNodes = saveBuildFlag;
                                return;
                            }
                        }
                    }
                }

                using ( MCSDisableBlock block = new MCSDisableBlock() )
                {
                    // Create the emitter set.
                    {
                        // 新しいエフェクトのパスを決める
                        string filePath = esetName + DocumentConstants.DotEset;

                        // fire a command to create new emitter set document
                        var command =
                        new EmitterSetDocumentCreationCommand( esetName, filePath, project );
                        TheApp.CommandManager.Execute( command );
                        emitterSetDoc = command.OriginalData as EmitterSetDocument;
                    }

                    // Add one emitter and game settings document to the emitter set.
                    if ( emitterSetDoc != null )
                    {
                        string defName = DocumentCreator.CreateDefaultEmitterName( emitterSetDoc );
                        TheApp.CommandManager.ExecuteIgnoreTargetDocument(
                            new EmitterDocumentCreationCommand( defName, emitterSetDoc ) );

                        defName = DocumentCreator.CreateDefaultGameSettingsDocName( emitterSetDoc );
                        TheApp.CommandManager.ExecuteIgnoreTargetDocument(
                            new GameSettingsDocumentCreationCommand( defName, emitterSetDoc ) );
                    }
                }

                Viewer.Message.SendUtility.SendEmitterSetBinary(emitterSetDoc);
            }
            project.EnableBuildTreeNodes = saveBuildFlag;
            project.BuildTreeNodes();

            ProjectManager.ActiveDocument = emitterSetDoc;
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileNewEmitter(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                bool bEnable = false;
                IDocument document = ProjectManager.ActiveDocument;
                if ( document==null )
                {
                    bEnable = false;
                }
                else if ( document.ObjectID==GuiObjectID.EmitterSet )
                {
                    bEnable = (document.Count < TheApp.MAX_EMITTER_COUNT);
                }
                else if ( document.ObjectID==GuiObjectID.Emitter ||
                          document.ObjectID==GuiObjectID.GameSettings )
                {
                    IDocument parent = document.OwnerDocument;
                    bEnable = (parent.Count < TheApp.MAX_EMITTER_COUNT);
                }
                args.CommandUI.Enabled = bEnable;
                return;
            }
            else
            {
                IDocument document = ProjectManager.ActiveDocument;
                if ( document!=null &&
                     document.ObjectID==GuiObjectID.EmitterSet )
                {
                    var emitterSetDoc = document as EmitterSetDocument;
                    Command_FileNewEmitter( args, emitterSetDoc );
                }
                else if ( document!=null &&
                          document is IEmitterDocument )
                {
                    var emitterSetDoc = (document as IEmitterDocument).EmitterSetDocument;
                    Command_FileNewEmitter( args, emitterSetDoc as EmitterSetDocument );
                }
                else if ( document!=null &&
                          document is GameSettingsDocument )
                {
                    var emitterSetDoc = (document as GameSettingsDocument).EmitterSetDocument;
                    Command_FileNewEmitter( args, emitterSetDoc as EmitterSetDocument );
                }
            }
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileNewEmitter(
            MenuCommandArgs args, EmitterSetDocument emitterSetDocument)
        {
            if (args.RequireUpdate)
            {
                bool bEnable = false;
                IDocument document = ProjectManager.ActiveDocument;
                if ( (document!=null) &&
                     (document.ObjectID==GuiObjectID.EmitterSet ||
                      document.ObjectID==GuiObjectID.Emitter ||
                      document.ObjectID==GuiObjectID.GameSettings) )
                {
                    bEnable = (document.Count < TheApp.MAX_EMITTER_COUNT);
                }
                args.CommandUI.Enabled = bEnable;
                return;
            }

            // 追加先のプロジェクトです。
            IProjectDocument project = ProjectManager.ActiveProject;
            if (project == null)
            {
                return;
            }

            // 追加先のエフェクト・エミッタです。
            EmitterSetDocument emitterSetDoc = emitterSetDocument;
            if (emitterSetDoc == null)
            {
                emitterSetDoc = project.ActiveDocument as EmitterSetDocument;
            }

            if (emitterSetDoc == null)
            {
                // 不正なノード！
                return;
            }

            // 名前入力ダイアログで入力します。
            string defName = DocumentCreator.CreateDefaultEmitterName( emitterSetDoc );
            if (IsRequireInputName == true)
            {
                using (NameInputDialog dialog =
                    new NameInputDialog(defName, res.Strings.NAMEINPUT_EMITTER))
                {
                    dialog.ValidateNameWhenCustomToo = true;
                    dialog.CustomValidationData = new object[2] { null, emitterSetDoc };
                    dialog.InputValidator = new NameInputDialog.NameValidator( EmitterNameValidator );

                    dialog.ShowDialog();
                    if (dialog.DialogResult == DialogResult.OK)
                    {
                        defName = dialog.InputedName;
                        // fire a command to create new emitter set document
                        TheApp.CommandManager.Execute(
                            new EmitterDocumentCreationCommand(defName, emitterSetDoc));
                    }
                }
            }

            EmitterDocument newDoc = emitterSetDoc.FindEmitterDocument( defName );
            if ( newDoc!=null )
                ProjectManager.ActiveDocument = newDoc;
            else
                ProjectManager.ActiveDocument = emitterSetDoc;
        }
        */


        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileNewChildren(MenuCommandArgs args, EmitterDocument emitter)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled =
                    (ProjectManager.Projects.Count > 0 &&
                        ProjectManager.ActiveProject != null);
                return;
            }

            // 追加先のプロジェクトです。
            IProjectDocument project = ProjectManager.ActiveProject;
            if (project == null || emitter == null)
            {
                return;
            }

            // コマンド
            emitter.EmitterData.Type = EmitterType.Complex;
            CommandSet commandSet = new CommandSet();
            commandSet.Add(new ChildDocumentCreationCommand(emitter));
            commandSet.Add(new FieldDocumentCreationCommand(emitter));
            commandSet.Add(new FluctuationDocumentCreationCommand(emitter));

            // fire a command to create all 3 docs
            TheApp.CommandManager.Execute(
                commandSet,
                res.Strings.COMMAND_DESC_CONVERT_COMPLEX_EMITTER,
                emitter.DisplayDataPath);
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileNewGameSettings( MenuCommandArgs args )
        {
            if ( args.RequireUpdate==true )
            {
                bool bEnable = false;

                IDocument document = ProjectManager.ActiveDocument;
                if ( (document!=null) &&
                     (document.ObjectID==GuiObjectID.EmitterSet ||
                      document.ObjectID==GuiObjectID.Emitter ||
                      document.ObjectID==GuiObjectID.GameSettings) )
                {
                    bEnable = true;
                }

                args.CommandUI.Enabled = bEnable;
                return;
            }
            else
            {
                IDocument document = ProjectManager.ActiveDocument;
                if ( document!=null &&
                     document.ObjectID==GuiObjectID.EmitterSet )
                {
                    var emitterSetDoc = document as EmitterSetDocument;
                    Command_FileNewGameSettings( args, emitterSetDoc );
                }
                else if ( document!=null &&
                          document is IEmitterDocument )
                {
                    var emitterSetDoc = (document as IEmitterDocument).EmitterSetDocument;
                    Command_FileNewGameSettings( args, emitterSetDoc as EmitterSetDocument );
                }
                else if ( document!=null &&
                          document is GameSettingsDocument )
                {
                    var emitterSetDoc = (document as GameSettingsDocument).EmitterSetDocument;
                    Command_FileNewGameSettings( args, emitterSetDoc as EmitterSetDocument );
                }
            }
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileNewGameSettings( MenuCommandArgs args,
                                                        EmitterSetDocument emitterSetDocument )
        {
            if ( args.RequireUpdate==true )
            {
                bool bEnable = false;

                IDocument document = ProjectManager.ActiveDocument;
                if ( (document!=null) &&
                     (document.ObjectID==GuiObjectID.EmitterSet ||
                      document.ObjectID==GuiObjectID.Emitter ||
                      document.ObjectID==GuiObjectID.GameSettings) )
                {
                    bEnable = true;
                }

                args.CommandUI.Enabled = bEnable;
                return;
            }

            // 追加先のプロジェクトです。
            IProjectDocument project = ProjectManager.ActiveProject;
            if ( project==null )
                return;

            // 追加先のエフェクト・エミッタです。
            EmitterSetDocument emitterSetDoc = emitterSetDocument;
            if ( emitterSetDoc==null )
            {
                emitterSetDoc = project.ActiveDocument as EmitterSetDocument;
            }

            if ( emitterSetDoc==null )
            {
                // 不正なノード！
                return;
            }

            // 名前入力ダイアログで入力します。
            string defName = DocumentCreator.CreateDefaultGameSettingsDocName( emitterSetDoc );
            if ( IsRequireInputName==true )
            {
                using ( NameInputDialog dialog =
                    new NameInputDialog(defName, res.Strings.NAMEINPUT_GAMESETTINGS) )
                {
                    dialog.ValidateNameWhenCustomToo = true;
                    dialog.CustomValidationData = new object[2] { null, emitterSetDoc };
                    dialog.InputValidator = new NameInputDialog.NameValidator(GameSettingsNameValidator);

                    dialog.ShowDialog();
                    if (　dialog.DialogResult==DialogResult.OK　)
                    {
                        defName = dialog.InputedName;

                        if (defName.Length > DocumentIO.s_MaxFileNameLength)
                        {
                            DocumentIO.ErrorFileNameLengthCreate(defName);
                            return;
                        }

                        GameSettingsDocumentCreationCommand cmd =
                            new GameSettingsDocumentCreationCommand( defName, emitterSetDoc );

                        // fire a command to create new game settings document
                        TheApp.CommandManager.Execute( cmd );
                    }
                }
            }

            GameSettingsDocument newDoc = emitterSetDoc.FindGameSettingsDocument( defName );
            if ( newDoc!=null )
                ProjectManager.ActiveDocument = newDoc;
            else
                ProjectManager.ActiveDocument = emitterSetDoc;
        }
        */

        /// <summary>
        /// 削除メニューが選択されたときに発生します。
        /// </summary>
        /// <param name="sender">送信者です。</param>
        /// <param name="removeNodes">削除項目ドキュメントです。</param>
        /// <see cref="ToolStripItemClickedEventArgs"/>
        /// <returns>=true..削除成功</returns>
        public static bool Command_RemoveDocuments(object sender, IDocument[] removeNodes)
        {
            bool result = false;
            foreach (var removeNode in removeNodes)
            {
                switch (removeNode.ObjectID)
                {
                    case GuiObjectID.EmitterSet:
                        {
                            /*
                             *  エフェクトの削除
                             */
                            var parent = removeNode.OwnerDocument as IProjectDocument;
                            if (parent != null)
                            {
                                // EmitterSetDocumentを閉じる
                                var dialog = new DocumentCloser();
                                dialog.Close(removeNode);

                                // アクティブドキュメントを初期化します。
                                if (ProjectManager.ActiveProject.Contains(
                                    removeNode as IEmitterSetDocument) == false)
                                {
                                    ProjectManager.ActiveDocument = null;
                                    result = true;
                                }
                            }
                        }
                        break;

                    case GuiObjectID.Emitter:
                        {
                            /*
                             *  パーティクルセットの削除
                             */
                            var parent = removeNode.OwnerDocument as EmitterSetDocument;
                            if ( parent!=null &&
                                 parent.Count>1 )
                            {
                                /*
                                TheApp.CommandManager.Execute(
                                    new EmitterDocumentRemoveCommand(
                                        parent, (EmitterDocument)removeNode));
                                */

                                // アクティブドキュメントを初期化します。
                                if (parent.FindEmitterDocument(removeNode.Name) == null)
                                {
                                    ProjectManager.ActiveDocument = null;
                                    result = true;
                                }
                            }
                        }
                        break;

                    case GuiObjectID.GameSettings:
                        {
                            var parent = removeNode.OwnerDocument as EmitterSetDocument;
                            GameSettingsDocument doc = removeNode as GameSettingsDocument;
                            if ( parent!=null &&
                                 doc!=null &&
                                 parent.GetNumGameSettingsDocuments()>1 )
                            {
                                /*
                                GameSettingsDocumentRemoveCommand cmd =
                                    new GameSettingsDocumentRemoveCommand( parent, doc );
                                TheApp.CommandManager.Execute( cmd );
                                */

                                // アクティブドキュメントを初期化します。
                                if (parent.FindGameSettingsDocument(removeNode.Name) == null)
                                {
                                    ProjectManager.ActiveDocument = null;
                                    result = true;
                                }
                            }
                        }
                        break;

                    default:
                        Debug.Assert(false, "削除対象のノードではありません。");
                        break;
                }
            }

            //TheApp.MainFrame.RequestRebuildNodes();

            return result;
        }

        #endregion

        #region 開く

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileOpen(MenuCommandArgs args)
        {
            if (args.RequireUpdate == true)
            {
                args.CommandUI.Enabled =
                    (ProjectManager.ActiveProject.Count < TheApp.MAX_EMITTERSET_COUNT);
                return;
            }

            if (args.RequireExecute)
            {
                DocumentsOpener opener = new DocumentsOpener();
#if LoadAsynchronously
                ThreadingExtensions.FireAndForget(() => opener.Open());
#else
                opener.Open();
#endif
            }
        }
        */

        /*
        public static void Command_FileRecentFilesOpen(MenuCommandArgs args)
        {
            if (args.RequireUpdate == true)
            {
                args.CommandUI.Enabled =
                    (ProjectManager.ActiveProject.Count < TheApp.MAX_EMITTERSET_COUNT);
                return;
            }

            if (args.RequireExecute)
            {
                var opener = new DocumentsOpener();
                opener.Open(new [] { args.CommandData as string });
            }
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_TemplateOpen( MenuCommandArgs args )
        {
            if (args.RequireUpdate == true)
            {
                args.CommandUI.Enabled =
                    (ProjectManager.ActiveProject.Count < TheApp.MAX_EMITTERSET_COUNT);
                return;
            }

            if ( args.RequireExecute )
            {
                DocumentsOpener opener = new DocumentsOpener();

                #if LoadAsynchronously
                ThreadingExtensions.FireAndForget(() => opener.OpenTemplateFile());
                #else
                opener.OpenTemplateFile();
                #endif
            }
        }
        */

        /// <summary>
        /// load emitter set file
        /// </summary>
        /// <param name="filepath">path to load</param>
        /// <param name="project">The project to add the emitter set to.</param>
        /// <param name="loadTexture">=true(DEF)..テクスチャも読む</param>
        public static EmitterSetDocument LoadEmitterSetFile( string filePath,
                                                             IProjectDocument project,
                                                             bool loadTexture = true,
                                                             bool bSilent = false )
        {
            if ( ProjectManager.ActiveProject!=null &&
                 ProjectManager.ActiveProject.Count>=TheApp.MAX_EMITTERSET_COUNT )
            {
                // EmitterSetの最大数に達しているとき
                return null;
            }

            if ( File.Exists(filePath)==false )
            {
                TheApp.OutputLogMsg( LogLevels.Error,
                                     res.Strings.CONSOLE_FILE_NOT_FOUND_MSG,
                                     filePath );
                return null;
            }

            EmitterSetDataXml xml = null;
            try
            {
                byte[] buffer = File.ReadAllBytes( filePath );

                #if CONVERT_OLD_ESET_FILE
                // 旧データのファイルなあらば、新しいデータに変換します。
                using ( MemoryStream stream = new MemoryStream(buffer) )
                {
                    using ( StreamReader reader = new StreamReader(stream) )
                    {
                        string textContent = reader.ReadToEnd();

                        Regex expReplace = new Regex( @"<\s*UI_texPatTbl\[(\d+)\]>" );

                        string replacedContent =
                            expReplace.Replace( textContent, "<UI_texPatTbl_$1>" );

                        buffer =
                            System.Text.Encoding.UTF8.GetBytes( replacedContent );
                    }
                }
                #endif

                using ( MemoryStream stream = new MemoryStream(buffer) )
                {
                    // Load and update the data
                    UpdaterErrorTypes err;
                    object dataModel =
                        TheApp.UpdaterManager.LoadDataModel( DocumentConstants.DotEset,
                                                             stream,
                                                             out err );
                    // 読み込みエラー
                    if ( err!=UpdaterErrorTypes.Succeed )
                    {
                        TheApp.OutputLogMsg( LogLevels.Warning,
                                             res.Strings.IO_LOADFILE_FAILED_MSG,
                                             filePath );
                    }
                    else
                    {
                        xml = dataModel as EmitterSetDataXml;
                        if ( xml==null )
                        {
                            TheApp.OutputLogMsg( LogLevels.Warning,
                                                 res.Strings.IO_LOADFILE_FAILED_MSG,
                                                 filePath );
                        }
                    }
                }
            }
            catch
            {
                TheApp.OutputLogMsg( LogLevels.Warning,
                                     res.Strings.IO_LOADFILE_FAILED_MSG,
                                     filePath );
            }

            if ( xml==null )
                return null;

            var emitterSetDocument = EmitterSetDocument.CreateFromDeserializer(project, xml);
            emitterSetDocument.SetFileLocation(Path.GetDirectoryName(filePath));
            string fileTitle = Path.GetFileNameWithoutExtension(filePath);
            emitterSetDocument.FileTitle = fileTitle;
            emitterSetDocument.Name = fileTitle;
            emitterSetDocument.OnFinishedLoading( loadTexture, bSilent );
            return emitterSetDocument;
        }

        /// <summary>        /// デシリアライズ：ストリームからＸＭＬファイルを読み込みます。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="filePath">読み込みファイルパスです。</param>
        /// <returns>デシリアライズ結果です。</returns>
        public static TType DeserializeFromFile<TType>(Stream reader)
            where TType : class
        {
            // 読み込み
            XmlSerializer xmlSerializer = XmlUtility.CreateCustomXmlSerializer<TType>();
            return xmlSerializer.Deserialize(reader) as TType;
        }


        #endregion

        #region 保存

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileSave(MenuCommandArgs args)
        {
            // Determine if the selected document can be saved
            IDocument doc = ProjectManager.ActiveDocument;

            if ( doc is GameSettingsDocument )
            {
                Command_FilePrevSave( args );
                return;
            }

            bool bCanSave = false;
            while ( doc!=null )
            {
                // We do not allow .work or .prev files to be saved through here
                if ( (doc.FileExt!=null) &&
                     (doc.FileExt.Length>0) &&
                     (doc is EffectProjectDocument)==false )
                {
                    bCanSave = true;
                    break;
                }

                doc = doc.OwnerDocument;
            }

            // Do not allow unchanged document to be saved
            if ( doc!=null &&
                 doc.Modified==false )
            {
                bCanSave = false;
            }

            // Update the flag to enable/disable the UI
            if ( args.RequireUpdate==true )
            {
                args.CommandUI.Enabled = bCanSave;
                return;
            }

            // Save the document.
            if ( bCanSave==true )
            {
                DocumentSaver saver = new DocumentSaver();
                saver.Save( doc, doc.FilePath );
            }
        }
        */
        //---------------------------------------------------------------------
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileSaveAs(MenuCommandArgs args)
        {
            // Determine if the selected document can be saved
            IDocument doc = ProjectManager.ActiveDocument;

            if ( doc is GameSettingsDocument )
            {
                Command_FilePrevSaveAs( args );
                return;
            }

            bool bCanSave = false;
            while ( doc!=null )
            {
                // We do not allow .work or .prev files to be saved through here
                if ( (doc.FileExt!=null) &&
                     (doc.FileExt.Length>0) &&
                     (doc is GameSettingsDocument)==false &&
                     (doc is EffectProjectDocument)==false )
                {
                    bCanSave = true;
                    break;
                }

                doc = doc.OwnerDocument;
            }

            // Update the flag to enable/disable the UI
            if ( args.RequireUpdate==true )
            {
                args.CommandUI.Enabled = bCanSave;
                return;
            }

            if ( bCanSave==true )
            {
                // Save the document.
                DocumentSaver saver = new DocumentSaver();
                saver.SaveAs( doc );
            }
            else
            {
                ThreadSafeMsgBox.Show( res.Strings.IO_SAVE_TARGET_NOT_FOUND_MSG,
                                       res.Strings.WARNING_CAPTION,
                                       System.Windows.Forms.MessageBoxButtons.OK,
                                       System.Windows.Forms.MessageBoxIcon.Warning );
            }
        }
        */

        //---------------------------------------------------------------------
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileSaveAllTo(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                args.CommandUI.Enabled = (ProjectManager.ActiveProject!=null  &&
                                          ProjectManager.ActiveProject.Count>0);
                return;
            }

            // 実行
            DocumentSaver saver = new DocumentSaver();
            saver.SaveAllTo( ProjectManager.ActiveProject );
        }
        */

        //---------------------------------------------------------------------
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileSaveAll(MenuCommandArgs args)
        {
            IProjectDocument project = ProjectManager.ActiveProject;

            // ドキュメントがロードされていれば有効
            bool bShouldSave = false;
            if ( project!=null &&
                 project.Count>0 )
            {
                bool bModified = false;
                foreach ( IDocument doc in project.SubDocuments )
                {
                    if ( doc!=null && doc.Modified==true )
                    {
                        bModified = true;
                        break;
                    }

                    // Are there game settings documents to save?
                    if ( Config.Data.Option.SavePreviewOnSaveAll )
                    {
                        EmitterSetDocument eset = doc as EmitterSetDocument;
                        if ( eset!=null )
                        {
                            foreach ( GameSettingsDocument childDoc in eset.GameSettingsDocuments )
                            {
                                if ( childDoc!=null && childDoc.Modified==true )
                                {
                                    bModified = true;
                                    break;
                                }
                            }

                            if ( bModified==true )
                                break;
                        }
                    }
                }

                bShouldSave = bModified;
            }

            if ( args.RequireUpdate==true )
            {
                args.CommandUI.Enabled = bShouldSave;
                return;
            }

            if ( bShouldSave==true )
            {
                DocumentSaver saver = new DocumentSaver();
                saver.SaveAll();
            }
        }
        */

        #endregion

        #region エディット環境読み込み・保存

        /// <summary>
        /// コマンドハンドラ：エディット環境ファイルの読み込み
        /// </summary>
        /*
        public static void Command_FileEnvLoad( MenuCommandArgs args )
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = ( ProjectManager.Projects.Count>0 &&
                                           ProjectManager.ActiveProject!=null );
                return;
            }

            // 環境ファイルを読み込み
            using ( var dialog = new OpenFileDialog() )
            {
                dialog.Filter = res.Strings.IO_FILEFILTER_CAFE_FENV;
                dialog.InitialDirectory = Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.FEnv );
                dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;
                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    Command_FileEnvLoad(ProjectManager.ActiveProject, dialog.FileName);
                    Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.FEnv,
                                                               dialog.FileName );
                }
            }
        }
        */

        /// <summary>
        /// コマンドハンドラ：エディット環境ファイルの読み込み
        /// </summary>
        public static bool Command_FileEnvLoad( IProjectDocument project,
                                                string xmlFilePath )
        {
            return false;

            /*
            // 現在の設定が破棄されることを警告を出す
            if ( ProjectManager.ActiveProject.Count>0 )
            {
                var result = ThreadSafeMsgBox.Show( res.Strings.WANING_LOAD_ENVFILE,
                                                    res.Strings.WARNING_CAPTION,
                                                    MessageBoxButtons.YesNo );
                if ( result!=DialogResult.Yes )
                {
                    // キャンセルしたメッセージを出します。
                    ThreadSafeMsgBox.Show( res.Strings.WANING_LOADENV_CANCEL,
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK );
                    return false;
                }
            }

            // 全てを閉じる
            Command_FileCloseAll(new Command.MenuCommandArgs(false, null, null));

            // FENVファイル読み込み
            EnvConfigDataXml xmlModel = null;
            try
            {
                xmlModel = XmlUtility.DeserializeFromFile<EnvConfigDataXml>(xmlFilePath);
            }
            catch (Exception ex)
            {
                DebugConsole.WriteLine(ex.Message);
                return false;
            }

            // Load .prev files
            GameSettingsDocument gameSettingsDoc  = null;
            List<string>         missingEsetFiles = new List<string>();
            foreach ( GameConfigDataXml data in xmlModel.PrevDataList )
            {
                if ( missingEsetFiles.IndexOf( data.EmitterSetPath.ToLower() )>=0 )
                    continue;

                EmitterSetDocument esetDoc =
                    GetEmitterSetForGameSettingsDocument( project,
                                                          string.Empty,
                                                          data.EmitterSetPath,
                                                          null );

                if ( esetDoc==null )
                {
                    missingEsetFiles.Add( data.EmitterSetPath.ToLower() );
                    ThreadSafeMsgBox.Show( StringResource.Get( "ERR_ESET_FILE_NOT_FOUND",
                                                               data.EmitterSetPath),
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation );
                    continue;
                }

                // Does the game settings document already exist?
                if ( esetDoc.FindGameSettingsDocument( data.Name )!=null )
                {
                    ThreadSafeMsgBox.Show( res.Strings.ERR_PREV_FILE_ALREADY_EXISTS,
                                           res.Strings.ERROR_LOAD_PREV_FILE_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation );
                    return false;
                }

                // Load the game settings document.
                gameSettingsDoc = LoadGameSettingsDocument( data, esetDoc, null );
                gameSettingsDoc.SetFileLocation( esetDoc.FileLocation );

                // Make sure the UI definition has been loaded.
                TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

                // Load the game settings document.
                gameSettingsDoc.LoadFromDeserializer( data );

                // Refresh the page content by showing the first property page of the game settings panel.
                TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );
            }

            // ゲーム設定の反映
            project.EnvConfigData.Set( xmlModel.ConfigData );

            // Make sure the UI definition has been loaded.
            TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.PreviewModelPropertyPanel );

            // Load user functions
            foreach ( PreviewModelDocument modelDoc in project.PreviewDocument.PreviewModelDocuments )
            {
                modelDoc.LoadFromDeserializer( xmlModel );
                modelDoc.UpdateAnimation();
            }

            // Make sure the UI definition has been loaded.
            TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.PreviewModelPropertyPanel );

            // Set the file location to the project.
            project.SetFileLocation( Path.GetDirectoryName(xmlFilePath) );
            project.FileTitle = Path.GetFileNameWithoutExtension( xmlFilePath );

            // ツリー更新
            project.BuildTreeNodes();

            // コマンド履歴をクリアします。
            TheApp.CommandManager.ClearAll();

            App.PropertyEdit.HistoryDialog.UpdateForm();

            // 全情報を送信します。
            Viewer.Message.SendUtility.SendAll();
            return true;
            */
        }

        /// <summary>
        /// コマンドハンドラ：エディット環境ファイルの書き込み
        /// </summary>
        /*
        public static void Command_FileEnvSave(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = ( ProjectManager.Projects.Count>0 &&
                                           ProjectManager.ActiveProject!=null );
                return;
            }

            IProjectDocument project = ProjectManager.ActiveProject;
            if ( project==null )
                return;

            // 環境ファイルを保存します。
            var saver = new DocumentSaver();
            saver.Save( project, project.FilePath );
        }
        */

        /// <summary>
        /// コマンドハンドラ：エディット環境ファイルの書き込み
        /// </summary>
        /*
        public static void Command_FileEnvSaveAs( MenuCommandArgs args )
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = ( ProjectManager.Projects.Count>0 &&
                                           ProjectManager.ActiveProject!=null );
                return;
            }

            IProjectDocument project = ProjectManager.ActiveProject;
            if ( project==null )
                return;

            // 環境ファイルを保存します。
            var saver = new DocumentSaver();
            saver.SaveAs( ProjectManager.ActiveProject );
        }
        */


        #endregion

        #region プレビュー読み込み・保存

        /// <summary>
        /// コマンドハンドラ：プレビューファイルの読み込み
        /// </summary>
        /*
        public static void Command_FilePrevLoad( MenuCommandArgs args )
        {
            if ( args.RequireUpdate==true )
            {
                args.CommandUI.Enabled = ( ProjectManager.Projects.Count>0 &&
                                           ProjectManager.ActiveProject!=null );
                return;
            }

            // プレビューファイルを読み込み
            using ( OpenFileDialog dialog = new OpenFileDialog() )
            {
                dialog.Filter = res.Strings.IO_FILEFILTER_CAFE_PREV;
                dialog.InitialDirectory = Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Prev );
                dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;
                if ( dialog.ShowDialog()==DialogResult.OK )
                {
                    Command_FilePrevLoad( ProjectManager.ActiveProject,
                                          dialog.FileName );
                    Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Prev,
                                                               dialog.FileName );
                }
            }
        }
        */


        /// <summary>
        /// コマンドハンドラ：プレビューファイルの上書き保存
        /// </summary>
        /*
        public static void Command_FilePrevSave( MenuCommandArgs args )
        {
            if ( args.RequireUpdate==true )
            {
                args.CommandUI.Enabled =
                    ( ProjectManager.ActiveDocument is GameSettingsDocument &&
                      ProjectManager.ActiveDocument.Modified==true );
                return;
            }

            GameSettingsDocument doc = ProjectManager.ActiveDocument as GameSettingsDocument;
            if ( doc==null )
                return;

            DocumentSaver saver = new DocumentSaver();
            saver.Save( doc, doc.FilePath );
        }
        */


        /// <summary>
        /// コマンドハンドラ：プレビューファイルの名前を付けて保存
        /// </summary>
        /*
        public static void Command_FilePrevSaveAs( MenuCommandArgs args )
        {
            if ( args.RequireUpdate==true )
            {
                args.CommandUI.Enabled = (ProjectManager.ActiveDocument is GameSettingsDocument);
                return;
            }

            GameSettingsDocument doc = ProjectManager.ActiveDocument as GameSettingsDocument;
            if ( doc==null )
                return;

            DocumentSaver saver = new DocumentSaver();
            saver.SaveAs( doc );
        }
        */


        /// <summary>
        /// コマンドハンドラ：プレビューファイルの読み込み
        /// </summary>
        public static bool Command_FilePrevLoad( IProjectDocument project,
                                                 string filePath,
                                                 List<string> loadedEmitterSets = null )
        {
            if ( (project is EffectProjectDocument)==false )
                return false;

            GameSettingsDocument gameSettingsDoc = null;

            string dirPath = Path.GetDirectoryName( filePath );

            // .prevファイル読み込み
            GameConfigDataXml content = null;
            try
            {
                content = XmlUtility.DeserializeFromFile<GameConfigDataXml>( filePath );

                EmitterSetDocument esetDoc = null;
                using ( MCSDisableBlock block = new MCSDisableBlock() )
                {
                    esetDoc =
                        GetEmitterSetForGameSettingsDocument( project,
                                                              dirPath,
                                                              content.EmitterSetPath,
                                                              loadedEmitterSets );
                }
                if ( esetDoc==null )
                {
                    /*
                    ThreadSafeMsgBox.Show( StringResource.Get( "ERR_ESET_FILE_NOT_FOUND",
                                                               content.EmitterSetPath ),
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation );
                    */
                    return false;
                }

                // Does the game settings document already exist?
                if ( esetDoc.FindGameSettingsDocument( content.Name )!=null )
                {
                    /*
                    ThreadSafeMsgBox.Show( res.Strings.ERR_PREV_FILE_ALREADY_EXISTS,
                                           res.Strings.ERROR_LOAD_PREV_FILE_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation );
                    */
                    return false;
                }

                // Load the game settings document.
                gameSettingsDoc = LoadGameSettingsDocument( content, esetDoc, null );
                gameSettingsDoc.SetFileLocation( dirPath );
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine(ex.Message);
                return false;
            }

            // ツリー更新
            project.BuildTreeNodes();

            // Make sure the UI definition has been loaded.
            //MainFrame.SynchronizationContext.Send(() =>
            //    TheApp.UIManager.UpdatePropertyPanel(PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel));

            // Load the game settings document.
            gameSettingsDoc.LoadFromDeserializer( content );

            // Refresh the page content by showing the first property page of the game settings panel.
            //MainFrame.SynchronizationContext.Send(() =>
            //    TheApp.UIManager.UpdatePropertyPanel(PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel));

            // Send the emitter set to the viewer
            //Viewer.Message.SendUtility.SendEmitterSetBinary( gameSettingsDoc.EmitterSetDocument );

            return true;
        }


        /// <summary>
        /// コマンドハンドラ：プレビューファイルの追加読み込み
        /// </summary>
        public static bool Command_FilePrevLoad( IProjectDocument project,
                                                 EmitterSetDocument owner )
        {
            if ( (project is EffectProjectDocument)==false )
                return false;

            if ( owner==null )
                return false;

            System.Windows.Forms.OpenFileDialog dialog =
                new System.Windows.Forms.OpenFileDialog();

            dialog.Filter             = res.Strings.IO_FILEFILTER_CAFE_PREV;
            dialog.Multiselect        = true;
            dialog.InitialDirectory   = Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Prev );
            dialog.FilterIndex        = 0;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            /*
            if ( dialog.ShowDialog(TheApp.MainFrame)!=System.Windows.Forms.DialogResult.OK )
                return false;
            */

            string filePath = dialog.FileName;

            Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Prev,
                                                       dialog.FileName );

            GameSettingsDocument gameSettingsDoc = null;

            string dirPath = Path.GetDirectoryName( filePath );

            string name = Path.GetFileNameWithoutExtension(filePath);
            if (name.Length > DocumentIO.s_MaxFileNameLength)
            {
                DocumentIO.ErrorFileNameLengthLoad(name);
                return false;
            }

            // .prevファイル読み込み
            GameConfigDataXml content = null;
            try
            {
                content = XmlUtility.DeserializeFromFile<GameConfigDataXml>( filePath );

                // Does the game settings document already exist?
                if ( owner.FindGameSettingsDocument( content.Name )!=null )
                {
                    /*
                    ThreadSafeMsgBox.Show( res.Strings.ERR_PREV_FILE_ALREADY_EXISTS,
                                           res.Strings.ERROR_LOAD_PREV_FILE_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation );
                    */
                    return false;
                }

                // Load the game settings document.
                gameSettingsDoc = LoadGameSettingsDocument( content, owner, null );
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine(ex.Message);
                return false;
            }

            // ツリー更新
            project.BuildTreeNodes();

            // Make sure the UI definition has been loaded.
            //TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            // Load the game settings document.
            gameSettingsDoc.LoadFromDeserializer( content );

            // Refresh the page content by showing the first property page of the game settings panel.
            //TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            // Send the emitter set to the viewer
            //Viewer.Message.SendUtility.SendEmitterSetBinary( gameSettingsDoc.EmitterSetDocument );

            return true;
        }


        /// <summary>
        /// コマンドハンドラ：プレビューファイルの読み込み
        /// </summary>
        public static bool Command_FilePrevLoad( IProjectDocument project,
                                                 EmitterSetDocument owner,
                                                 GameSettingsDocument targetDoc )
        {
            if ( (project is EffectProjectDocument)==false )
                return false;

            if ( owner==null )
                return false;

            if ( targetDoc==null )
                return false;

            System.Windows.Forms.OpenFileDialog dialog =
                new System.Windows.Forms.OpenFileDialog();

            dialog.Filter             = res.Strings.IO_FILEFILTER_CAFE_PREV;
            dialog.Multiselect        = true;
            dialog.InitialDirectory   = Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Prev );
            dialog.FilterIndex        = 0;
            dialog.AutoUpgradeEnabled = Config.Data.DocumentIO.AutoUpgradeEnabled;

            /*
            if ( dialog.ShowDialog(TheApp.MainFrame)!=System.Windows.Forms.DialogResult.OK )
                return false;
            */

            Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Prev,
                                                       dialog.FileName );

            string filePath = dialog.FileName;
            string dirPath  = Path.GetDirectoryName( filePath );

            string name = Path.GetFileNameWithoutExtension(filePath);
            if (name.Length > DocumentIO.s_MaxFileNameLength)
            {
                DocumentIO.ErrorFileNameLengthLoad(name);
                return false;
            }

            // .prevファイル読み込み
            GameConfigDataXml content = null;
            try
            {
                content = XmlUtility.DeserializeFromFile<GameConfigDataXml>( filePath );

                /*
                GameSettingsDocumentImportCommand cmd =
                    new GameSettingsDocumentImportCommand( content, targetDoc );

                TheApp.CommandManager.ScheduleExec( cmd,
                                                    cmd.Description,
                                                    cmd.DataSrcDesc );
                */
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine(ex.Message);
                return false;
            }

            // ツリー更新
            project.BuildTreeNodes();

            // Send the emitter set to the viewer
            //Viewer.Message.SendUtility.SendEmitterSetBinary( owner );

            return true;
        }


        /// <summary>
        /// Load game settings document from the loaded XML content.
        /// </summary>
        /// <param name="content">The XML contents to load the game settings from.</param>
        /// <returns>The loaded game settings document.</returns>
        private static GameSettingsDocument LoadGameSettingsDocument( GameConfigDataXml content,
                                                                      EmitterSetDocument esetDoc,
                                                                      GameSettingsDocument targetDoc )
        {
            IProjectDocument project = ProjectManager.ActiveProject;

            // Process post-deserialize event.
            content.PostDeserialize();

            // Create the game settings document.
            GameSettingsDocument gameSettingsDoc = targetDoc;
            if ( gameSettingsDoc==null )
            {
                gameSettingsDoc = new GameSettingsDocument( esetDoc,
                                                            content.Name );

                // Add it to the emitter set.
                esetDoc.AddGameSettingsDocument( gameSettingsDoc );
            }

            return gameSettingsDoc;
        }


        /// <summary>
        /// Get or load emitter set for the game settings document.
        /// </summary>
        /// <param name="project">The project document.</param>
        /// <returns>The emitter set document.</returns>
        private static EmitterSetDocument
            GetEmitterSetForGameSettingsDocument( IProjectDocument project,
                                                  string basePath,
                                                  string esetPath,
                                                  List<string> loadedEmitterSets )
        {
            EmitterSetDocument esetDoc = null;

            // Check if the emitter set has already been loaded.
            string esetName = Path.GetFileNameWithoutExtension( esetPath );
            esetDoc = (project as EffectProjectDocument).FindByName( esetName );

            // The emitter set is not there, load it.
            if ( esetDoc!=null )
                return esetDoc;

            // Compose the full emitter set path.
            string esetFullPath = esetPath;
            if ( Path.IsPathRooted( esetFullPath )==false )
            {
                esetFullPath = Path.Combine( basePath, esetFullPath );
            }

            // The path is not correct, unable to load the emitter set.
            if ( Path.IsPathRooted(esetFullPath)==false ||
                 File.Exists(esetFullPath)==false )
            {
                string msg = string.Format( res.Strings.CONSOLE_FILE_NOT_FOUND_MSG,
                                            esetFullPath );
                TheApp.Logger.Error.AddMessage( msg );
                return null;
            }

            // エミッタセットの読み込み
            esetDoc = DocumentIO.LoadEmitterSetFile( esetFullPath, project );
            if ( esetDoc==null )
                return null;

            if ( loadedEmitterSets!=null )
            {
                // 一緒に読み込んだエミッタセットを通知する
                loadedEmitterSets.Add( esetDoc.FilePath );
            }

            // エミッタセットをプロジェクトへ追加
            /*
            EmitterSetDocumentAddRemoveCommand cmdAddEmitterSet =
                new EmitterSetDocumentAddRemoveCommand( esetDoc,
                                                        project,
                                                        EmitterSetDocumentAddRemoveCommand.OpFlg.Add );
            TheApp.CommandManager.Execute( cmdAddEmitterSet );
            */

            return esetDoc;
        }

        #endregion

        #region バイナリ保存

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileSaveBinary(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                // ドキュメントがロードされていれば有効
                bool bEnable = ( ProjectManager.ActiveProject!=null &&
                                 ProjectManager.ActiveProject.Count>0 );
                args.CommandUI.Enabled = bEnable;
                return;
            }

            // 実行
            Debug.Assert( ProjectManager.ActiveProject!=null );

            List<IDocument> documents = new List<IDocument>();
            foreach ( IDocument item in ProjectManager.ActiveProject.EmitterSetDocuments )
            {
                // 拡張子が NULL 文字列のときはバイナリー保存リストから外します。
                if (String.IsNullOrEmpty(item.BinaryFileExt) == false)
                {
                    documents.Add(item);
                }
            }

            if (documents.Count > 0)
            {
                BinarySaveDialog dialog = new BinarySaveDialog(documents);

                // Use the most recent directory as default.
                dialog.CurrentDirectorPath =
                    Config.Data.DocumentIO.GetLastAccessedDir( DocumentConstants.Beset );
                if ( dialog.ShowDialog(TheApp.MainFrame)==DialogResult.OK )
                {
                    // Remember the directory path.
                    Config.Data.DocumentIO.SetLastAccessedDir( DocumentConstants.Beset,
                                                               dialog.CurrentDirectorPath );
                    if ( dialog.Save()==false )
                    {
                        // 終了メッセージ
                        ThreadSafeMsgBox.Show( dialog.CurrentDirectorPath + "\n" + res.Strings.WARNING_BINARY_SAVE_FAILED,
                                               res.Strings.BINARY_SAVE_CAPTION,
                                               MessageBoxButtons.OK,
                                               MessageBoxIcon.Exclamation );
                    }
                    else
                    {
                        // 終了メッセージ
                        ThreadSafeMsgBox.Show( dialog.CurrentDirectorPath + "\n" + res.Strings.BINARY_SAVE_END_MSG,
                                               res.Strings.BINARY_SAVE_CAPTION,
                                               MessageBoxButtons.OK,
                                               MessageBoxIcon.Information );
                    }
                }
            }
        }
        */

        #endregion

        #region 閉じる

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileClose(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                bool bEnable = (ProjectManager.ActiveProject != null
                        && ProjectManager.ActiveProject.Count > 0);

                if (bEnable == true)
                {
                    IDocument doc = ProjectManager.ActiveDocument;
                    if (doc != null)
                    {
                        if (doc.ObjectID != GuiObjectID.EmitterSet)
                        {
                            bEnable = false;
                        }
                    }
                }

                args.CommandUI.Enabled = bEnable;
                return;
            }

            // 実行
            Debug.Assert(ProjectManager.ActiveDocument != null);
            DocumentCloser closer = new DocumentCloser();
            closer.CloseWithChildren(ProjectManager.ActiveDocument);
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileCloseAll(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled =
                    (ProjectManager.ActiveProject != null
                        && ProjectManager.ActiveProject.Count > 0);
                return;
            }

            // 実行
            Debug.Assert(DocumentManager.Documents.Count > 0);

            Viewer.Message.PreviewCommunicator.CloseAllModelInfo();

            DocumentCloser closer = new DocumentCloser();
            if (closer.CloseAll()) { MCSManager.WaitTransmission(); }
        }
        */

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileCloseSelection(MenuCommandArgs args)
        {
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled =
                    (ProjectManager.ActiveProject != null
                        && ProjectManager.ActiveProject.Count > 0);
                return;
            }

            // 実行
            Debug.Assert(DocumentManager.Documents.Count > 0);
            List<IDocument> documents = new List<IDocument>();
            foreach (var project in ProjectManager.Projects)
            {
                foreach (var document in project.SubDocuments)
                {
                    // プロジェクト内のドキュメントのみ追加します。
                    documents.Add(document);
                }
            }
            DocumentsCloseSelectionDialog dialog = new DocumentsCloseSelectionDialog(documents);

            if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK)
            {
                DocumentCloser closer = new DocumentCloser();
                closer.Close(dialog.CloseDocuments);
            }
        }
        */

        #endregion

        #region リネーム
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        /*
        public static void Command_FileRename(MenuCommandArgs args)
        {
            IProjectDocument project  = ProjectManager.ActiveProject;
            IDocument        document = ProjectManager.ActiveDocument;

            App.PropertyEdit.ObjectPropertyDialog dialog =
                App.PropertyEdit.ObjectPropertyDialog.GetFocusedInstance();
            if ( dialog==null )
                return;

            if (args.RequireUpdate)
            {
                bool bEnable = false;
                if (document != null)
                {
                    switch (document.ObjectID)
                    {
                        case GuiObjectID.EmitterSet:
                        case GuiObjectID.Emitter:
                        case GuiObjectID.GameSettings:
                            bEnable = true;
                            break;
                    }
                }
                args.CommandUI.Enabled = bEnable;
                return;
            }

            if ( project==null )
                return;

            if ( document==null)
                return;

            if ( document.ObjectID!=GuiObjectID.EmitterSet &&
                 document.ObjectID!=GuiObjectID.Emitter &&
                 document.ObjectID!=GuiObjectID.GameSettings )
            {
                return;
            }

            dialog.ProjectTreeControl.BeginRename();
        }
        */
        #endregion

        #region Utitlity Functions

        /// <summary>
        /// shows a message box to inform name collision
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameCollision(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_COLLISION_MSG, Path.GetFileName(filePath));
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning );
            */
        }


        /// <summary>
        /// shows a message box to inform name length exceeds maximum
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameLengthCreate(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_LENGTH_CREATE_MSG, Path.GetFileName(filePath));
            /*
            ThreadSafeMsgBox.Show(message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning);
            */
        }

        /// <summary>
        /// shows a message box to inform name length exceeds maximum
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameLengthLoad(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_LENGTH_LOAD_MSG, Path.GetFileName(filePath));
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning );
            */
        }

        /// <summary>
        /// shows a message box to inform name length exceeds maximum
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameLengthSave(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_LENGTH_SAVE_MSG, Path.GetFileName(filePath));
            /*
            ThreadSafeMsgBox.Show(message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning);
            */
        }

        /// <summary>
        /// shows a message box to inform name length exceeds maximum
        /// </summary>
        /// <param name="filepath">path already loaded</param>
        public static void ErrorFileNameLengthCopy(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_LENGTH_COPY_MSG, Path.GetFileName(filePath));
            /*
            ThreadSafeMsgBox.Show(message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning);
            */
        }

        /// <summary>
        /// Read .ftx image data and create bitmaps from it.
        /// </summary>
        /// <param name="data">The loaded ftx image object.</param>
        /// <param name="iMipLevel">Mip map level.</param>
        /// <param name="iDataSize">Data size in bytes.</param>
        /// <param name="iFormat">Image pixel format.</param>
        /// <param name="colorBitmap">The color bitmap created from ftx image.</param>
        /// <param name="alphaBitmap">The alpha bitmap created from ftx image.</param>
        /// <returns>True on success.</returns>
        /*
        public static bool FtxImageDataToBitmaps( nw4f_3difType data,
                                                  out int iMipLevel,
                                                  out int iDataSize,
                                                  out int iFormat,
                                                  out Bitmap colorBitmap,
                                                  out Bitmap alphaBitmap )
        {
            iMipLevel = 1;
            iDataSize = 0;
            iFormat   = 0;

            if ( (data==null) ||
                 (data.Item is textureType)==false )
            {
                colorBitmap = null;
                alphaBitmap = null;
                return false;
            }

            // Get texture info
            textureType textureInfo = data.Item as textureType;

            if ( textureInfo.original_image_array==null ||
                 textureInfo.original_image_array.length<=0 )
            {
                colorBitmap = null;
                alphaBitmap = null;
                return false;
            }

            iDataSize = textureInfo.texture_info.size;

            // Get original image info
            original_imageType origImageInfo =
                textureInfo.original_image_array.original_image[0];
            if ( origImageInfo==null )
            {
                colorBitmap = null;
                alphaBitmap = null;
                return false;
            }

            // Get image size
            int iWidth  = origImageInfo.width;
            int iHeight = origImageInfo.height;

            // Get stream index
            int iStreamIndex = origImageInfo.stream_index;

            if ( textureInfo.stream_array==null ||
                 textureInfo.stream_array.length<=iStreamIndex )
            {
                colorBitmap = null;
                alphaBitmap = null;
                return false;
            }

            // Get the data stream for the original image
            streamType stream = textureInfo.stream_array.stream[iStreamIndex];
            if ( stream==null ||
                 stream.type!=stream_typeType.@byte )
            {
                colorBitmap = null;
                alphaBitmap = null;
                return false;
            }

            int i;

            #region Parse the byte array

            // Parse byte array
            int    iBufIndex = 0;
            int    iHexIndex = 0;
            int    iNum      = 0;
            byte[] buffer    = new byte[stream.count];
            char   ch;
            for ( i=0;i<stream.Value.Length;++i )
            {
                ch = stream.Value[i];
                if ( ch>='0' && ch<='9' )
                    iNum = (int)(ch - '0');
                else if ( ch>='A' && ch<='F' )
                    iNum = (int)(ch - 'A') + 10;
                else
                    continue;

                if ( iHexIndex==0 )
                    buffer[iBufIndex]  = (byte)(iNum << 4);
                else
                    buffer[iBufIndex] += (byte)iNum;

                iHexIndex = (++iHexIndex) % 2;
                if ( iHexIndex==0 )
                    ++iBufIndex;
            }

            #endregion

            // Create bitmaps
            colorBitmap = new Bitmap( iWidth, iHeight, PixelFormat.Format24bppRgb );
            alphaBitmap = new Bitmap( iWidth, iHeight, PixelFormat.Format24bppRgb );

            // The rectangle to lock the bitmap buffer
            Rectangle lockRect = new Rectangle( 0, 0, iWidth, iHeight );

            // Lock the bitmaps for writing
            BitmapData colorData =
                colorBitmap.LockBits( lockRect,
                                      ImageLockMode.ReadWrite,
                                      PixelFormat.Format24bppRgb );
            BitmapData alphaData =
                alphaBitmap.LockBits( lockRect,
                                      ImageLockMode.ReadWrite,
                                      PixelFormat.Format24bppRgb );

            #region Fill in the bitmap data

            // Fill in the bitmap data
            int iNumPixels = iWidth * iHeight;
            if ( origImageInfo.format==original_image_formatType.rgb8 )
            {
                unsafe
                {
                    byte *pixels = (byte*)(void*)colorData.Scan0;
                    byte *alpha  = (byte*)(void*)alphaData.Scan0;

                    int iIndex = 0;
                    for ( i=0;i<iNumPixels;++i )
                    {
                        pixels[iIndex+2] = buffer[iIndex];
                        pixels[iIndex+1] = buffer[iIndex+1];
                        pixels[iIndex  ] = buffer[iIndex+2];

                        alpha[iIndex]    = 255;
                        alpha[iIndex+1]  = 255;
                        alpha[iIndex+2]  = 255;

                        iIndex += 3;
                    }
                }

                iFormat = (int)TextureFormat.Rgb8;
            }
            else
            {
                unsafe
                {
                    byte *pixels = (byte*)(void*)colorData.Scan0;
                    byte *alpha  = (byte*)(void*)alphaData.Scan0;

                    int iDstIndex = 0;
                    int iSrcIndex = 0;
                    for ( i=0;i<iNumPixels;++i )
                    {
                        pixels[iDstIndex+2] = buffer[iSrcIndex];
                        pixels[iDstIndex+1] = buffer[iSrcIndex+1];
                        pixels[iDstIndex  ] = buffer[iSrcIndex+2];
                        alpha[iDstIndex]    = buffer[iSrcIndex+3];
                        alpha[iDstIndex+1]  = buffer[iSrcIndex+3];
                        alpha[iDstIndex+2]  = buffer[iSrcIndex+3];

                        iDstIndex += 3;
                        iSrcIndex += 4;
                    }
                }

                iFormat = (int)TextureFormat.Rgba8;
            }

            #endregion

            return true;
        }
        */

        /// <summary>
        /// check the versions between NW4F_nw4f_3dif.dll and read texture are matched
        /// </summary>
        /// <param name="texPath">texture file path</param>
        public static void CheckTextureVersion(string texPath)
        {
            /*
            // バージョンが不一致であるかチェックします。
            {
                string em4fPath = System.Reflection.Assembly.GetEntryAssembly().Location;
                FileVersionInfo fileversion = FileVersionInfo.GetVersionInfo(Path.GetDirectoryName(em4fPath) + "/NW4F_nw4f_3dif.dll");

                string texVersion;
                {
                    byte[] ftxBinary = null;
                    using (FileStream fr = new FileStream(texPath, FileMode.Open))
                    {
                        using (BinaryReader br = new BinaryReader(fr))
                        {
                            ftxBinary = br.ReadBytes(128);
                        }
                    }
                    texVersion = IfReadUtility.GetVersion(ftxBinary);
                }

                if (0 != String.Compare(fileversion.FileVersion, 0, texVersion, 0, texVersion.Length))
                {
                    TheApp.Logger.Warn.AddMessage(string.Format(res.Strings.IO_INVALID_TEXTURE_VERSION, texVersion, fileversion.FileVersion));
                }
            }
            */
        }

        #endregion

        #region Drag/Drop

        /// <summary>
        /// ドラッグ＆ドロップを受け付けられるファイルかチェックします。
        /// </summary>
        public static bool IsAcceptFile(string filePath)
        {
            string fileext = System.IO.Path.GetExtension(filePath).ToLower();
            switch (fileext)
            {
                /*
                 *  ドロップを許可する拡張子を列挙します。
                 */
                case DocumentConstants.DotFEnv:
                    return true;

                case DocumentConstants.DotEset:
                    return (ProjectManager.ActiveProject.Count < TheApp.MAX_EMITTERSET_COUNT);

                case DocumentConstants.DotPrev:
                    return true;

                default:
                    break;
            }
            return false;
        }

        /// <summary>
        /// ファイルのドロップ処理を行います
        /// </summary>
        /// <param name="file">ドロップするファイルのパス</param>
        public static void Command_DropFile(IProjectDocument project, string filePath)
        {
            string fileext = System.IO.Path.GetExtension(filePath).ToLower();
            switch (fileext)
            {
                case DocumentConstants.DotFEnv:
                    {
                        // エディット環境ファイルの読み込み
                        Command_FileEnvLoad(project, filePath);
                    }
                    break;

                case DocumentConstants.DotEset:
                    {
                        // エミッタセットの読み込み
                        var emitterSetDocument =
                            DocumentIO.LoadEmitterSetFile(filePath, project);

                        if (emitterSetDocument != null)
                        {
                            /*
                            // エミッタセットをプロジェクトへ追加
                            TheApp.CommandManager.Execute(
                                new Command.EmitterSetDocumentAddRemoveCommand(
                                    emitterSetDocument, project,
                                    Command.EmitterSetDocumentAddRemoveCommand.OpFlg.Add));

                            // Create default game settings document
                            string defName =
                                DocumentCreator.CreateDefaultGameSettingsDocName( emitterSetDocument );
                            TheApp.CommandManager.ExecuteIgnoreTargetDocument(
                                new GameSettingsDocumentCreationCommand(defName, emitterSetDocument));

                            Viewer.Message.SendUtility.SendEmitterSetBinary(emitterSetDocument);
                            */
                        }
                    }
                    break;

                case DocumentConstants.DotPrev:
                    {
                        // エディット環境ファイルの読み込み
                        Command_FilePrevLoad( project, filePath );
                    }
                    break;

                default:
                    break;
            }
        }

        #endregion

        #region テクスチャー読み込み

        /// <summary>
        /// テクスチャーファイルの読み込み
        /// </summary>
        /// <param name="xml">Deserializer of emitter data.</param>
        /// <returns>=true..成功</returns>
        public static bool ReadTextureFile( ComplexEmitterDataXml xml )
        {
            bool bUseRelPath    = xml.UseRelativePath;
            xml.UseRelativePath = false;

            string fullPath = string.Empty;

            ReadTextureFile( string.Empty,
                             xml.EditData.TexPatData0.UI_texPatFileName,
                             xml.EditData.TexRes0,
                             ref fullPath );

            ReadTextureFile( string.Empty,
                             xml.EditData.TexPatData1.UI_texPatFileName,
                             xml.EditData.TexRes1,
                             ref fullPath );

            ReadTextureFile( string.Empty,
                             xml.EditData.TexPatData2.UI_texPatFileName,
                             xml.EditData.TexRes2,
                             ref fullPath );

            ReadTextureFile( string.Empty,
                             xml.EditData.ChildTexPatData.UI_texPatFileName,
                             xml.EditData.ChildData.TextureRes,
                             ref fullPath );

            xml.UseRelativePath = bUseRelPath;

            return true;
        }


        /// <summary>
        /// テクスチャーファイルの読み込み
        /// </summary>
        /// <param name="emitterDoc">EmitterDocument</param>
        /// <returns>=true..成功</returns>
        /// <remarks>fullTexPath..に読み込んだフルパスを出力します。</remarks>
        public static bool ReadTextureFile( string baseFolder,
                                            string texPath,
                                            TextureRes textureRes,
                                            ref string fullTexPath )
        {
            return false;

            /*
            if (String.IsNullOrEmpty(texPath) == true)
            {
                // Clear the images
                textureRes.SetTexture(null, null, 0, 0, -1);
                textureRes.TextureInfo.Initialize();
                return false;
            }

            if ( Path.IsPathRooted( texPath )==true )
            {
                fullTexPath = texPath;
            }
            else if ( DocumentConstants.LocateTextureByFileName(baseFolder,
                                                                texPath,
                                                                out fullTexPath)==false )
            {
                textureRes.SetTexture( null, null, 0, 0, -1 );
                textureRes.TextureInfo.Initialize();
                return false;
            }

            int      iMipLevel = 1;
            int      iDataSize = 0;
            int      iFormat   = 0;
            Bitmap[] colorImgs = null;
            Bitmap[] alphaImgs = null;

            Enum format;
            nw.g3d.nw4f_3dif.texture_info_comp_selValue[] compCelValue;

            TextureInfo texInfo = new TextureInfo();

            bool bResult = ReadTextureFile( fullTexPath,
                                            ref texInfo,
                                            out format,
                                            out iDataSize,
                                            out iMipLevel,
                                            out compCelValue,
                                            out colorImgs,
                                            out alphaImgs );
            if ( bResult==false )
            {
                textureRes.SetTexture( null, null, 0, 0, -1 );
                textureRes.TextureInfo.Initialize();
                return false;
            }

            textureRes.TextureInfo.Set( texInfo );

            // Bitmaps are not needed on console mode.
            // Bitmaps are only for displaying on the UI.
            if ( TheApp.ConsoleMode==true )
                return true;

            if ( format is TextureFormat )
            {
                App.Utility.TGAPixelFormatInfo info =
                    TheApp.PixelFormatUtil.GetPixelFormatInfo( (TextureFormat)format );
                if ( info==null )
                    return false;

                if ( info.EnableA==true )
                    iFormat = (int)TextureFormat.Rgba8;
                else
                    iFormat = (int)TextureFormat.Rgb8;
            }
            else
            {
                App.Utility.FTXPixelFormatInfo info =
                    TheApp.PixelFormatUtil.GetPixelFormatInfo( (texture_info_quantize_typeType)format );
                if ( info==null )
                    return false;

                if ( info.EnableA==true )
                    iFormat = (int)TextureFormat.Rgba8;
                else
                    iFormat = (int)TextureFormat.Rgb8;
            }

            textureRes.SetTexture( colorImgs,
                                   alphaImgs,
                                   iMipLevel,
                                   iDataSize,
                                   iFormat );

            return true;
            */
        }

        /// <summary>
        /// Read texture file.
        /// </summary>
        /// <param name="path">The file path to the texture file.</param>
        /// <param name="format">Format of the texture.</param>
        /// <param name="iDataSize">Data size of the biggest mipmap.</param>
        /// <param name="iNumMipMaps">Number of mipmaps.</param>
        /// <param name="compSelector">Values of color component selector.</param>
        /// <param name="colorImgs">Color bitmap for the mipmaps.</param>
        /// <param name="alphaImgs">Alpha bitmap for the mipmaps.</param>
        /*
        private static bool ReadTextureFile( string path,
                                             ref TextureInfo texInfo,
                                             out Enum format,
                                             out int iDataSize,
                                             out int iNumMipMaps,
                                             out texture_info_comp_selValue[] compSelector,
                                             out Bitmap[] colorImgs,
                                             out Bitmap[] alphaImgs )
        {
            ConvertTimer.Start(TimerType.TIMER_TEXTURE_LOAD);

            format       = TextureFormat.Unknown;
            iDataSize    = 0;
            iNumMipMaps  = 0;
            compSelector = null;
            colorImgs    = null;
            alphaImgs    = null;

            LoadTextureResults result;
            TextureManager     texMgr = TheApp.TextureManager;
            if (texInfo == null)
            {
                TextureInfo tmpTexInfo;
                result = texMgr.LoadTexture( path,
                                             out tmpTexInfo,
                                             out colorImgs,
                                             out alphaImgs,
                                             out format,
                                             out compSelector);
            }
            else
            {
                result = texMgr.LoadTexture( path,
                                             out texInfo,
                                             out colorImgs,
                                             out alphaImgs,
                                             out format,
                                             out compSelector);
            }

            ConvertTimer.Stop(TimerType.TIMER_TEXTURE_LOAD);

            return true;
        }
        */

        /// <summary>
        /// Try to load the texture file and see if the file is valid.
        /// </summary>
        /// <param name="path">The file path to the texture file.</param>
        public static bool TryLoadingTextureFile( string path )
        {
            return false;

            /*
            TextureInfo texInfo = null;
            Enum        format  = TextureFormat.Unknown;

            texture_info_comp_selValue[] compSelector = null;
            Bitmap[]                     colorImgs    = null;
            Bitmap[]                     alphaImgs    = null;

            LoadTextureResults result;
            TextureManager     texMgr = TheApp.TextureManager;

            using ( new TextureManager.TextureLoadingMsgBoxBlock() )
            {
                result = texMgr.LoadTexture( path,
                                             out texInfo,
                                             out colorImgs,
                                             out alphaImgs,
                                             out format,
                                             out compSelector );
                if ( result!=LoadTextureResults.Success )
                    return false;
            }

            return true;
            */
        }

        #endregion

        #region Node Clone/Copy/Paste

        /// <summary>
        /// Clone the current active document node.
        /// </summary>
        /// <param name="srcDocument">The source document to clone from.</param>
        /// <param name="targetParentDoc">
        /// The parent document to add the cloned document to.
        /// Null to add to the same parent document of the source document.
        /// This is only effective when cloning emitters or game settings documents.
        /// </param>
        /// <param name="iIndex">The index to insert the new document to.</param>
        /// <param name="bUseSrcDocName">Use the name of the source document.</param>
        /// <returns>True on success.</returns>
        public static bool CloneDocumentNode( IDocument srcDocument = null,
                                              IDocument targetParentDoc = null,
                                              int iIndex = -1,
                                              bool bUseSrcDocName = false,
                                              string docName = "" )
        {
            DocumentClipboardData data = null;
            IProjectDocument project = ProjectManager.ActiveProject;
            if ( project == null )
                return false;

            // Copy the source document.
            IDocument srcDoc = srcDocument;
            if ( srcDoc==null )
                srcDoc = project.ActiveDocument;
            if ( srcDoc is EmitterSetDocument )
            {
                if ( project.NumEmitterSetDocuments>=TheApp.MAX_EMITTERSET_COUNT )
                    return false;

                // Copy emitter set document.
                data = new DocumentClipboardData( srcDoc as EmitterSetDocument );
            }
            else if ( srcDoc is EmitterDocument )
            {
                IEmitterSetDocument parentDoc = (srcDoc as EmitterDocument).EmitterSetDocument;
                if ( parentDoc==null ||
                     parentDoc.Count>=TheApp.MAX_EMITTER_COUNT )
                {
                    return false;
                }

                // Copy emitter document.
                data = new DocumentClipboardData( srcDoc as EmitterDocument );
            }
            else if ( srcDoc is GameSettingsDocument )
            {
                // Copy game settings document.
                data = new DocumentClipboardData( srcDoc as GameSettingsDocument );
            }
            else
            {
                return false;
            }

            IDocument toSelect = null;
            IEmitterSetDocument toSend = null;

            // Paste the destination document
            if ( data.DocumentType==GuiObjectID.EmitterSet )
            {
                EmitterSetDocument destDoc =
                    data.DeserializeEmitterSetDocument( project,
                                                        res.Strings.COMMAND_DESC_CLONE_DOC_NODE );
                if (destDoc != null)
                {
                    toSelect = destDoc;
                    toSend = destDoc;
                }
            }
            else if ( data.DocumentType==GuiObjectID.Emitter )
            {
                IEmitterSetDocument parentDoc = targetParentDoc as EmitterSetDocument;
                if ( parentDoc==null )
                {
                    parentDoc = (srcDoc as EmitterDocument).EmitterSetDocument;
                    if ( parentDoc==null )
                        return false;
                }

                EmitterDocument destDoc =
                    data.DeserializeEmitterDocument( parentDoc as EmitterSetDocument,
                                                     true,
                                                     true,
                                                     res.Strings.COMMAND_DESC_CLONE_DOC_NODE,
                                                     iIndex,
                                                     bUseSrcDocName );
                if (destDoc != null)
                {
                    toSelect = destDoc;
                    toSend = destDoc.EmitterSetDocument;
                }
            }
            else if ( data.DocumentType==GuiObjectID.GameSettings )
            {
                IEmitterSetDocument parentDoc = targetParentDoc as EmitterSetDocument;
                if ( parentDoc==null )
                {
                    parentDoc = (srcDoc as GameSettingsDocument).EmitterSetDocument;
                    if ( parentDoc==null )
                        return false;
                }

                GameSettingsDocument destDoc =
                    data.DeserializeGameSettingsDocument( parentDoc as EmitterSetDocument,
                                                          true,
                                                          res.Strings.COMMAND_DESC_CLONE_DOC_NODE,
                                                          iIndex,
                                                          bUseSrcDocName );
                if (destDoc != null)
                {
                    toSelect = destDoc;
                    toSend = destDoc.EmitterSetDocument;
                }
            }

            if (null == toSend)
                return false;

            ProjectManager.ActiveDocument = toSelect;

            // パケット送信
            //Viewer.Message.SendUtility.SendEmitterSetBinary(toSend);

            return true;
        }

        /// <summary>
        /// Clone the the specified emitter set.
        /// </summary>
        /// <param name="srcDocument">The source document to clone from.</param>
        /// <param name="docName">Name for the new emitter set document.</param>
        /// <returns>True on success.</returns>
        public static bool CloneEmitterSet( IDocument srcDocument,
                                            string docName )
        {
            DocumentClipboardData data = null;
            IProjectDocument project = ProjectManager.ActiveProject;
            if ( project == null )
                return false;

            // Copy the source document.
            IDocument srcDoc = srcDocument;
            if ( srcDoc==null )
                srcDoc = project.ActiveDocument;
            if ( srcDoc is EmitterSetDocument )
            {
                if ( project.NumEmitterSetDocuments>=TheApp.MAX_EMITTERSET_COUNT )
                    return false;

                // Copy emitter set document.
                data = new DocumentClipboardData( srcDoc as EmitterSetDocument );
            }
            else
            {
                return false;
            }

            IDocument toSelect = null;
            IEmitterSetDocument toSend = null;

            // Paste the destination document
            if ( data.DocumentType==GuiObjectID.EmitterSet )
            {
                EmitterSetDocument destDoc =
                    data.DeserializeEmitterSetDocument( project,
                                                        res.Strings.COMMAND_DESC_CLONE_DOC_NODE,
                                                        docName );
                if ( destDoc!=null )
                {
                    toSelect = destDoc;
                    toSend   = destDoc;
                }
            }

            if ( toSend==null )
                return false;

            ProjectManager.ActiveDocument = toSelect;

            // パケット送信
            //Viewer.Message.SendUtility.SendEmitterSetBinary(toSend);

            return true;
        }


        /// <summary>
        /// Check if the specified document can be copied on the project tree.
        /// </summary>
        /// <param name="doc">The document.</param>
        /// <returns>True if the document can be copied.</returns>
        public static bool CanCopyDocument( IDocument doc )
        {
            if ( doc is EmitterDocument ||
                 doc is GameSettingsDocument ||
                 doc is FieldDocument ||
                 doc is FluctuationDocument ||
                 doc is ChildDocument )
            {
                return true;
            }

            return false;
        }


        /// <summary>
        /// Copy the selected document project tree node.
        /// </summary>
        /// <param name="doc">The document to copy.</param>
        /// <returns>True on success.</returns>
        public static bool CopyDocumentNode( IDocument doc )
        {
            DocumentClipboardData data = null;

            // Copy the document.
            if ( doc is EmitterDocument )
            {
                // Copy emitter document.
                data = new DocumentClipboardData( doc as EmitterDocument );
            }
            else if ( doc is GameSettingsDocument )
            {
                // Copy game settings document.
                data = new DocumentClipboardData( doc as GameSettingsDocument );
            }
            else if ( doc is EmitterSetDocument )
            {
                // Copy emitter set document.
                data = new DocumentClipboardData( doc as EmitterSetDocument );
            }
            else if ( doc is ChildDocument )
            {
                // Copy child document.
                data = new DocumentClipboardData( doc as ChildDocument );
            }
            else if ( doc is FieldDocument )
            {
                // Copy field document.
                data = new DocumentClipboardData( doc as FieldDocument );
            }
            else if ( doc is FluctuationDocument )
            {
                // Copy fluctuation document.
                data = new DocumentClipboardData( doc as FluctuationDocument );
            }
            /* Emitter set copy/paste is now disabled.
            else if ( doc is EmitterSetDocument )
            {
                // Copy emitter set document.
                data = new DocumentClipboardData( doc as EmitterSetDocument );
            }
            */
            else
            {
                return false;
            }

            // Set the serialized document data to clipboard.
            Clipboard.SetData( CLIPBOARDID_PROJECTNODE, data );

            return true;
        }


        /// <summary>
        /// Paste the copied node from system clipboard to the project tree.
        /// </summary>
        /// <param name="doc">The new document will be pasted after this specified document.</param>
        /// <returns>True on success.</returns>
        public static bool PasteDocumentNode( IDocument doc )
        {
            // Paste to the tree node.
            object data =
                Clipboard.GetData( CLIPBOARDID_PROJECTNODE );

            DocumentClipboardData clipboardData = data as DocumentClipboardData;
            if ( clipboardData==null )
                return false;

            switch ( clipboardData.DocumentType )
            {
                /* Emitter set copy/paste is now disabled.
                case GuiObjectID.EmitterSet :
                    {
                        IProjectDocument project = ProjectManager.ActiveProject;
                        if ( project==null )
                            return false;

                        if ( project.NumEmitterSetDocuments>=TheApp.MAX_EMITTERSET_COUNT )
                            return false;

                        EmitterSetDocument document =
                            clipboardData.DeserializeEmitterSetDocument( project );
                        ProjectManager.ActiveDocument = document;

                        // パケット送信
                        Viewer.Message.SendUtility.SendEmitterSetBinary( document );
                    }
                    break;
                */
                case GuiObjectID.Emitter :
                    {
                        EmitterDocument    emitter = doc as EmitterDocument;
                        EmitterSetDocument parent  = emitter.OwnerDocument as EmitterSetDocument;
                        if ( emitter==null ||
                             parent==null ||
                             parent.Count>=TheApp.MAX_EMITTER_COUNT )
                        {
                            return false;
                        }

                        int iIndex = -1;
                        for ( int i=0;i<parent.GetEmitterDocumentCount();++i )
                        {
                            if ( parent.GetEmitterDocumentByIndex(i)==emitter )
                            {
                                iIndex = i;
                                break;
                            }
                        }

                        if ( iIndex<0 )
                            return false;

                        EmitterDocument document =
                            clipboardData.DeserializeEmitterDocument( parent,
                                                                      true,
                                                                      false,
                                                                      res.Strings.COMMAND_DESC_PASTE_DOCUMENT,
                                                                      iIndex+1 );
                        ProjectManager.ActiveDocument = document;

                        // パケット送信
                        //Viewer.Message.SendUtility.SendEmitterSetBinary( parent );
                    }
                    break;

                case GuiObjectID.GameSettings :
                    {
                        GameSettingsDocument gameSettings = doc as GameSettingsDocument;
                        EmitterSetDocument   parent       = gameSettings.OwnerDocument as EmitterSetDocument;
                        if ( gameSettings==null || parent==null )
                        {
                            return false;
                        }

                        int iIndex = -1;
                        for ( int i=0;i<parent.GetNumGameSettingsDocuments();++i )
                        {
                            if ( parent.GetGameSettingsDocumentByIndex(i)==gameSettings )
                            {
                                iIndex = i;
                                break;
                            }
                        }

                        if ( iIndex<0 )
                            return false;

                        GameSettingsDocument document =
                            clipboardData.DeserializeGameSettingsDocument( parent,
                                                                           true,
                                                                           res.Strings.COMMAND_DESC_PASTE_DOCUMENT,
                                                                           iIndex+1 );
                        ProjectManager.ActiveDocument = document;

                        // パケット送信
                        //Viewer.Message.SendUtility.SendEmitterSetBinary( parent );
                    }
                    break;

                default :
                    return false;
            }

            return true;
        }


        /// <summary>
        /// Paste the copied node from system clipboard to the project tree.
        /// </summary>
        /// <param name="doc">The new document will be pasted as the last child.</param>
        /// <returns>True on success.</returns>
        public static bool PasteAsChild( IDocument doc )
        {
            // Paste to the tree node.
            object data =
                Clipboard.GetData( CLIPBOARDID_PROJECTNODE );

            DocumentClipboardData clipboardData = data as DocumentClipboardData;
            if ( clipboardData==null )
                return false;

            switch ( clipboardData.DocumentType )
            {
                case GuiObjectID.Emitter :
                    {
                        EmitterSetDocument parent = doc as EmitterSetDocument;
                        if ( parent==null ||
                             parent.Count>=TheApp.MAX_EMITTER_COUNT )
                        {
                            return false;
                        }

                        EmitterDocument document =
                            clipboardData.DeserializeEmitterDocument( parent,
                                                                      true,
                                                                      true,
                                                                      res.Strings.COMMAND_DESC_PASTE_DOCUMENT );
                        ProjectManager.ActiveDocument = document;

                        // パケット送信
                        //Viewer.Message.SendUtility.SendEmitterSetBinary( parent );
                    }
                    break;

                case GuiObjectID.GameSettings :
                    {
                        EmitterSetDocument parent = doc as EmitterSetDocument;
                        if ( parent==null ||
                             parent.Count>=TheApp.MAX_EMITTER_COUNT )
                        {
                            return false;
                        }

                        GameSettingsDocument document =
                            clipboardData.DeserializeGameSettingsDocument( parent,
                                                                           true,
                                                                           res.Strings.COMMAND_DESC_PASTE_DOCUMENT );
                        ProjectManager.ActiveDocument = document;

                        // パケット送信
                        //Viewer.Message.SendUtility.SendEmitterSetBinary( parent );
                    }
                    break;

                default :
                    return false;
            }

            return true;
        }


        /// <summary>
        /// Paste the copied node from system clipboard to an existing document.
        /// </summary>
        /// <param name="doc">The target document to paste to.</param>
        /// <returns>True on success.</returns>
        public static bool PasteToExistingDocumentNode( IDocument doc )
        {
            // Paste to the tree node.
            object data =
                Clipboard.GetData( CLIPBOARDID_PROJECTNODE );

            DocumentClipboardData clipboardData = data as DocumentClipboardData;
            if ( clipboardData==null )
                return false;

            string srcDocName = string.Copy( clipboardData.DocumentName );
            string dstDocName = string.Empty;
            if ( clipboardData.DocumentType==GuiObjectID.Emitter )
            {
                EmitterDocument dstDoc = doc as EmitterDocument;
                if ( dstDoc==null )
                    return false;

                EmitterSetDocument eset = dstDoc.EmitterSetDocument as EmitterSetDocument;
                if ( eset==null )
                    return false;

                EmitterDocument srcDoc =
                    clipboardData.DeserializeEmitterDocument( eset, false );

                /*
                ICommand cmd = new OverwriteEmitterDocumentCommand( srcDoc, dstDoc, dstDoc.Name );

                TheApp.CommandManager.Execute( cmd );
                */

                // Show message to inform the user the content has been pasted.
                if ( srcDocName!=null && dstDocName!=null )
                {
                    TheApp.Logger.Info.FormatMessage( res.Strings.MSG_PASTE_TO_EXISTING_EMITTER,
                                                      srcDocName,
                                                      dstDocName );
                }

                // パケット送信
                //Viewer.Message.SendUtility.SendEmitterSetBinary( eset );
            }
            else if ( clipboardData.DocumentType==GuiObjectID.GameSettings )
            {
                GameSettingsDocument dstDoc = doc as GameSettingsDocument;
                if ( dstDoc==null )
                    return false;

                EmitterSetDocument eset = dstDoc.EmitterSetDocument as EmitterSetDocument;
                if ( eset==null )
                    return false;

                /*
                OverwriteGameSettingsDocumentCommand cmd =
                    new OverwriteGameSettingsDocumentCommand( clipboardData, dstDoc, dstDoc.Name );
                TheApp.CommandManager.Execute( cmd );
                */

                // Show message to inform the user the content has been pasted.
                if ( srcDocName!=null && dstDocName!=null )
                {
                    TheApp.Logger.Info.FormatMessage( res.Strings.MSG_PASTE_TO_EXISTING_GAMESETTINGS,
                                                      srcDocName,
                                                      dstDocName );
                }

                // パケット送信
                //Viewer.Message.SendUtility.SendEmitterSetBinary(eset);
            }
            else if ( clipboardData.DocumentType==GuiObjectID.Field ||
                      clipboardData.DocumentType==GuiObjectID.Fluctuation ||
                      clipboardData.DocumentType==GuiObjectID.Child )
            {

                IDocument dstDoc = null;
                EmitterSetDocument eset = null;
                switch (clipboardData.DocumentType)
                {
                    case GuiObjectID.Field:
                        FieldDocument fieldDoc = doc as FieldDocument;
                        dstDoc = fieldDoc;
                        eset = fieldDoc.EmitterDocument.EmitterSetDocument as EmitterSetDocument;
                        break;
                    case GuiObjectID.Fluctuation:
                        FluctuationDocument fluctuationDoc = doc as FluctuationDocument;
                        dstDoc = fluctuationDoc;
                        eset = fluctuationDoc.EmitterDocument.EmitterSetDocument as EmitterSetDocument;
                        break;
                    case GuiObjectID.Child:
                        ChildDocument childDoc = doc as ChildDocument;
                        dstDoc = childDoc;
                        eset = childDoc.EmitterDocument.EmitterSetDocument as EmitterSetDocument;
                        break;
                }

                if ( dstDoc==null )
                    return false;

                if ( eset==null )
                    return false;

                dstDocName = string.Copy( doc.Name );

                EmitterDocument srcDoc =
                    clipboardData.DeserializeEmitterDocument( eset, false );

                /*
                DocumentCommand cmd = null;
                switch (clipboardData.DocumentType)
                {
                    case GuiObjectID.Field:
                        cmd = new OverwriteFieldDocumentCommand( srcDoc, dstDoc as FieldDocument);
                        break;
                    case GuiObjectID.Fluctuation:
                        cmd = new OverwriteFluctuationDocumentCommand(srcDoc, dstDoc as FluctuationDocument);
                        break;
                    case GuiObjectID.Child:
                        cmd = new OverWriteChildDocumentCommand(srcDoc, dstDoc as ChildDocument);
                        break;
                }

                TheApp.CommandManager.Execute( cmd );
                */

                // Show message box to inform the user the content has been pasted.
                if ( srcDocName!=null && dstDocName!=null )
                {
                    TheApp.Logger.Info.FormatMessage( res.Strings.MSG_PASTE_TO_EXISTING_EMITTER,
                                                      srcDocName,
                                                      dstDocName );
                }

                // パケット送信
                //Viewer.Message.SendUtility.SendEmitterSetBinary( eset );
            }
            else
            {
                return false;
            }

            return true;
        }

        #endregion
    }
}
