﻿// ========================================================================
// <copyright file="ConsoleApplication.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Xml;

using NWCore.DataModel;
//using NWCore.Converter;

//using App.Controls;
using App.IO;
using App.Data;
using App.Utility;

namespace App
{
    /// <summary>
    /// コンソールアプリケーションクラス。
    /// </summary>
    public static class ConsoleApplication
    {
        #region Enumerator for binary conversion types

        /// <summary>
        /// Enumerator for the types of target platform of the binary conversion.
        /// </summary>
        public enum BinaryConversionTypes
        {
            AllPlatforms,
            Windows,
            Cafe,
            CTR
        }

        #endregion

        #region Enumerator for argument process result

        /// <summary>
        /// Enumerator for argument process result.
        /// </summary>
        private enum ArgProcessResult
        {
            Success,
            Fail,
            UnknownArgument,
            RequestNextArgument,
            ReportEndOfArgument,
            SkipArgument
        }

        #endregion

        #region Abstract class for argument processor

        /// <summary>
        /// Abstract class for argument processors.
        /// </summary>
        private abstract class BaseArgumentProcessor
        {
            /// <summary>Get the flag indicating whether this processor has processed arguments.</summary>
            public virtual bool IsProcessed { get; protected set; }

            /// <summary>
            /// Check if the argument can be processed.
            /// </summary>
            /// <param name="arg">The argument.</param>
            /// <returns>
            /// - Success means the argument can be processed.
            /// - UnknowArgument means the argument is unknown to this processor.
            /// - SkipArgument means the argument is known but should be skipped.
            /// </returns>
            public abstract ArgProcessResult CanProcess( string arg );

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public abstract ArgProcessResult Process( string arg );
        }

        #endregion

        #region Argument processors

        #region Known argument

        private abstract class KnownArgumentProcessor : BaseArgumentProcessor
        {
            public abstract string[] Commands { get; }
            public virtual string[] Arguments { get { return new string[] { }; } }
            public abstract string Description { get; }

            public virtual string ShortUsage()
            {
                if (Arguments.Any())
                    return string.Format("{0} {1}", Commands.First(), string.Join(" ", Arguments));
                else
                    return Commands.First();
            }

            public virtual string LongUsage()
            {
                if (Arguments.Any())
                    return string.Format("{0} {1}", string.Join(", ", Commands), string.Join(" ", Arguments));
                else
                    return string.Join(", ", Commands);
            }

            /// <summary>
            /// Check if the argument can be processed.
            /// </summary>
            /// <param name="arg">The argument.</param>
            /// <returns>
            /// - Success means the argument can be processed.
            /// - UnknowArgument means the argument is unknown to this processor.
            /// - SkipArgument means the argument is known but should be skipped.
            /// </returns>
            public override ArgProcessResult CanProcess(string arg)
            {
                if (Commands.Any(x => arg == x))
                    return ArgProcessResult.Success;

                return ArgProcessResult.UnknownArgument;
            }
        }

        #endregion

        #region Help page ( -? )

        /// <summary>
        /// Argument processor for "help" screen.
        /// </summary>
        private class HelpPageArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-?" }; } }
            public override string Description { get { return res.CommandLineHelp.HELP_PAGE_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process(string arg)
            {
                if (Commands.Any(x => arg == x) == false)
                    return ArgProcessResult.UnknownArgument;

                ConsoleApplication._showHelp = true;

                return ArgProcessResult.Success;
            }
        }

        #endregion

        #region Silent mode ( -q / --quiet )

        /// <summary>
        /// Argument processor for "silent mode" flag.
        /// </summary>
        private class SilentArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] {"-q", "--quiet"}; } }
            public override string Description { get { return res.CommandLineHelp.SILENT_MODE_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if (Commands.Any(x => arg == x) == false)
                    return ArgProcessResult.UnknownArgument;

                ConsoleApplication._quiet = true;

                return ArgProcessResult.Success;
            }
        }

        #endregion

        #region Connect target ( -Connect )

        /// <summary>
        /// Argument processor for connect target.
        /// </summary>
        private class ConnectTargetArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-Connect" }; } }
            public override string Description { get { return res.CommandLineHelp.CONNECT_TARGET_DESC; } }
            public override string LongUsage() { return "-Connect=target"; }

            /// <summary>
            /// Check if the argument can be processed.
            /// </summary>
            /// <param name="arg">The argument.</param>
            /// <returns>
            /// - Success means the argument can be processed.
            /// - UnknowArgument means the argument is unknown to this processor.
            /// - SkipArgument means the argument is known but should be skipped.
            /// </returns>
            public override ArgProcessResult CanProcess( string arg )
            {
                if ( arg=="-Connect" )
                    return ArgProcessResult.SkipArgument;

                return ArgProcessResult.UnknownArgument;
            }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if ( m_bProcessing==false )
                {
                    if ( arg.StartsWith( "-" )==true )
                    {
                        return ArgProcessResult.UnknownArgument;
                    }
                    else if ( arg=="-Connect" )
                    {
                        m_bProcessing = true;
                        return ArgProcessResult.RequestNextArgument;
                    }
                }

                // Skip this argument
                m_bProcessing = false;

                return ArgProcessResult.SkipArgument;
            }

            private bool m_bProcessing = false;
        }

        #endregion

        #region Convert emitter sets to binary file ( -a )

        /// <summary>
        /// Argument processor for converting emitter sets to binary file.
        /// </summary>
        private class ConvertBinaryForAllPlatformsArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-a" }; } }
            public override string[] Arguments { get { return new string[] { "convert.xml", "file"}; } }
            public override string Description { get { return res.CommandLineHelp.CONVERT_BINARY_FOR_ALL_PLATFORMS_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process(string arg)
            {
                // m_iProcessStage == 0, initial state.
                // m_iProcessStage == 1, expecting binary conversion argument file name.
                // m_iProcessStage == 2, expecting binary file name.
                if (m_iProcessStage <= 0)
                {
                    if (Commands.Any(x => arg == x))
                    {
                        //_convType = BinaryConversionTypes.AllPlatforms;
                        m_iProcessStage = 1;
                        return ArgProcessResult.RequestNextArgument;
                    }
                    else
                    {
                        return ArgProcessResult.UnknownArgument;
                    }
                }

                if (m_iProcessStage == 1)
                {
                    ++m_iProcessStage;

                    _binConvArgFile = ShellUtility.ToAbsolutePath( arg );
                    if (File.Exists(_binConvArgFile) == false)
                    {
                        Error("CONSOLE_MSG_BIN_CONV_PARAM_FILE_NOT_FOUND", _binConvArgFile);
                        return ArgProcessResult.Fail;
                    }

                    // Use the argument file directory as the working folder.
                    _convWorkingDir = Path.GetDirectoryName(_binConvArgFile);

                    return ArgProcessResult.RequestNextArgument;
                }
                else if (m_iProcessStage == 2)
                {
                    m_iProcessStage = 0;

                    _binFileName = ShellUtility.ToAbsolutePath( arg, _convWorkingDir );

                    return ArgProcessResult.ReportEndOfArgument;
                }

                return ArgProcessResult.Fail;
            }

            private int m_iProcessStage = 0;
        }

        #endregion

        #region Convert emitter sets to binary file ( -c or -f )

        /// <summary>
        /// Argument processor for converting emitter sets to binary file.
        /// </summary>
        private class ConvertBinaryForDeviceArgProcessor : KnownArgumentProcessor
        {
#if BUILD_FOR_CTR
            public override string[] Commands { get { return new string[] { "-c" }; } }
#else
            public override string[] Commands { get { return new string[] { "-f" }; } }
#endif
            public override string[] Arguments { get { return new string[] { "convert.xml", "file" }; } }
            public override string Description { get { return res.CommandLineHelp.CONVERT_BINARY_FOR_DEVICE_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process(string arg)
            {
                // m_iProcessStage == 0, initial state.
                // m_iProcessStage == 1, expecting binary conversion argument file name.
                // m_iProcessStage == 2, expecting binary file name.
                if (m_iProcessStage <= 0)
                {

                    if (Commands.Any(x => arg == x))
                    {
#if BUILD_FOR_CTR
                        _convType = BinaryConversionTypes.CTR;
#else
                        //_convType = BinaryConversionTypes.Cafe;
#endif
                        m_iProcessStage = 1;
                        return ArgProcessResult.RequestNextArgument;
                    }
                    else
                    {
                        return ArgProcessResult.UnknownArgument;
                    }
                }

                if (m_iProcessStage == 1)
                {
                    ++m_iProcessStage;

                    _binConvArgFile = ShellUtility.ToAbsolutePath( arg );
                    if (File.Exists(_binConvArgFile) == false)
                    {
                        Error("CONSOLE_MSG_BIN_CONV_PARAM_FILE_NOT_FOUND", _binConvArgFile);
                        return ArgProcessResult.Fail;
                    }

                    // Use the argument file directory as the working folder.
                    _convWorkingDir = Path.GetDirectoryName(_binConvArgFile);

                    return ArgProcessResult.RequestNextArgument;
                }
                else if (m_iProcessStage == 2)
                {
                    m_iProcessStage = 0;

                    _binFileName = ShellUtility.ToAbsolutePath( arg, _convWorkingDir );

                    return ArgProcessResult.ReportEndOfArgument;
                }

                return ArgProcessResult.Fail;
            }

            private int m_iProcessStage = 0;
        }

        #endregion

        #region Convert emitter sets to binary file ( -w )

        /// <summary>
        /// Argument processor for converting emitter sets to binary file.
        /// </summary>
        private class ConvertBinaryForWindowsArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-w" }; } }
            public override string[] Arguments { get { return new string[] { "convert.xml", "file" }; } }
            public override string Description { get { return res.CommandLineHelp.CONVERT_BINARY_FOR_WINDOWS_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process(string arg)
            {
                // m_iProcessStage == 0, initial state.
                // m_iProcessStage == 1, expecting binary conversion argument file name.
                // m_iProcessStage == 2, expecting binary file name.
                if (m_iProcessStage <= 0)
                {
                    if (Commands.Any(x => arg == x))
                    {
                        //_convType = BinaryConversionTypes.Windows;
                        m_iProcessStage = 1;
                        return ArgProcessResult.RequestNextArgument;
                    }
                    else
                    {
                        return ArgProcessResult.UnknownArgument;
                    }
                }

                if (m_iProcessStage == 1)
                {
                    ++m_iProcessStage;

                    _binConvArgFile = ShellUtility.ToAbsolutePath( arg );
                    if (File.Exists(_binConvArgFile) == false)
                    {
                        Error("CONSOLE_MSG_BIN_CONV_PARAM_FILE_NOT_FOUND", _binConvArgFile);
                        return ArgProcessResult.Fail;
                    }

                    // Use the argument file directory as the working folder.
                    _convWorkingDir = Path.GetDirectoryName(_binConvArgFile);

                    return ArgProcessResult.RequestNextArgument;
                }
                else if (m_iProcessStage == 2)
                {
                    m_iProcessStage = 0;

                    _binFileName = ShellUtility.ToAbsolutePath( arg, _convWorkingDir );

                    return ArgProcessResult.ReportEndOfArgument;
                }

                return ArgProcessResult.Fail;
            }

            private int m_iProcessStage = 0;
        }

        #endregion

        #region Convert emitter sets to binary file ( -h )

        /// <summary>
        /// Argument processor for converting emitter sets to binary file.
        /// </summary>
        private class BinaryHeaderArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-h" }; } }
            public override string[] Arguments { get { return new string[] { "header" }; } }
            public override string Description { get { return res.CommandLineHelp.BINARY_HEADER_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process(string arg)
            {
                if (Commands.Any(x => arg == x))
                {
                    m_bProcessing = true;
                    return ArgProcessResult.RequestNextArgument;
                }
                else if (m_bProcessing == false)
                {
                    return ArgProcessResult.UnknownArgument;
                }

                m_bProcessing = false;

                //_bOutputHeader = true;
                _binHeaderName = string.Copy(arg);

                return ArgProcessResult.ReportEndOfArgument;
            }

            private bool m_bProcessing = false;
        }

        #endregion

        #region User shader path ( -us )

        /// <summary>
        /// Argument processor for user shader path.
        /// </summary>
        private class UserShaderPathArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-us" }; } }
            public override string[] Arguments { get { return new string[] { "file" }; } }
            public override string Description { get { return res.CommandLineHelp.USER_SHADER_PATH_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if (Commands.Any(x => arg == x))
                {
                    m_bProcessing = true;
                    return ArgProcessResult.RequestNextArgument;
                }
                else if ( m_bProcessing==false )
                {
                    return ArgProcessResult.UnknownArgument;
                }

                m_bProcessing = false;

                // Environment variable is allowed in the path.
                string shaderPath =
                    RequireDocument.GetFullPathForRequireDocument( arg );
                if ( shaderPath==null )
                {
                    _userShaderPath = string.Empty;
                    return ArgProcessResult.ReportEndOfArgument;
                }

                // We don't allow relative path.
                if ( Path.IsPathRooted(shaderPath)==false )
                    return ArgProcessResult.ReportEndOfArgument;

                // Check if the directory exists.
                if ( File.Exists(shaderPath)==false )
                    return ArgProcessResult.ReportEndOfArgument;

                // Set the user shader path.
                _userShaderPath = shaderPath;

                return ArgProcessResult.ReportEndOfArgument;
            }

            private bool m_bProcessing = false;
        }

        #endregion

        #region User config path ( -uc )

        /// <summary>
        /// Argument processor for user config file path.
        /// </summary>
        private class UserConfigPathArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-uc" }; } }
            public override string[] Arguments { get { return new string[] { "folder" }; } }
            public override string Description { get { return res.CommandLineHelp.USER_CONFIG_PATH_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if (Commands.Any(x => arg == x))
                {
                    m_bProcessing = true;
                    return ArgProcessResult.RequestNextArgument;
                }
                else if ( m_bProcessing==false )
                {
                    return ArgProcessResult.UnknownArgument;
                }

                m_bProcessing = false;

                // Environment variable is allowed in the path.
                string path =
                    RequireDocument.GetFullPathForRequireDocument( arg );
                if ( path==null )
                {
                    _userConfigPath = string.Empty;
                    return ArgProcessResult.ReportEndOfArgument;
                }

                // We don't allow relative path.
                if ( Path.IsPathRooted(path)==false )
                    return ArgProcessResult.ReportEndOfArgument;

                // Check if the directory or file exists.
                if ( Directory.Exists(path)==false &&
                     File.Exists(path)==false )
                    return ArgProcessResult.ReportEndOfArgument;

                // Set the user config path.
                _userConfigPath = path;

                return ArgProcessResult.ReportEndOfArgument;
            }

            private bool m_bProcessing = false;
        }

        #endregion

        #region Show warning dialogs ( -ow )

        /// <summary>
        /// Argument processor for showing warning dialogs.
        /// </summary>
        private class ShowWarningDialogArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-ow" }; } }
            public override string Description { get { return res.CommandLineHelp.SHOW_WARNING_DIALOG_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if (Commands.Any(x => arg == x) == false)
                    return ArgProcessResult.UnknownArgument;

                TheApp.EnableConsoleModeWarningDialogs = true;

                return ArgProcessResult.Success;
            }
        }

        #endregion

        #region Linear mode ( -lm )

        /// <summary>
        /// Argument processor for linear mode.
        /// </summary>
        private class LinearModeArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-lm" }; } }
            public override string[] Arguments { get { return new string[] { "on|off" }; } }
            public override string Description { get { return res.CommandLineHelp.LINEAR_MODE_DESC; } }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if (Commands.Any(x => arg == x))
                {
                    m_bProcessing = true;
                    return ArgProcessResult.RequestNextArgument;
                }
                else if ( m_bProcessing==false )
                {
                    return ArgProcessResult.UnknownArgument;
                }

                m_bProcessing = false;

                if ( string.Compare(arg, "on", true)==0 )
                {
                    TheApp.IsGammaCorrectionEnabled = true;
                }
                else if ( string.Compare(arg, "off", true)==0 )
                {
                    TheApp.IsGammaCorrectionEnabled = false;
                }
                else
                {
                    return ArgProcessResult.UnknownArgument;
                }

                return ArgProcessResult.ReportEndOfArgument;
            }

            private bool m_bProcessing = false;
        }

        #endregion

        #region Command line eset updater ( -u )

        /// <summary>
        /// Argument processor for updating eset files to the latest version.
        /// </summary>
        private class EsetUpdaterArgProcessor : KnownArgumentProcessor
        {
            public override string[] Commands { get { return new string[] { "-u" }; } }
            public override string[] Arguments { get { return new string[] { "convert.xml" }; } }
            public override string Description { get { return res.CommandLineHelp.ESET_UPDATER_DESC; } }

            /// <summary>Get the eset list XML file path.</summary>
            public string EsetListFilePath { get; private set; }

            /// <summary>Get the working folder path.</summary>
            public string WorkingFolderPath { get; private set; }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if (Commands.Any(x => arg == x))
                {
                    m_bProcessing = true;
                    return ArgProcessResult.RequestNextArgument;
                }
                else if ( m_bProcessing==false )
                {
                    return ArgProcessResult.UnknownArgument;
                }

                m_bProcessing = false;

                this.EsetListFilePath = ShellUtility.ToAbsolutePath( arg );
                if ( File.Exists(this.EsetListFilePath)==false )
                {
                    Error( "CONSOLE_MSG_BIN_CONV_PARAM_FILE_NOT_FOUND", this.EsetListFilePath );
                    return ArgProcessResult.Fail;
                }

                // Use the argument file directory as the working folder.
                this.WorkingFolderPath = Path.GetDirectoryName( this.EsetListFilePath );

                this.IsProcessed = true;

                return ArgProcessResult.ReportEndOfArgument;
            }

            private bool m_bProcessing = false;
        }

        #endregion

        #region Unknown argument

        /// <summary>
        /// Argument processor for unknown arguments.
        /// ( This processor must process after all the processors )
        /// </summary>
        private class UnknownArgProcessor : BaseArgumentProcessor
        {
            /// <summary>
            /// Check if the argument can be processed.
            /// </summary>
            /// <param name="arg">The argument.</param>
            /// <returns>
            /// - Success means the argument can be processed.
            /// - UnknowArgument means the argument is unknown to this processor.
            /// - SkipArgument means the argument is known but should be skipped.
            /// </returns>
            public override ArgProcessResult CanProcess( string arg )
            {
                return ArgProcessResult.SkipArgument;
            }

            /// <summary>
            /// Process the argument.
            /// </summary>
            /// <param name="arg">The argument to be processed.</param>
            /// <returns>The result of the process.</returns>
            public override ArgProcessResult Process( string arg )
            {
                if ( arg.StartsWith( "-" ) )
                    Error( "CONSOLE_UNKNOWN_OPTION_MSG", arg );
                else
                    Error( "CONSOLE_UNKNOWN_ARGUMENT_MSG", arg );

                return ArgProcessResult.UnknownArgument;
            }
        }

        #endregion

        #endregion

        #region Methods for accessing the argument processors

        /// <summary>
        /// Add a new argument processor.
        /// </summary>
        /// <param name="processor">The argument processor.</param>
        private static void AddArgumentProcessor( BaseArgumentProcessor processor )
        {
            if ( processor==null )
                return;

            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( processor.GetType().Name );
            if ( _argProcessorMap.ContainsKey(iNameCRC)==true )
                return;

            _argProcessorList.Add( processor );
            _argProcessorMap.Add( iNameCRC, processor );
        }


        /// <summary>
        /// Find the argument processor with the specified name.
        /// </summary>
        /// <param name="name">The name of the argument processor.</param>
        /// <returns>The argument processor or null if not found.</returns>
        private static BaseArgumentProcessor FindArgumentProcessor( string name )
        {
            if ( string.IsNullOrEmpty(name)==true )
                return null;

            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );
            if ( _argProcessorMap.ContainsKey(iNameCRC)==false )
                return null;

            return _argProcessorMap[ iNameCRC ];
        }

        #endregion

        #region Initialize argument list

        /// <summary>
        /// Initialize argument list.
        /// </summary>
        public static void InitArgumentList()
        {
            _argProcessorList.Clear();

            AddArgumentProcessor( new HelpPageArgProcessor() );
            AddArgumentProcessor( new ConvertBinaryForWindowsArgProcessor() );
            AddArgumentProcessor( new ConvertBinaryForDeviceArgProcessor() );
            AddArgumentProcessor( new ConvertBinaryForAllPlatformsArgProcessor() );
            AddArgumentProcessor( new BinaryHeaderArgProcessor() );
            AddArgumentProcessor( new SilentArgProcessor() );
            AddArgumentProcessor( new ConnectTargetArgProcessor() );
            AddArgumentProcessor( new UserShaderPathArgProcessor() );
            AddArgumentProcessor( new UserConfigPathArgProcessor() );
            AddArgumentProcessor( new ShowWarningDialogArgProcessor() );
            AddArgumentProcessor( new EsetUpdaterArgProcessor() );
            AddArgumentProcessor( new LinearModeArgProcessor() );

            // This must be added after all other processors.
            AddArgumentProcessor( new UnknownArgProcessor() );
        }

        #endregion

        #region Eset binary conversion

        /// <summary>
        /// Helper method for parsing the emitter set file list.
        /// </summary>
        /// <param name="esetListNode">The XML element that contains the eset file list.</param>
        /// <param name="workingPath">The working folder path.</param>
        /// <param name="esetFileList">The eset file list parsed from the given XML element.</param>
        /// <returns>True on success.</returns>
        private static bool ParseEsetFileList( XmlElement esetListNode,
                                               string workingPath,
                                               out List<string> esetFileList )
        {
            esetFileList = new List<string>();

            if ( esetListNode==null )
                return false;

            string filePath = string.Empty;
            foreach ( XmlNode esetNode in esetListNode.ChildNodes )
            {
                if ( esetNode.Name!="EmitterSet" )
                    continue;

                foreach ( XmlNode child in esetNode.ChildNodes )
                {
                    #region Parse the emitter set file path or directory

                    if ( child.Name=="file" )
                    {
                        // Verify the file path.
                        filePath = ShellUtility.ToAbsolutePath( child.InnerText, workingPath );
                        if ( Path.IsPathRooted(filePath)==false ||
                             Path.GetExtension(filePath).ToLower()!=DocumentConstants.DotEset ||
                             File.Exists(filePath)==false )
                        {
                            Error( "CONSOLE_MSG_BIN_CONV_ESET_FILE_NOT_FOUND", filePath );
                            continue;
                        }

                        // Add this eset file path to the list.
                        esetFileList.Add( filePath );
                    }
                    else if ( child.Name=="dir" )
                    {
                        // Verify the directory path.
                        filePath = ShellUtility.ToAbsolutePath( child.InnerText, workingPath );
                        if ( Path.IsPathRooted(filePath)==false ||
                             Directory.Exists(filePath)==false )
                        {
                            Error( "CONSOLE_MSG_BIN_CONV_ESET_DIR_NOT_FOUND", filePath );
                            continue;
                        }

                        // Get the eset files in the directory.
                        DirectoryInfo dirInfo      = new DirectoryInfo( filePath );
                        FileInfo[]    fileInfoList = dirInfo.GetFiles( "*.eset",
                                                                       SearchOption.AllDirectories );
                        foreach ( FileInfo fileInfo in fileInfoList )
                        {
                            if ( fileInfo.Exists==false )
                                continue;

                            // Add the eset file path to the list.
                            esetFileList.Add( fileInfo.FullName );
                        }
                    }

                    #endregion
                }
            }

            return true;
        }

        /// <summary>
        /// Helper method for parsing the binary conversion argument XML file.
        /// </summary>
        /// <param name="argFileName">The argument file path.</param>
        /// <param name="workingPath">The working folder path.</param>
        /// <param name="projectName">The project name parsed from the argument file.</param>
        /// <param name="esetFileList">The list of eset file paths parsed from the argument file.</param>
        /// <returns>True on success.</returns>
        private static bool ParseBinConvArgFile( string argFileName,
                                                 string workingPath,
                                                 out string projectName,
                                                 out List<string> esetFileList )
        {
            projectName  = string.Empty;
            esetFileList = null;

            try
            {
                XmlDocument doc = new XmlDocument();
                doc.Load( argFileName );

                XmlElement root = doc.DocumentElement;
                if ( root==null )
                    return false;

                // Parse the child nodes.
                foreach ( XmlNode child in root.ChildNodes )
                {
                    if ( child.Name=="name" )
                    {
                        // Set the project name.
                        projectName = string.Copy( child.InnerText );
                    }
                    else if ( child.Name=="EmitterSetList" )
                    {
                        // Parse the eset file list.
                        if ( ParseEsetFileList(child as XmlElement,
                                               workingPath,
                                               out esetFileList)==false )
                        {
                            return false;
                        }
                    }
                }

                // Have we found the project name?
                // Set the default project name if not.
                if ( string.IsNullOrEmpty(projectName)==true )
                    projectName = "no name";
            }
            catch ( Exception e )
            {
                Console.WriteLine( "Failed parsing the emitter set list XML file." );
                Console.WriteLine( "Error message : " + e.Message );
                return false;
            }

            return true;
        }

        /// <summary>
        /// Log event handler for the converter; output goes to the console
        /// </summary>
        /// <param name="level">Define the importance of the message</param>
        /// <param name="message">Message to display</param>
        private static void LogEventHandler(int level, String message)
        {
            switch (level)
            {
                case 0:
                    Console.WriteLine( "[ERROR] " + message);
                    break;
                case 1:
                    Console.WriteLine( "[WARNING] " + message);
                    break;
                case 2:
                    Console.WriteLine( "[INFO] " + message);
                    break;
                case 3:
                    Console.WriteLine( "[DEBUG] " + message);
                    break;
                default:
                    Console.WriteLine( "[INFO] " + message);
                    break;
            }
        }

        /// <summary>
        /// Console application for converting emitter set files to binary file.
        /// </summary>
        /// <param name="convType">The conversion type.</param>
        /// <param name="esetFileList">The emitter set file list.</param>
        /// <param name="binFileName">The output binary file name.</param>
        /// <param name="projectName">The project name.</param>
        /// <returns>True on success.</returns>
        /*
        public static bool ConvertEsetToBinary( BinaryConversionTypes convType,
                                                List<string> esetFileList,
                                                string binFileName,
                                                string projectName )
        {
            if ( esetFileList==null || esetFileList.Count<=0 )
                return false;

            // The output target file name.
            if ( string.IsNullOrEmpty( binFileName )==true )
                return false;

            ConvertTimer.Start( TimerType.TIMER_TOTAL );
            ConvertTimer.Start( TimerType.TIMER_ESET_LOAD );

            // Collect the eset file paths.
            List<IEmitterSetDocument> esetDocList = new List<IEmitterSetDocument>();

            IProjectDocument project = ProjectManager.ActiveProject;
            string fullPath = string.Empty;
            foreach ( string path in esetFileList )
            {
                // Check the file path and the existence of the emitter set file.
                fullPath = ShellUtility.ToAbsolutePath( path, _convWorkingDir );
                if ( string.IsNullOrEmpty(fullPath)==true )
                    continue;

                if ( Path.GetExtension(fullPath).ToLower()!=DocumentConstants.DotEset )
                    continue;

                if ( File.Exists( fullPath )==false )
                {
                    Error( "CONSOLE_MSG_BIN_CONV_ESET_FILE_NOT_FOUND", fullPath );
                    continue;
                }

                Console.WriteLine( "Loading emitter set file : " + fullPath );

                // Load emitter set from the file.
                EmitterSetDocument esetDoc = DocumentIO.LoadEmitterSetFile( fullPath, project );
                if ( esetDoc==null )
                    continue;

                // Validate each emitters of the emitter set.
                if ( TheApp.UserConfigData!=null )
                {
                    foreach ( IEmitterDocument emitter in esetDoc.EmitterDocuments )
                    {
                        if ( emitter==null )
                            continue;

                        if ( TheApp.UserConfigData.FindDrawPathByText(emitter.EmitterData.UI_drawPath)==null )
                        {
                            Error( "CONSOLE_MSG_INVALID_DRAW_PATH",
                                   esetDoc.Name,
                                   emitter.Name );
                            return false;
                        }
                    }
                }

                // Add the emitter set document to the list.
                esetDocList.Add( esetDoc );
            }

            ConvertTimer.Stop(TimerType.TIMER_ESET_LOAD);

            string saveDir = Path.GetDirectoryName( binFileName );
            if ( Directory.Exists( saveDir )==false )
                Directory.CreateDirectory( saveDir );

            // The application hasn't loaded the user shader definitions yet,
            // force it to load them now.
            TheApp.LoadUserShaderUIDefinitions();

            // Setup the user shader definition path if is specified on command line arguments.
            string origUserShaderPath = TheApp.UserShaderPath;
            if ( string.IsNullOrEmpty(_userShaderPath)==false )
            {
                TheApp.UserShaderPath = _userShaderPath;
                TheApp.OutputLogMsg( NWCore.LogLevels.Warning,
                                     res.Strings.CONSOLE_APP_USE_OVERWRITTEN_CUSTOM_SHADER_PATH,
                                     _userShaderPath );
            }

            PtclConverter converter =
                new PtclConverter( new ConverterContext(false,
                                                        true,
                                                        Config.Data.Application.IsEftCombinerEditorEnabled,
                                                        TheApp.G3dXsdBasePath,
                                                        Config.Data.Option.ShaderDefPath) );
            converter.LogEvent += new NWCore.LogHandler( TheApp.OutputLogMsg );

            Console.WriteLine( "Converting emitter sets to binary file..." );

            if ( _convType==BinaryConversionTypes.Cafe )
            {
                converter.GetConverterContext().enableNativeFormat = true;
                converter.GetConverterContext().isBigEndian        = true;

                // Convert the emitter sets.
                MemoryStream stream = new MemoryStream();
                Console.WriteLine( " WiiU binary : " + binFileName );
                converter.Convert( stream, esetDocList );

                // Get the buffer.
                byte[] buffer = stream.GetBuffer();

                // Write the result to file.
                FileStream file = new FileStream( binFileName, FileMode.Create );
                file.Write(buffer, 0, (int)stream.Length);
                file.Close();
            }
            else if ( _convType==BinaryConversionTypes.CTR )
            {
                converter.GetConverterContext().enableNativeFormat = true;
                converter.GetConverterContext().isBigEndian        = false;

                // Convert the emitter sets.
                MemoryStream stream = new MemoryStream();
                Console.WriteLine(" CTR binary : " + binFileName );
                converter.Convert( stream, esetDocList );

                // Get the buffer.
                byte[] buffer = stream.GetBuffer();

                // Write the result to file.
                FileStream file = new FileStream( binFileName, FileMode.Create );
                file.Write(buffer, 0, (int)stream.Length);
                file.Close();
            }
            else if ( _convType==BinaryConversionTypes.Windows )
            {
                converter.GetConverterContext().enableNativeFormat = false;
                converter.GetConverterContext().isBigEndian        = false;

                // Convert the emitter sets.
                MemoryStream stream = new MemoryStream();
                Console.WriteLine(" Windows binary : " + binFileName );
                converter.Convert( stream, esetDocList );

                // Get the buffer.
                byte[] buffer = stream.GetBuffer();

                // Write the result to file.
                FileStream file = new FileStream( binFileName, FileMode.Create );
                file.Write(buffer, 0, (int)stream.Length);
                file.Close();
            }
            else
            {
                #if BUILD_FOR_CTR

                converter.GetConverterContext().enableNativeFormat = true;
                converter.GetConverterContext().isBigEndian        = false;

                // Convert the emitter sets for CTR.
                MemoryStream stream = new MemoryStream();
                Console.WriteLine(" CTR binary : " + binFileName + "_CTR.ptcl");
                converter.Convert( stream, esetDocList );

                // Get the buffer.
                byte[] buffer = stream.GetBuffer();

                // Write the result to file.
                FileStream file = new FileStream( binFileName + "_CTR.ptcl", FileMode.Create );
                file.Write(buffer, 0, (int)stream.Length);
                file.Close();

                #else

                converter.GetConverterContext().enableNativeFormat = true;
                converter.GetConverterContext().isBigEndian        = true;

                // Convert the emitter sets for Cafe.
                MemoryStream stream = new MemoryStream();
                Console.WriteLine(" WiiU binary : " + binFileName + "_Cafe.ptcl");
                converter.Convert( stream, esetDocList );

                // Get the buffer.
                byte[] buffer = stream.GetBuffer();

                // Write the result to file.
                FileStream file = new FileStream( binFileName + "_Cafe.ptcl", FileMode.Create );
                file.Write(buffer, 0, (int)stream.Length);
                file.Close();

                #endif

                converter.GetConverterContext().enableNativeFormat = false;
                converter.GetConverterContext().isBigEndian        = false;

                // Convert the emitter sets for windows.
                stream = new MemoryStream();
                Console.WriteLine(" Windows binary : " + binFileName + "_Win.ptcl");
                converter.Convert( stream, esetDocList );

                // Get the buffer.
                buffer = stream.GetBuffer();

                // Write the result to file.
                file = new FileStream( binFileName + "_Win.ptcl", FileMode.Create );
                file.Write(buffer, 0, (int)stream.Length);
                file.Close();
            }

            // Create the header file.
            if ( _bOutputHeader==true )
            {
                fullPath = ShellUtility.ToAbsolutePath( _binHeaderName, _convWorkingDir );
                Console.WriteLine( "Creating header file : " + fullPath );

                // Use the converter to generate the header file contents.
                string contents = converter.CreateHeader( esetDocList,
                                                          projectName );
                if ( string.IsNullOrEmpty(contents)==true )
                {
                    Error( "CONSOLE_MSG_BIN_CONV_FAIL_CREATE_HEADER" );
                    return false;
                }

                // Create the file stream for write.
                FileStream file = new FileStream( fullPath, FileMode.Create );

                // Create SJIS text encoder for converting the header contents to byte array.
                byte[] buffer = System.Text.Encoding.GetEncoding("Shift_JIS").GetBytes(contents);

                // Write the encoded contents to file.
                file.Write( buffer, 0, buffer.Length );

                // Close the file.
                file.Close();
            }

            ConvertTimer.Stop( TimerType.TIMER_TOTAL );

            ConvertTimer.Output();
            converter.OutputConvertTime();

            Console.WriteLine( "Successfully converted {0} emitter sets into binary file.",
                               esetDocList.Count );
            converter.Dispose();

            // Reset the original user shader path.
            TheApp.UserShaderPath = origUserShaderPath;

            return true;
        }
        */

        #endregion

        #region Update eset file version

        /// <summary>
        /// Update the version of the given eset files.
        /// </summary>
        /// <param name="filePaths">The file path of the eset files.</param>
        /// <param name="workingPath">The path to the working folder.</param>
        /// <returns>True on success.</returns>
        private static bool UpdateEsetFiles( List<string> filePaths,
                                             string workingPath )
        {
            if ( filePaths==null || filePaths.Count<=0 )
                return false;

            int iNumUpdated = 0;

            DocumentSaver saver = new DocumentSaver();
            saver.EnableMessageBox = false;

            IProjectDocument project = ProjectManager.ActiveProject;
            foreach ( string path in filePaths )
            {
                // Check the file path and the existence of the emitter set file.
                string fullPath = ShellUtility.ToAbsolutePath( path, workingPath );
                if ( string.IsNullOrEmpty(fullPath)==true )
                    continue;

                if ( Path.GetExtension(fullPath).ToLower()!=DocumentConstants.DotEset )
                    continue;

                if ( File.Exists(fullPath)==false )
                {
                    Error( "CONSOLE_MSG_ESET_UPDATER_ESET_FILE_NOT_FOUND", fullPath );
                    continue;
                }

                Console.WriteLine( "Updating emitter set file : " + fullPath );

                // Load emitter set from the file.
                EmitterSetDocument esetDoc = DocumentIO.LoadEmitterSetFile( fullPath, project );

                // Save the emitter set of the updated version.
                saver.Save( esetDoc, fullPath );

                ++iNumUpdated;
            }

            Console.WriteLine( "Successfully updated {0} eset files.",
                               iNumUpdated );

            return true;
        }

        #endregion

        #region Check command line arguments

        /// <summary>
        /// 引数チェック。
        /// </summary>
        public static bool CheckArguments(string[] args)
        {
            int iNumArgs = 0;
            foreach (string arg in args)
            {
                foreach ( BaseArgumentProcessor processor in _argProcessorList )
                {
                    if ( processor.CanProcess( arg )==ArgProcessResult.Success )
                        ++iNumArgs;
                }
            }
            return (iNumArgs>0);
        }

        /// <summary>
        /// 引数の解析
        /// </summary>
        /// <param name="args">The command line arguments.</param>
        /// <returns>True if console application should execute.</returns>
        private static bool analyzeArguments(string[] args)
        {
            BaseArgumentProcessor prevProcessor = null;

            int iNumArgs = 0;
            foreach ( string arg in args )
            {
                ArgProcessResult result = ArgProcessResult.UnknownArgument;
                if ( prevProcessor!=null )
                    result = prevProcessor.Process( arg );

                if (result == ArgProcessResult.RequestNextArgument)
                {
                    ++iNumArgs;
                    continue;
                }

                if (result == ArgProcessResult.ReportEndOfArgument)
                {
                    ++iNumArgs;
                    prevProcessor = null;
                    continue;
                }

                foreach ( BaseArgumentProcessor processor in _argProcessorList )
                {
                    if ( result==ArgProcessResult.UnknownArgument )
                        result = processor.Process( arg );

                    switch ( result )
                    {
                        case ArgProcessResult.Success :
                            prevProcessor = null;
                            ++iNumArgs;
                            break;

                        case ArgProcessResult.Fail:
                            prevProcessor = null;
                            ++iNumArgs;
                            break;

                        case ArgProcessResult.RequestNextArgument :
                            prevProcessor = processor;
                            break;

                        case ArgProcessResult.ReportEndOfArgument :
                            prevProcessor = null;
                            ++iNumArgs;
                            break;
                    }

                    if ( prevProcessor!=null )
                        break;
                }
            }

            if ( iNumArgs>0 )
                return true;

            return false;
        }

        #endregion

        #region Execute console application

        /// <summary>
        /// 実行。
        /// </summary>
        public static int Execute( string[] args )
        {
            OperatingSystem osInfo = Environment.OSVersion;
            if ( (osInfo.Platform==PlatformID.Win32NT) &&
                 (osInfo.Version.CompareTo(new Version( 5, 1, 0 ))>0) )
            {
                // Try to attach the process to parent's console window.
                if ( AttachConsole( ATTACH_PARENT_PROCESS )==false )
                {
                    // Failed attaching to parent's console, allocate one.
                    _consoleEnabled = AllocConsole();
                    _freeConsole    = _consoleEnabled;
                }
                else
                {
                    Console.WriteLine(""); // Just an empty new line
                    _consoleEnabled = true;
                    _freeConsole    = true;
                }
            }

            // 引数の解析
            if ( analyzeArguments(args)==false )
                return 1;

            if (_showHelp)
            {
                ShowHelpPage(_argProcessorList);
                if (_consoleEnabled == true)
                    FreeConsole();
                return 0;
            }

            // Is custom user config file path assigned?
            string origUserConfigPath = TheApp.UserConfigPath;
            if ( string.IsNullOrEmpty(_userConfigPath)==false )
                TheApp.UserConfigPath = _userConfigPath;
            else
                TheApp.UserConfigPath = string.Empty;

            // Load user config file.
            TheApp.LoadUserConfig(false);

            // Execute the emitter set file updater.
            EsetUpdaterArgProcessor esetUpdaterProcessor =
                FindArgumentProcessor( "EsetUpdaterArgProcessor" ) as EsetUpdaterArgProcessor;
            if ( esetUpdaterProcessor!=null &&
                 esetUpdaterProcessor.IsProcessed==true )
            {
                string       projectName;
                List<string> esetFileList;
                if ( ParseBinConvArgFile(esetUpdaterProcessor.EsetListFilePath,
                                         esetUpdaterProcessor.WorkingFolderPath,
                                         out projectName,
                                         out esetFileList)==false )
                {
                    Error( "CONSOLE_MSG_ESET_UPDATER_FAIL_PARSE_PARAM_FILE" );

                    TheApp.UserConfigPath = origUserConfigPath;
                    if ( _consoleEnabled==true )
                        FreeConsole();

                    return 1;
                }

                if ( UpdateEsetFiles(esetFileList,
                                     esetUpdaterProcessor.WorkingFolderPath)==false )
                {
                    Error( "CONSOLE_MSG_ESET_UPDATER_FAIL_UPDATE" );

                    TheApp.UserConfigPath = origUserConfigPath;
                    if ( _consoleEnabled==true )
                        FreeConsole();

                    return 1;
                }
            }

            /*
            if ( string.IsNullOrEmpty(_binConvArgFile)==false &&
                 string.IsNullOrEmpty(_binFileName)==false )
            {
                string       projectName;
                List<string> esetFileList;
                if ( ParseBinConvArgFile( _binConvArgFile,
                                          _convWorkingDir,
                                          out projectName,
                                          out esetFileList )==false )
                {
                    Error( "CONSOLE_MSG_BIN_CONV_FAIL_PARSE_PARAM_FILE" );

                    TheApp.UserConfigPath = origUserConfigPath;
                    if ( _consoleEnabled==true )
                        FreeConsole();

                    return 1;
                }

                if ( ConvertEsetToBinary(_convType,
                                         esetFileList,
                                         _binFileName,
                                         projectName)==false )
                {
                    Error( "CONSOLE_MSG_BIN_CONV_FAIL_CONV_BIN" );

                    TheApp.UserConfigPath = origUserConfigPath;
                    if ( _consoleEnabled==true )
                        FreeConsole();

                    return 1;
                }
            }
            */

            TheApp.UserConfigPath = origUserConfigPath;
            if ( _consoleEnabled==true )
                FreeConsole();

            return 0;
        }

        #endregion

        #region File utility methods

        //---------------------------------------------------------------------
        // ファイルのチェック
        private static bool fileCheck( string arg,
                                       bool bCheckExists,
                                       ref string fileName )
        {
            if ( fileName!=null )
            {
                Error("CONSOLE_FILE_ARGUMENT_REPEATED_MSG");
                return false;
            }

            if ( bCheckExists==true &&
                 File.Exists(arg)==false )
            {
            	Error("CONSOLE_FILE_NOT_FOUND_MSG", arg);
            	return false;
            }

            fileName = arg;

            return true;
        }

        #endregion

        #region Help page display

        private static void ShowHelpPage(IEnumerable<BaseArgumentProcessor> argProcessors)
        {
            if (_consoleEnabled == false)
                return;

            var knownArgProcessors = argProcessors
                .Where(x => x is KnownArgumentProcessor)
                .Select(x => (KnownArgumentProcessor)x);

            var commandLine = string.Format("{0} {1}",
                System.AppDomain.CurrentDomain.FriendlyName,
                string.Join(" ", knownArgProcessors.Select(x => "[" + x.LongUsage() + "]")));
            TheApp.OutputLogMsg(NWCore.LogLevels.Console, commandLine);

            var usageLength = knownArgProcessors.Max(x => x.LongUsage().Length);

            foreach (var argProcessor in knownArgProcessors)
            {
                var helpLine = string.Format("  {0}  {1}", argProcessor.LongUsage().PadRight(usageLength), argProcessor.Description);
                TheApp.OutputLogMsg(NWCore.LogLevels.Console, helpLine);
            }
        }

        #endregion

        #region Error outputting utility methods

        //---------------------------------------------------------------------
        // エラー
        private static void Error(string errorId)
        {
            if (_quiet) { return; }
            // メッセージの構築
            ShowError(StringResource.Get(errorId));

        }

        // エラー
        private static void Error(string errorId, params object[] args)
        {
            if (_quiet) { return; }
            ShowError(StringResource.Get(errorId, args));
        }

        // エラー表示
        private static void ShowError(string message)
        {
            string msg = message;
            // エラーの表示
            if (_consoleEnabled)
            {
                Console.Error.WriteLine(msg);
            }
            else
            {
                /*
                ThreadSafeMsgBox.Show( msg,
                                       res.Strings.ERROR_CAPTION,
                                       System.Windows.Forms.MessageBoxButtons.OK,
                                       System.Windows.Forms.MessageBoxIcon.Error );
                */
            }
        }

        #endregion

        #region C++ console functions DLL imports

        [DllImport("kernel32.dll")]
        private static extern bool AttachConsole( int dwProcessId );
        [DllImport( "kernel32.dll" )]
        private static extern IntPtr GetConsoleWindow();
        [DllImport("kernel32.dll")]
        private static extern bool AllocConsole();
        [DllImport("kernel32.dll")]
        private static extern bool FreeConsole();

        private const int ATTACH_PARENT_PROCESS = -1;

        #endregion

        #region Variables

        private static bool _showHelp = false;
        // 静寂モード
        private static bool _quiet = false;
        // コンソールが有効か
        private static bool _consoleEnabled = false;
        private static bool _freeConsole    = false;

        //private static BinaryConversionTypes _convType       = BinaryConversionTypes.AllPlatforms;
        private static string                _binConvArgFile = string.Empty;
        private static string                _binFileName    = string.Empty;
        private static string                _convWorkingDir = string.Empty;
        private static string                _userShaderPath = string.Empty;
        private static string                _userConfigPath = string.Empty;
        private static string                _binHeaderName  = string.Empty;
        //private static bool                  _bOutputHeader  = false;

        private static List<BaseArgumentProcessor> _argProcessorList = new List<BaseArgumentProcessor>();

        private static Dictionary<uint, BaseArgumentProcessor> _argProcessorMap =
            new Dictionary<uint, BaseArgumentProcessor>();

        #endregion
    }
}
