﻿// ========================================================================
// <copyright file="ProjectManager.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using NWCore.DataModel;
using App.Utility;

namespace App.Data
{
    /// <summary>
    /// プロジェクトマネージャクラス。
    /// </summary>
    public static class ProjectManager
    {
        // ドキュメントリスト
        private static readonly List<IProjectDocument> m_projects = new List<IProjectDocument>();

        // 選択ドキュメント
        private static IProjectDocument m_activeProject = null;

        #region イベント

        /// <summary>
        /// アクティブなプロジェクトが変更されたときにコール
        /// </summary>
        public static event EventHandler ActiveProjectChanged = null;

        /// <summary>
        /// アクティブなイベントが変更されたときにコール
        /// </summary>
        public static event DocumentChangedEventHandler ActiveDocumentChanged = null;

        #endregion

        /// <summary>
        /// 初期化処理。
        /// </summary>
        public static void Initialize()
        {
            m_projects.Clear();
            m_activeProject = null;

            // ＩＤ生成カウンタの初期化です。
            ProjectNameIDGenerator = 1;
            EmitterSetNameIDGenerator = 1;
            EmitterNameIDGenerator = 1;
            ChildNameIDGenerator = 1;
            TextureNameIDGenerator = 1;
            PreviewNameIDGenerator = 1;
        }

        /// <summary>
        /// ドキュメントリスト。
        /// </summary>
        public static ReadOnlyList<IProjectDocument> Projects
        {
            get { return new ReadOnlyList<IProjectDocument>(m_projects); }
        }

        /// <summary>
        /// 保持しているドキュメント数
        /// </summary>
        public static int Count
        {
            get { return m_projects.Count; }
        }

        /// <summary>
        /// アクティブなプロジェクトを設定・取得します。
        /// </summary>
        public static IProjectDocument ActiveProject
        {
            get { return m_activeProject; }
            set
            {
                if (m_activeProject != value)
                {
                    IProjectDocument oldProject = m_activeProject;
                    if (oldProject != null)
                    {
                        oldProject.ActiveDocumentChanged -= ActiveDocumentChanged;
                    }
                    // ログ
                    if (m_projects.Contains(value) == true && value != null)
                    {
                        m_activeProject = value;
                        DebugConsole.WriteLine("selected project:[{0}]", m_activeProject.Name);
                    }
                    else
                    {
                        m_activeProject = null;
                        DebugConsole.WriteLine("selected project:[null]");
                    }
                    // アクティブプロジェクト変更のイベント発行
                    if (ActiveProjectChanged != null)
                    {
                        m_activeProject.ActiveDocumentChanged += ActiveDocumentChanged;
                        ActiveProjectChanged(null, EventArgs.Empty);
                    }
                }
            }
        }

        /// <summary>
        /// アクティブなドキュメントを設定・取得します。
        /// </summary>
        public static IDocument ActiveDocument
        {
            get
            {
                if (ActiveProject != null)
                {
                    return ActiveProject.ActiveDocument;
                }
                return null;
            }
            set
            {
                if (ActiveProject != null)
                {
                    IDocument oldDoc = ActiveProject.ActiveDocument;
                    // アクティブドキュメント変更のイベント発行
                    if (ActiveDocumentChanged != null)
                    {
                        // ドキュメントの切り替え
                        //MainFrame.SynchronizationContext.Send(() => ActiveProject.ActiveDocument = value);
                        //MainFrame.SynchronizationContext.Send(() => ActiveDocumentChanged(null, new DocumentChangedEventArgs(oldDoc, ActiveProject.ActiveDocument)));
                    }
                }
            }
        }

        /// <summary>
        /// ProjectNameIDGenerator
        /// </summary>
        public static int ProjectNameIDGenerator { get; set; }

        /// <summary>
        /// EmitterSetNameIDCount
        /// </summary>
        public static int EmitterSetNameIDGenerator { get; set; }

        /// <summary>
        /// EmitterNameIDGenerator
        /// </summary>
        public static int EmitterNameIDGenerator { get; set; }

        /// <summary>
        /// ChildNameIDGenerator
        /// </summary>
        public static int ChildNameIDGenerator { get; set; }

        /// <summary>
        /// TextureNameIDGenerator
        /// </summary>
        public static int TextureNameIDGenerator { get; set; }

        /// <summary>
        /// PreviewNameIDGenerator
        /// </summary>
        public static int PreviewNameIDGenerator { get; set; }

        /// <summary>
        /// ドキュメントを追加。
        /// </summary>
        public static void Add(IProjectDocument project)
        {
            Debug.Assert(project != null);
            Debug.Assert(!m_projects.Contains(project));

            // プロジェクトリストへ追加します。
            m_projects.Add(project);
#if true //TODO:CUT
            // ドキュメントマネージャにも追加します。
            DocumentManager.Add(project);
#endif
        }

        /// <summary>
        /// ドキュメントを削除。
        /// </summary>
        public static void Remove(IProjectDocument project)
        {
            Debug.Assert(project != null);
            Debug.Assert(m_projects.Contains(project));

            m_projects.Remove(project);

#if true //TODO:CUT
            // ドキュメントマネージャからも追加します。
            DocumentManager.Remove(project);
#endif
            // 選択ドキュメント解除
            if (m_activeProject == project)
            {
                // プロパティ経由でイベント発生させる
                ActiveProject = null;
            }
        }

        /// <summary>
        /// ドキュメントが既にあるかチェックする
        /// </summary>
        public static bool Contains(IProjectDocument project)
        {
            Debug.Assert(project != null);
            return m_projects.Contains(project);
        }

        /// <summary>
        /// 名前で検索をします。
        /// </summary>
        public static IProjectDocument FindByName(string projectName)
        {
            foreach (IProjectDocument document in m_projects)
            {
                if (document.Name.Equals(projectName, StringComparison.InvariantCulture) == true)
                {
                    return document;
                }
            }
            return null;
        }

        /// <summary>
        /// 指定プロジェクト以外のプロジェクトから検索します。
        /// </summary>
        /// <param name="project">除外したいプロジェクト(=null..全対象)</param>
        /// <param name="findItem">検索するEffectDocument</param>
        /// <returns>見つかった</returns>
        public static IProjectDocument[] FindProject(
            IProjectDocument ignoreProject, IEmitterSetDocument findItem)
        {
            List<IProjectDocument> list = new List<IProjectDocument>();
            foreach (IProjectDocument projItem in ProjectManager.Projects)
            {
                if (projItem != ignoreProject || ignoreProject == null)
                {
#if false ////TODO:未対応
                    if (projItem.Contains(findItem) == true)
                    {
                        list.Add(projItem);
                    }
#endif
                }
            }
            return list.ToArray();
        }

        /// <summary>
        /// プロジェクト全体を更新します。
        /// </summary>
        public static void UpdateAll()
        {
#if false
            foreach (IProjectDocument proj in ProjectManager.Projects)
            {
                proj.BuildTreeNodes();
                proj.UpdateTreeNode();
            }
#endif
        }
    }
}
