﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.DataModel.DataModels;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.DataModel.Serializer
{
    /// <summary>
    /// Base class for serializing data models and updating data model from previous version.
    /// </summary>
    public abstract class DataModelSerializerBase
    {
        /// <summary>
        /// Get the version of the data model serializer.
        /// </summary>
        public abstract Version Version { get; }

        /// <summary>
        /// Enumerate the extra data model types for serialization.
        /// </summary>
        public abstract IEnumerable<Type> ExtraDataModelTypes { get; }

        /// <summary>
        /// Get the dictionary that maps the data model types
        /// from the previous to the current version.
        /// </summary>
        protected abstract Dictionary<Type, Type> TypeMap { get; }

        /// <summary>
        /// Get the dictionary that maps the data model types
        /// from the current to the previous version.
        /// </summary>
        protected abstract Dictionary<Type, Type> InverseTypeMap { get; }

        /// <summary>
        /// Serialize the data model to the given XML document.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <param name="xmlDocument">The XML document to serialize to.</param>
        /// <returns>True on success.</returns>
        public bool Serialize(DataModelBase dataModel, XmlDocument xmlDocument)
        {
            SerializationHelper.RequestExtraTypes += this.OnRequestExtraTypes;
            SerializationHelper.SaveXmlDocSerializable(dataModel, xmlDocument);
            SerializationHelper.RequestExtraTypes -= this.OnRequestExtraTypes;
            return true;
        }

        /// <summary>
        /// Deserialize the data model from the XML document.
        /// </summary>
        /// <param name="type">The type of object to deserialize.</param>
        /// <param name="xmlDocument">The XML document to deserialize from.</param>
        /// <returns>The deserialized data model.</returns>
        public DataModelBase Deserialize(Type type, XmlDocument xmlDocument)
        {
            SerializationHelper.RequestExtraTypes += this.OnRequestExtraTypes;
            var result = SerializationHelper.LoadXmlDocSerializable(type, xmlDocument, true) as DataModelBase;
            SerializationHelper.RequestExtraTypes -= this.OnRequestExtraTypes;

            return result;
        }

        /// <summary>
        /// Serialize the data model to the given XML writer.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <param name="writer">The XML writer.</param>
        /// <returns>True on success.</returns>
        public bool SerializeWithXmlSerializer(DataModelBase dataModel, XmlWriter writer)
        {
            SerializationHelper.RequestExtraTypes += this.OnRequestExtraTypes;
            SerializationHelper.Save(dataModel.GetType(), dataModel, writer, true);
            SerializationHelper.RequestExtraTypes -= this.OnRequestExtraTypes;
            return true;
        }

        /// <summary>
        /// Deserialize the data model from the XML reader.
        /// </summary>
        /// <param name="type">The type of object to deserialize.</param>
        /// <param name="reader">The XML reader to deserialize from.</param>
        /// <returns>The deserialized data model.</returns>
        public DataModelBase DeserializeWithXmlSerializer(Type type, XmlReader reader)
        {
            SerializationHelper.RequestExtraTypes += this.OnRequestExtraTypes;
            var result = SerializationHelper.Load(type, reader, true) as DataModelBase;
            SerializationHelper.RequestExtraTypes -= this.OnRequestExtraTypes;

            return result;
        }

        /// <summary>
        /// Create the data model and convert data from the one of the previous version.
        /// The new data model is of the current version.
        /// </summary>
        /// <param name="prevVersionDataModel">The source data model of the previous version.</param>
        /// <returns>The new upgraded data model.</returns>
        public DataModelBase UpgradeDataModel(DataModelBase prevVersionDataModel)
        {
            // Find the corresponding type at the current version.
            Type currType = this.FindCurrentType(prevVersionDataModel.GetType());
            if (currType == null)
            {
                return null;
            }

            // Create the instance and convert data from the previous version.
            var currVersionDataModel = Activator.CreateInstance(currType) as DataModelBase;
            currVersionDataModel.ConvertFromPreviousVersion(prevVersionDataModel);

            return currVersionDataModel;
        }

        /// <summary>
        /// Find the type of the current version with the type of the previous version.
        /// </summary>
        /// <param name="previousVersionType">The type of the previous version.</param>
        /// <returns>The type of the current version or null if not found.</returns>
        public Type FindCurrentType(Type previousVersionType)
        {
            Type currentVersionType;
            if (this.TypeMap.TryGetValue(previousVersionType, out currentVersionType) == false)
            {
                return null;
            }

            return currentVersionType;
        }

        /// <summary>
        /// Find the type of the previous version with the type of the current version.
        /// </summary>
        /// <param name="currentVersionType">The type of the current version.</param>
        /// <returns>The type of the previous version or null if not found.</returns>
        public Type FindPreviousType(Type currentVersionType)
        {
            Type previousVersionType;
            if (this.InverseTypeMap.TryGetValue(currentVersionType, out previousVersionType) == false)
            {
                return null;
            }

            return previousVersionType;
        }

        /// <summary>
        /// Handle RequestExtraTypes event from SerializationHelper.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnRequestExtraTypes(object sender, RequestExtraTypesEventArgs e)
        {
            e.AddExtraTypes(this.ExtraDataModelTypes);
        }
    }
}
