﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.DataModel.AnimationTable
{
    /// <summary>
    /// Class to store a key frame data.
    /// </summary>
    public class KeyFrameData : IXmlSerializable,
                                ICloneable,
                                ISettable,
                                IEquatable<object>
    {
        /// <summary>The value for the key frame.</summary>
        private readonly Vector4f value = new Vector4f(0.0f, 0.0f, 0.0f, 0.0f);

        /// <summary>The frame number of the key frame.</summary>
        private int frame = 0;

        /// <summary>The flag indicating whether this key frame is selected.</summary>
        private bool isSelected = false;

        /// <summary>The tuple object to help up with generating hash code.</summary>
        private Tuple<int, float, float, float, float> hashHelper = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public KeyFrameData()
        {
            this.UpdateHashHelper();
        }

        /// <summary>
        /// Value setting constructor.
        /// </summary>
        /// <param name="frame">The frame number of the key frame.</param>
        /// <param name="value">The value of the key frame.</param>
        public KeyFrameData(int frame, Vector4f value)
        {
            this.frame = frame;
            this.value.Set(value);

            this.UpdateHashHelper();
        }

        /// <summary>
        /// Value setting constructor.
        /// </summary>
        /// <param name="frame">The frame number of the key frame.</param>
        /// <param name="x">The value X of the key frame.</param>
        /// <param name="y">The value Y of the key frame.</param>
        /// <param name="z">The value Z of the key frame.</param>
        /// <param name="w">The value W of the key frame.</param>
        public KeyFrameData(int frame, float x, float y, float z, float w)
        {
            this.frame = frame;
            this.value.Set(x, y, z, w);

            this.UpdateHashHelper();
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source key frame data to copy from.</param>
        public KeyFrameData(KeyFrameData src)
        {
            this.Set(src);
        }

        /// <summary>
        /// Get or set the owner of the key frame.
        /// </summary>
        public AnimationTableData Owner { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the key frame is selected or not.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    if (value == true && this.Owner != null)
                    {
                        // Tell the owner that the key frame is selected
                        // so that the owner can handle single selection
                        // correctly.
                        this.Owner.NotifyKeyFrameSelected(this);
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the frame number (time) of the key frame.
        /// </summary>
        public int Frame
        {
            get
            {
                return this.frame;
            }

            set
            {
                this.frame = value;
                this.UpdateHashHelper();

                if (this.Owner != null)
                {
                    this.Owner.SortKeyFrames();
                }
            }
        }

        /// <summary>
        /// Get or set the value of the key frame.
        /// </summary>
        public Vector4f Value
        {
            get
            {
                return this.value.Clone() as Vector4f;
            }

            set
            {
                this.value.Set(value);
                this.UpdateHashHelper();
            }
        }

        /// <summary>
        /// Get or set the color of the key frame.
        /// </summary>
        public ColorRgba ColorValue
        {
            get
            {
                return new ColorRgba(this.Value.X, this.Value.Y, this.Value.Z, this.Value.W);
            }

            set
            {
                this.Value = new Vector4f(value.R, value.G, value.B, value.A);
            }
        }

        /// <summary>
        /// Get or set the value as a ColorRgba, instead of a Vector4f.
        /// </summary>
        public ColorRgba ValueAsColor
        {
            get { return new ColorRgba(this.value.X, this.value.Y, this.value.Z, this.value.W); }
            set { this.Value = new Vector4f(value.R, value.G, value.B, value.A); }
        }

        /// <summary>
        /// Compare with the given object and check if they are equal.
        /// </summary>
        /// <param name="other">The object to compare with.</param>
        /// <returns>True if the instances are equal.</returns>
        public override bool Equals(object other)
        {
            var otherKey = other as KeyFrameData;
            if (otherKey == null)
            {
                return false;
            }

            return (this.frame == otherKey.frame) && (this.value.Equals(otherKey.value) == true);
        }

        /// <summary>
        /// Get hash code of this key frame data.
        /// </summary>
        /// <returns>The hash code according to the frame number and value.</returns>
        public override int GetHashCode()
        {
            return this.hashHelper.GetHashCode();
        }

        /// <summary>
        /// Set data.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public bool Set(object src)
        {
            var srcKeyFrame = src as KeyFrameData;
            if (srcKeyFrame == null)
            {
                return false;
            }

            this.isSelected = srcKeyFrame.isSelected;
            this.frame = srcKeyFrame.frame;
            this.value.Set(srcKeyFrame.value);

            this.UpdateHashHelper();

            //// Do not set owner, it's not supposed to be changed.

            return true;
        }

        /// <summary>
        /// Clone this key frame data.
        /// </summary>
        /// <returns>The create key frame data.</returns>
        public object Clone()
        {
            return new KeyFrameData(this);
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            var lineInfo = reader as IXmlLineInfo;

            bool tmpSelected;
            string strSelected = reader.GetAttribute("IsSelected");
            if (string.IsNullOrEmpty(strSelected) == false &&
                bool.TryParse(strSelected, out tmpSelected) == true)
            {
                this.isSelected = tmpSelected;
            }
            else
            {
                this.isSelected = false;
            }

            int tmpFrame;
            string strFrame = reader.GetAttribute("Frame");
            if (string.IsNullOrEmpty(strFrame) == true ||
                int.TryParse(strFrame, out tmpFrame) == false)
            {
                throw this.GenerateReadXmlException(
                    "The XML node must contain an integer 'Frame' attribute.",
                    lineInfo);
            }

            this.frame = tmpFrame;

            string strValue = reader.GetAttribute("Value");
            if (string.IsNullOrEmpty(strValue) == true)
            {
                throw this.GenerateReadXmlException(
                    "The XML node must contain a 'Value' attribute with 4 comma-separated floating point values.",
                    lineInfo);
            }

            // Split the comma-separated string.
            string[] tokens = strValue.Split(',');

            for (int i = 0; (i < this.value.Count) && (i < tokens.Length); ++i)
            {
                // Parse the elements and store them.
                float value;
                if (float.TryParse(tokens[i].Trim(), out value) == false)
                {
                    throw this.GenerateReadXmlException(
                        "The XML node must contain a 'Value' attribute with 4 comma-separated floating point values.",
                        lineInfo);
                }

                this.value[i] = value;
            }

            this.UpdateHashHelper();
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            writer.WriteAttributeString("IsSelected", this.isSelected.ToString());
            writer.WriteAttributeString("Frame", this.frame.ToString());
            writer.WriteAttributeString("Value", this.value.ToString());
        }

        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// Generate exception for ReadXml().
        /// </summary>
        /// <param name="message">The message for the exception.</param>
        /// <param name="lineInfo">The line info.</param>
        /// <returns>The exception.</returns>
        private XmlException GenerateReadXmlException(string message, IXmlLineInfo lineInfo)
        {
            if (lineInfo != null)
            {
                return new XmlException(message, null, lineInfo.LineNumber, lineInfo.LinePosition);
            }
            else
            {
                return new XmlException(message);
            }
        }

        /// <summary>
        /// Update values to the hash helper.
        /// </summary>
        private void UpdateHashHelper()
        {
            this.hashHelper = Tuple.Create(
                this.frame,
                this.value.X,
                this.value.Y,
                this.value.Z,
                this.value.W);
        }
    }
}
