﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;
using EffectMaker.BusinessLogic.Commands;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class that handles the Workspace button clicks.
    /// </summary>
    public class WorkspaceController
    {
        /// <summary>
        /// Stores the WorkspaceRootViewModel instance.
        /// </summary>
        private WorkspaceRootViewModel workspaceRoot;

        /// <summary>
        /// Initializes the WorkspaceController instance.
        /// </summary>
        /// <param name="workspaceRoot">The WorkspaceRootViewModel instance.</param>
        public WorkspaceController(WorkspaceRootViewModel workspaceRoot)
        {
            if (workspaceRoot == null)
            {
                throw new ArgumentNullException("workspaceRoot");
            }

            this.workspaceRoot = workspaceRoot;
        }

        /// <summary>
        /// PlaybackControlのBackボタンが押されたときの処理を行います。
        /// </summary>
        public void PlaybackBack()
        {
            ViewerController.Instance.SendResetFrame();
        }

        /// <summary>
        /// PlaybackControlのStopボタンが押されたときの処理を行います。
        /// </summary>
        public void PlaybackStop()
        {
            ViewerController.Instance.SendStop();
        }

        /// <summary>
        /// PlaybackControlのPlayボタンが押されたときの処理を行います。
        /// </summary>
        public void PlaybackPlay()
        {
            ViewerController.Instance.SendPause();
        }

        /// <summary>
        /// PlaybackControlのStepボタンが押されたときの処理を行います。
        /// </summary>
        public void PlaybackStep()
        {
            ViewerController.Instance.SendStepFrame();
        }

        /// <summary>
        /// PlaybackControlのFadeボタンが押されたときの処理を行います。
        /// </summary>
        public void PlaybackFade()
        {
            ViewerController.Instance.SendForceFade();
        }

        /// <summary>
        /// PlaybackControlのLoopボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="isLoop">ループ再生する場合はtrueを設定.</param>
        /// <param name="frame">再生するフレーム.</param>
        public void PlaybackLoop(bool isLoop, int frame)
        {
            ViewerController.Instance.SendSetFrameRange(0, frame, isLoop);
        }

        /// <summary>
        /// Called when the 'New EmitterSet' button is clicked.
        /// </summary>
        public void CreateNewEmitterSet()
        {
            WorkspaceViewModel workspace = this.GetWorkspaceViewModel();

            if (workspace != null && workspace is WorkspaceViewModel)
            {
                workspace.OnCreateNewEmitterSet(0);
            }
        }

        /// <summary>
        /// 同一階層にエミッタを作成する.
        /// </summary>
        public void CreateNewEmitter()
        {
            WorkspaceViewModel workspace = this.GetWorkspaceViewModel();

            if (workspace != null && workspace is WorkspaceViewModel)
            {
                workspace.OnCreateNewEmitter(0);
            }
        }

        /// <summary>
        /// Called when the 'New Emitter' button is clicked.
        /// </summary>
        public void CreateNewChildEmitter()
        {
            WorkspaceViewModel workspace = this.GetWorkspaceViewModel();

            if (workspace != null && workspace is WorkspaceViewModel)
            {
                workspace.OnCreateNewChildEmitter(0);
            }
        }

        /// <summary>
        /// エミッタの階層化を解除します。
        /// </summary>
        public void UnparentEmitter()
        {
            WorkspaceViewModel workspace = this.GetWorkspaceViewModel();

            if (workspace != null && workspace is WorkspaceViewModel)
            {
                workspace.OnUnparentEmitter();
            }
        }

        /// <summary>
        /// フィールドの作成.
        /// </summary>
        /// <param name="type">フィールドタイプ.</param>
        public void CreateNewField(FieldTypes type)
        {
            var viewModel = this.GetSelectedOrParentEmitterViewModel();
            if (viewModel == null)
            {
                return;
            }

            // 選択ノードによって作成方法を分岐
            var selected = this.GetSelectedViewModel();
            if (selected is FieldViewModel)
            {
                // フィールドノードが選択されていたら、そのノードの後ろに挿入して作成
                CommandManager.Execute(new CreateFieldCommand(viewModel, type, viewModel.Children.IndexOf(selected)));
            }
            else
            {
                // それ以外のノードだったら普通に作成してフィールド内の末尾に追加作成
                CommandManager.Execute(new CreateFieldCommand(viewModel, type));
            }
        }

        /// <summary>
        /// カスタムアクションを作成する.
        /// </summary>
        public void CreateNewCustomAction()
        {
            var viewModel = this.GetSelectedOrParentEmitterViewModel();
            if (viewModel == null)
            {
                return;
            }

            // すでにカスタムアクションが存在する場合、作成しない。
            if (viewModel.Proxy.CustomActionData != null)
            {
                return;
            }

            CommandManager.Execute(new CreateCustomActionCommand(viewModel));
        }

        /// <summary>
        /// エミッタ拡張パラメータを作成する.
        /// </summary>
        public void CreateNewEmitterExtParams()
        {
            var viewModel = this.GetSelectedOrParentEmitterViewModel();
            if (viewModel == null)
            {
                return;
            }

            // すでにエミッタ拡張パラメータが存在する場合、作成しない。
            if (viewModel.Proxy.EmitterExtParams != null)
            {
                return;
            }

            CommandManager.Execute(new CreateEmitterExtParamsCommand(viewModel));
        }

        /// <summary>
        /// Called when the 'New Preview' button is clicked.
        /// </summary>
        public void CreateNewPreview()
        {
            WorkspaceViewModel workspace = this.GetWorkspaceViewModel();

            if (workspace != null && workspace is WorkspaceViewModel)
            {
                workspace.OnCreateNewPreview();
            }
        }

        /// <summary>
        /// Create reserved shader.
        /// </summary>
        /// <param name="parameter">The parameter as the user data info of the reserved shader.</param>
        public void CreateNewReservedShader(object parameter)
        {
            var viewModel = this.GetSelectedOrParentEmitterViewModel();
            if (viewModel == null)
            {
                return;
            }

            var userDataInfo = parameter as BusinessLogic.UserData.UserDataInfo;
            if (userDataInfo == null)
            {
                return;
            }

            // Do not create it if there is already a reserved shader.
            if (viewModel.Proxy.ReservedShader != null)
            {
                return;
            }

            CommandManager.Execute(new CreateReservedShaderCommand(viewModel, userDataInfo));
        }

        /// <summary>
        /// Called when the 'Remove' button is clicked.
        /// </summary>
        public void Remove()
        {
            HierarchyViewModel viewModel = this.GetSelectedViewModel();
            var parent = viewModel.Parent as WorkspaceNodeViewModelBase;

            if (viewModel == null || parent == null)
            {
                return;
            }

            if (viewModel is EmitterSetViewModel)
            {
                // エミッタセットを削除
                CommandManager.Execute(new RemoveEmitterSetCommand((EmitterSetViewModel)viewModel));
            }
            else if (viewModel is EmitterViewModel)
            {
                // エミッタを削除
                CommandManager.Execute(new RemoveEmitterCommand((EmitterViewModel)viewModel));
            }
            else if (viewModel is PreviewViewModel)
            {
                // プレビューを削除.
                CommandManager.Execute(new RemovePreviewCommand((PreviewViewModel)viewModel));
            }
            else if (viewModel is FieldViewModel)
            {
                // フィールドを削除.
                CommandManager.Execute(new RemoveFieldCommand((FieldViewModel)viewModel));
            }
            else if (viewModel is CustomActionViewModel)
            {
                // カスタムアクションを削除.
                CommandManager.Execute(new RemoveCustomActionCommand((CustomActionViewModel)viewModel));
            }
            else if (viewModel is EmitterExtParamsViewModel)
            {
                // エミッタ拡張パラメータを削除.
                CommandManager.Execute(new RemoveEmitterExtParamsCommand((EmitterExtParamsViewModel)viewModel));
            }
            else if (viewModel is ReservedShaderNodeViewModel)
            {
                // エミッタプラグインを削除.
                CommandManager.Execute(new RemoveReservedShaderCommand((ReservedShaderNodeViewModel)viewModel));
            }

            this.workspaceRoot.UpdateUIStates();
        }

        /// <summary>
        /// 機能の有効 / 無効を切り替えます.
        /// </summary>
        public void SwitchEnableConvert()
        {
            ViewModelBase selectedViewModel = this.GetSelectedViewModel();

            // エミッタを選択しているときtrue
            var emitterViewModel = selectedViewModel as EmitterViewModel;
            if (emitterViewModel != null)
            {
                emitterViewModel.EnableConvert = !emitterViewModel.EnableConvert;
                return;
            }

            // フィールドを選択しているときtrue
            var fieldViewModel = selectedViewModel as FieldViewModel;
            if (fieldViewModel != null)
            {
                fieldViewModel.EnableConvert = !fieldViewModel.EnableConvert;
                return;
            }

            // カスタムアクションを選択しているときtrue
            var customActionViewModel = selectedViewModel as CustomActionViewModel;
            if (customActionViewModel != null)
            {
                customActionViewModel.EnableConvert = !customActionViewModel.EnableConvert;
                return;
            }

            // エミッタプラグインを選択しているときtrue
            var pluginViewModel = selectedViewModel as ReservedShaderNodeViewModel;
            if (pluginViewModel != null)
            {
                pluginViewModel.EnableConvert = !pluginViewModel.EnableConvert;
                return;
            }
        }

        /// <summary>
        /// Copy the selected node to clipboard.
        /// </summary>
        public void CopySelectedNode()
        {
            var viewModel = this.GetSelectedViewModel() as WorkspaceNodeViewModelBase;

            if (viewModel == null)
            {
                return;
            }

            IExecutable exec = viewModel.NodeCopyExecutable;
            if (exec == null)
            {
                return;
            }

            if (exec.CanExecute(null))
            {
                exec.Execute(null);
                this.workspaceRoot.RaiseCopyPastePropertyChanged();
            }
        }

        /// <summary>
        /// Paste the selected node to clipboard.
        /// </summary>
        public void PasteSelectedNode()
        {
            var viewModel = this.GetSelectedViewModel() as WorkspaceNodeViewModelBase;

            if (viewModel == null)
            {
                return;
            }

            IExecutable exec = viewModel.NodePasteExecutable;
            if (exec == null)
            {
                return;
            }

            if (exec.CanExecute(null))
            {
                exec.Execute(null);
                this.workspaceRoot.RaiseCopyPastePropertyChanged();
            }
        }

        /// <summary>
        /// 選択されているノードの子ノードとしてペーストする.
        /// </summary>
        public void PasteChildOfSelectedNode()
        {
            var viewModel = this.GetSelectedViewModel() as WorkspaceNodeViewModelBase;

            if (viewModel == null)
            {
                return;
            }

            IExecutable exec = viewModel.NodePasteAsChildExecutable;
            if (exec == null)
            {
                return;
            }

            if (exec.CanExecute(null))
            {
                exec.Execute(null);
            }
        }

        /// <summary>
        /// Duplicates the selected node.
        /// </summary>
        public void DuplicateSelectedNode()
        {
            var viewModel = this.GetSelectedViewModel() as WorkspaceNodeViewModelBase;

            if (viewModel == null)
            {
                return;
            }

            if (viewModel is FieldViewModel)
            {
                return;
            }

            IExecutable exec = viewModel.NodeDuplicateExecutable;
            if (exec == null)
            {
                return;
            }

            if (exec.CanExecute(null))
            {
                exec.Execute(null);
            }
        }

        /// <summary>
        /// Get the work space view model.
        /// </summary>
        /// <returns>Returns the work space view model, or null otherwise.</returns>
        private WorkspaceViewModel GetWorkspaceViewModel()
        {
            if (this.workspaceRoot.Children.IsNullOrEmpty())
            {
                return null;
            }

            var ws = (WorkspaceViewModel)this.workspaceRoot.Children
                .First(ho => ho is WorkspaceViewModel);

            return ws;
        }

        /// <summary>
        /// Retrieve the currently selected view model.
        /// </summary>
        /// <returns>Returns the currently selected view model, or null otherwise.</returns>
        private HierarchyViewModel GetSelectedViewModel()
        {
            return this.workspaceRoot.SelectedNodeViewModel as HierarchyViewModel;
        }

        /// <summary>
        /// 選択したノードが所属するエミッタビューモデルを返します。
        /// </summary>
        /// <returns></returns>
        private EmitterViewModel GetSelectedOrParentEmitterViewModel()
        {
            var selected = this.workspaceRoot.SelectedNodeViewModel as HierarchyViewModel;
            if (selected == null)
            {
                return null;
            }

            var result = selected as EmitterViewModel;
            if (result != null)
            {
                return result;
            }

            return selected.Parent as EmitterViewModel;
        }
    }
}
