﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the ViewerBasicPtclViewModel.
    /// </summary>
    public class ViewerBasicPtclViewModel : PropertyGroupViewModel<ViewerBasicPtclData>, IModificationFlagOwner
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ViewerBasicPtclViewModel(HierarchyViewModel parent, ViewerBasicPtclData dataModel)
            : base(parent, dataModel)
        {
            this.OnReloadPtclFileExecutable = new AnonymousExecutable(this.OnReloadPtclFile);
            this.OnPrevIndexExecutable = new AnonymousExecutable(this.OnPrevIndex);
            this.OnNextIndexExecutable = new AnonymousExecutable(this.OnNextIndex);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// PTCLファイルをリロードするためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnReloadPtclFileExecutable { get; private set; }

        /// <summary>
        /// 再生するエミッタセットのインデックスを戻すIExecutableを取得します。
        /// </summary>
        public IExecutable OnPrevIndexExecutable { get; private set; }

        /// <summary>
        /// 再生するエミッタセットのインデックスを送るIExecutableを取得します。
        /// </summary>
        public IExecutable OnNextIndexExecutable { get; private set; }

        /// <summary>
        /// ファイルパスを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public string FilePath
        {
            get
            {
                return this.GetDataModelValue<string>(() => this.FilePath);
            }

            set
            {
                // 転送するのに適しているかチェックする
                if (this.CheckPtclFile(value) == false)
                {
                    return;
                }

                if (this.SetDataModelValue(value, () => this.FilePath))
                {
                    // ビューアノードをリロード
                    var viewer = GetParent<ViewerViewModel>(this);
                    ViewerMessageHelper.SendViewer(viewer.DataModel);
                }

                WorkspaceRootViewModel.Instance.UpdateCloseAllState();
            }
        }

        /// <summary>
        /// ptclファイルがViewerに転送するのに適しているかチェックします.
        /// </summary>
        /// <param name="filePath">ptclファイルのパス</param>
        /// <returns></returns>
        private bool CheckPtclFile(string filePath)
        {
            // ファイルパスが空ならそもそもチェックする必要がない
            if (filePath.IsNullOrEmpty() == true)
            {
                return true;
            }

            try
            {
                int result = BinaryFileHeader.IsSuitableHeader(filePath);
                if (result == 1 || result == 3)
                {
                    // ヘッダ不一致
                    Logger.Log("LogView", LogLevels.Error, Properties.Resources.WarningNotPtclFile);
                    return false;
                }

                if (result == 2)
                {
                    // バイナリバージョン不一致
                    Logger.Log("LogView", LogLevels.Error, Properties.Resources.WarningPtclBinaryVersion);
                    return false;
                }
            }
            catch (Exception)
            {
                Logger.Log("LogView", LogLevels.Error, Properties.Resources.WarningPtclFilePathCheck, filePath);
                return false;
            }

            return true;
        }

        /// <summary>
        /// PTCLファイルのリロード処理を行います。
        /// エディタはリロード呼び出しのみを行い、実際のリロードはビューア側で処理します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnReloadPtclFile(object parameter)
        {
            // ReloadCtrlの変更は履歴に積まない
            int reloadCtrl = this.GetDataModelValue<int>(() => this.DataModel.ReloadCtrl);
            ++reloadCtrl;

            var binder = new EffectMakerSetMemberBinder("ReloadCtrl", false, false);
            this.TrySetMember(binder, reloadCtrl);

            // 転送するのに適しているかチェックする
            if (this.CheckPtclFile(this.FilePath) == false)
            {
                return;
            }

            // ビューアノードをリロード
            var viewer = GetParent<ViewerViewModel>(this);
            ViewerMessageHelper.SendViewer(viewer.DataModel);
        }

        /// <summary>
        /// 再生するエミッタセットのインデックスを戻します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnPrevIndex(object parameter)
        {
            // エミッタセットのインデックスを戻す
            // EsetIndexCtrlの変更は履歴に積まない
            int indexCtrl = this.GetDataModelValue<int>(() => this.DataModel.EsetIndexCtrl);
            --indexCtrl;

            var binder = new EffectMakerSetMemberBinder("EsetIndexCtrl", false, false);
            this.TrySetMember(binder, indexCtrl);

            // ビューアノードをリロード
            var viewer = GetParent<ViewerViewModel>(this);
            ViewerMessageHelper.SendViewer(viewer.DataModel);
        }

        /// <summary>
        /// 再生するエミッタセットのインデックスを送ります。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnNextIndex(object parameter)
        {
            // エミッタセットのインデックスを送る
            // EsetIndexCtrlの変更は履歴に積まない
            int indexCtrl = this.GetDataModelValue<int>(() => this.DataModel.EsetIndexCtrl);
            ++indexCtrl;

            var binder = new EffectMakerSetMemberBinder("EsetIndexCtrl", false, false);
            this.TrySetMember(binder, indexCtrl);

            // ビューアノードをリロード
            var viewer = GetParent<ViewerViewModel>(this);
            ViewerMessageHelper.SendViewer(viewer.DataModel);
        }
    }
}
