﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the PreviewData.
    /// </summary>
    [Tag("WorkspaceNodeViewModelKind")]
    public class StripeViewModel : WorkspaceNodeViewModelBase<StripeData>, IModificationFlagOwner
    {
        /// <summary>
        /// ストライプ表示名(ノード).
        /// </summary>
        private string name;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public StripeViewModel(HierarchyViewModel parent, StripeData dataModel)
            : base(parent, dataModel)
        {
            this.UpdateUserData(true);

            this.name = Properties.Resources.StripeTreeNodeName;

            // ユーザーデータのアップデート通知イベントを登録.
            UserDataManager.UserDataUpdated += this.OnUserDataUpdated;

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// 連結式ストライプかチェックする.
        /// </summary>
        public bool IsStripeComplex
        {
            get
            {
                return this.DataModel is StripeComplexData;
            }
        }

        /// <summary>
        /// 履歴式ストライプかチェックする.
        /// </summary>
        public bool IsStripeHistory
        {
            get
            {
                return this.DataModel is StripeHistoryData;
            }
        }

        /// <summary>
        /// スーパーストライプかチェックする.
        /// </summary>
        public bool IsStripeSuper
        {
            get
            {
                return this.DataModel is StripeSuperData;
            }
        }

        /// <summary>
        /// ストライプ表示名.
        /// </summary>
        public string DisplayName
        {
            get
            {
                return this.name;
            }

            set
            {
                this.name = value;
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel
        {
            get;
            private set;
        }

        /// <summary>
        /// The default property page view model to use
        /// on the first time the workspace node is selected.
        /// </summary>
        public override HierarchyViewModel DefaultPropertyPageViewModel
        {
            get
            {
                if (this.Children.Count > 0)
                {
                    return this.Children[0] as HierarchyViewModel;
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// Get the selected item.
        /// </summary>
        public HierarchyViewModel SelectedItem { get; set; }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            base.Dispose();

            // ユーザーデータアップデートの通知イベントの解除.
            UserDataManager.UserDataUpdated -= this.OnUserDataUpdated;
        }

        /// <summary>
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new StripeDataProxy(dataModel);
        }

        /// <summary>
        /// Handle UserDataUpdated event from the user data manager.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnUserDataUpdated(
            object sender,
            UserDataUpdatedEventArgs e)
        {
            if (e.IsUserDataTypeUpdated(UpdatedUserDataTypes.GeneralUserPage) == false)
            {
                // The updated user data is not for stripe data, bail out.
                return;
            }

            this.UpdateUserData(false);
        }

        /// <summary>
        /// Helper method for updating user data.
        /// </summary>
        /// <param name="onlyUpdateViewModels">
        /// True to update only the view models for to the existing user data models.
        /// Otherwise, remove all the user data models and update them.
        /// </param>
        private void UpdateUserData(bool onlyUpdateViewModels)
        {
            // First remove the user data view model.
            var userPages = this.Children.OfType<StripeUserDataViewModel>().ToArray();
            foreach (var userPage in userPages)
            {
                this.Children.Remove(userPage);
            }

            if (onlyUpdateViewModels == false)
            {
                // Update user data models.
                (this.Proxy as StripeDataProxy).UpdateUserData();
            }

            UserDataBase dm = null;
            if (this.DataModel is StripeComplexData)
            {
                dm = ((StripeComplexData)this.DataModel).UserPageData;
            }
            else if (this.DataModel is StripeHistoryData)
            {
                dm = ((StripeHistoryData)this.DataModel).UserPageData;
            }
            else if (this.DataModel is StripeSuperData)
            {
                dm = ((StripeSuperData)this.DataModel).UserPageData;
            }
            else
            {
                Logger.Log(LogLevels.Error, "StripeViewModel.UpdateUserData : The data model type is wrong.");
                return;
            }

            // Create user page view models with the new user data models.
            var userViewModel = new StripeUserDataViewModel(this, dm);
            this.Children.Add(userViewModel);
        }
    }
}
