﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the CustomActionUserDataViewModel.
    /// </summary>
    public class EmitterExtParamsUserDataGroupViewModel : PropertyGroupViewModel<EmitterExtParams>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "AvailableSettingViewModels",
            "SelectedSettingViewModel",
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterExtParamsUserDataGroupViewModel(
            HierarchyViewModel parent,
            EmitterExtParams dataModel)
            : base(parent, dataModel)
        {
            // Generate view model for the Emitter Extension Parameters user data models.
            this.UpdateUserData(true);

            UserDataManager.UserDataUpdated += this.OnUserDataUpdated;

            // Always create the modification flag view model IN THE END of the constructor,
            // in case any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// Get the available Emitter Extension Parameters setting view models.
        /// </summary>
        public IEnumerable<object> AvailableSettingViewModels
        {
            get
            {
                return from ch in this.Children
                       where ch is EmitterExtParamsUserSettingDataViewModel
                       select (EmitterExtParamsUserSettingDataViewModel)ch;
            }
        }

        /// <summary>
        /// Get the selected Emitter Extension Parameters setting data view model.
        /// </summary>
        public object SelectedSettingViewModel
        {
            get
            {
                foreach (IHierarchyObject child in this.Children)
                {
                    var vm = child as EmitterExtParamsUserSettingDataViewModel;
                    if (vm != null && vm.DataModel.Parent == this.DataModel)
                    {
                        return vm;
                    }
                }

                return null;
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// Dispose this object.
        /// </summary>
        public override void Dispose()
        {
            UserDataManager.UserDataUpdated -= this.OnUserDataUpdated;

            base.Dispose();
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            this.UpdateUserData(true);
            base.UpdateChildViewModels();
        }

        /// <summary>
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new EmitterExtParamsProxy(dataModel);
        }

        /// <summary>
        /// Handle UserDataUpdated event from the user data manager.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnUserDataUpdated(
            object sender,
            UserDataUpdatedEventArgs e)
        {
            if (e.IsUserDataTypeUpdated(UpdatedUserDataTypes.EmitterExtParams) == false)
            {
                // The updated user data is not for custom shader, bail out.
                return;
            }

            this.UpdateUserData(false);
        }

        /// <summary>
        /// Helper method for updating user data.
        /// </summary>
        /// <param name="onlyUpdateViewModels">
        /// True to update only the view models for to the existing user data models.
        /// Otherwise, remove all the user data models and update them.
        /// </param>
        private void UpdateUserData(bool onlyUpdateViewModels)
        {
            // First remove all the user page view models.
            var viewModels = this.Children.Where(
                ch => ch is EmitterExtParamsUserSettingDataViewModel).ToArray();
            foreach (var vm in viewModels)
            {
                this.Children.Remove(vm);
            }

            if (onlyUpdateViewModels == false)
            {
                // Update user data models.
                (this.Proxy as EmitterExtParamsProxy).UpdateUserData();
            }

            this.AddChildViewModel(new EmitterExtParamsUserSettingDataViewModel(
                this, this.DataModel.EmitterExtParamsSettingData));

            // Clear modification flags.
            if (this.ModificationFlagViewModel != null)
            {
                this.ModificationFlagViewModel.ClearModificationFlags();
                this.ModificationFlagViewModel.ClearChildModificationFlags();
            }

            // Notify UI to update.
            this.OnPropertyChanged(() => this.AvailableSettingViewModels);
            this.OnPropertyChanged(() => this.SelectedSettingViewModel);
        }
    }
}
