﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Primitives;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// カラーテーブルアニメーション用のViewModel
    /// </summary>
    public class EmitterColorAnimationViewModel : EmitterAnimationViewModel
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="parent">親ViewModel</param>
        /// <param name="dataModel">データモデル</param>
        /// <param name="defaultValue">初期値</param>
        /// <param name="onPropertyChanges">
        /// データの編集に伴うOnPropertyChangedアクションの配列です。
        /// 0:Modified, 1:AnimationTable, 2:EnableAnimation, 3:LoopMode, 4:InterpolationMode
        /// </param>
        public EmitterColorAnimationViewModel(
            HierarchyViewModel parent,
            EmitterAnimationCommonData dataModel,
            Vector4f defaultValue,
            Action[] onPropertyChanges)
            : base(parent, dataModel, defaultValue)
        {
            this.PropertyChanged += (s, e) =>
            {
                onPropertyChanges[0]();
                if (IsRaisedProperty(e, () => this.AnimationTable))
                {
                    onPropertyChanges[1]();
                }
                else if (IsRaisedProperty(e, () => this.EnableAnimation))
                {
                    onPropertyChanges[2]();
                }
                else if (IsRaisedProperty(e, () => this.LoopMode))
                {
                    onPropertyChanges[3]();
                }
                else if (IsRaisedProperty(e, () => this.InterpolationMode))
                {
                    onPropertyChanges[4]();
                }
            };
        }

        /// <summary>
        /// Gets the loop mode.
        /// </summary>
        [UseDataModelOriginalValue]
        public int LoopMode
        {
            get { return this.GetDataModelValue(() => this.LoopMode); }
            set { this.SetDataModelValue(value, () => this.LoopMode); }
        }

        /// <summary>
        /// Gets the interpolation mode.
        /// </summary>
        [UseDataModelOriginalValue]
        public int InterpolationMode
        {
            get { return this.GetDataModelValue(() => this.InterpolationMode); }
            set { this.SetDataModelValue(value, () => this.InterpolationMode); }
        }

        /// <summary>
        /// アニメーションテーブルを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public new AnimationTableData AnimationTable
        {
            get
            {
                return this.GetDataModelValue(() => this.AnimationTable);
            }

            set
            {
                int count = this.AnimationTable.Count;
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.SetDataModelValue(value, () => this.AnimationTable),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether reset settings.
        /// </summary>
        protected override void OnReset()
        {
            if (ExportableViewModel.IsPasting)
            {
                return;
            }

            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            using (new CommandCombiner())
            {
                this.SetDataModelValue(0, () => this.LoopMode);
                this.SetDataModelValue(0, () => this.InterpolationMode);
                this.InitData();
            }
        }
    }

    /// <summary>
    /// カーブエディタを持つViewModelが必要なプロパティを集約したクラス
    /// </summary>
    public class EmitterAnimationViewModel : PropertyGroupViewModel<EmitterAnimationCommonData>, IModificationFlagOwner, IModificationPropertyOwner
    {
        /// <summary>
        /// 初期化中フラグです。
        /// </summary>
        private static bool isInitializing;

        /// <summary>
        /// オプションに保存するスナップ設定名です。
        /// </summary>
        private string snapSettingName = string.Empty;

        /// <summary>
        /// EnableOtherKeySnap プロパティのバッキングフィールドです。
        /// </summary>
        private bool enableOtherKeySnap;

        /// <summary>
        /// SnapTime プロパティのバッキングフィールドです。
        /// </summary>
        private int snapTime;

        /// <summary>
        /// SnapValue プロパティのバッキングフィールドです。
        /// </summary>
        private int snapValue;

        /// <summary>
        /// 無視プロパティ
        /// </summary>
        private readonly string[] ignores = new[]
        {
            "IsInitializing",
            "DefaultValue",
            "IsModified",
            "IsDefault",
            "EnableX",
            "EnableY",
            "EnableZ",
            "EnableW",
            "EnableOtherKeySnap",
            "SnapTime",
            "SnapValue",
            "ZeroPinMode",
        };

        /// <summary>
        /// キー値スケール(重力用)
        /// </summary>
        private readonly float innerScale;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        /// <param name="defaultValue">初期キーに設定する初期値.</param>
        /// <param name="scale">キーの値に乗算するスケール値(デフォルトは1.0)</param>
        public EmitterAnimationViewModel(
            HierarchyViewModel parent, EmitterAnimationCommonData dataModel, Vector4f defaultValue, float scale = 1.0f)
            : base(parent, dataModel)
        {
            this.InitSettings();
            this.OnResetExecutable = new AnonymousExecutable(this.OnReset);
            this.DefaultValue = defaultValue;
            this.innerScale = scale;

            dataModel.AnimationTable.Scale = this.innerScale;
            if (dataModel.AnimationTable.Count == 0)
            {
                dataModel.AnimationTable.AddKeyFrame(0, defaultValue, false);
            }

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignores);
        }

        /// <summary>
        /// 初期化終了時にカーブエディタウィンドウを更新するためのデリゲート
        /// </summary>
        public static Action OnEndBlocking { get; set; }

        /// <summary>
        /// 初期化中はカーブエディタに通知だけして欲しいのでそのフラグ
        /// </summary>
        public static bool IsInitializing
        {
            get
            {
                return isInitializing;
            }

            set
            {
                isInitializing = value;
                if (!value && OnEndBlocking != null)
                {
                    OnEndBlocking();
                }
            }
        }

        /// <summary>
        /// エミッタ時間アニメに変更があるか
        /// </summary>
        public bool IsModified
        {
            get { return this.ModificationFlagViewModel.IsAnyValueModified; }
        }

        /// <summary>
        /// エミッタ時間アニメがデフォルトの状態か
        /// </summary>
        public bool IsDefault
        {
            get
            {
                bool isDefault = this.DataModel.EnableAnimation == false;
                isDefault &= this.DataModel.LoopMode == 0;
                isDefault &= this.DataModel.InterpolationMode == 0;
                isDefault &= this.DataModel.AnimationTable.Count == 1;
                if (isDefault)
                {
                    isDefault &= this.DataModel.AnimationTable[0].Value.Equals(this.DefaultValue);
                }

                return isDefault;
            }
        }

        /// <summary>
        /// 初期キーとして使用するデフォルト値
        /// </summary>
        public Vector4f DefaultValue { get; private set; }

        /// <summary>
        /// Get an executable to run when reset.
        /// </summary>
        public IExecutable OnResetExecutable { get; private set; }

        /// <summary>
        /// Gets the loop mode.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableAnimation
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableAnimation);
            }

            set
            {
                using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
                {
                    this.SetDataModelValue(value, () => this.EnableAnimation);
                }
            }
        }

        /// <summary>
        /// X軸のチェックボックス
        /// </summary>
        public bool EnableX { get; set; }

        /// <summary>
        /// Y軸のチェックボックス
        /// </summary>
        public bool EnableY { get; set; }

        /// <summary>
        /// W軸のチェックボックス
        /// </summary>
        public bool EnableZ { get; set; }

        /// <summary>
        /// Z軸のチェックボックス
        /// </summary>
        public bool EnableW { get; set; }

        /// <summary>
        /// 他のキーにスナップするモードのボタン
        /// </summary>
        public bool EnableOtherKeySnap
        {
            get
            {
                return this.enableOtherKeySnap;
            }

            set
            {
                this.enableOtherKeySnap = value;
                this.UpdateSnapSettingOption();
            }
        }

        /// <summary>
        /// 時間スナップ
        /// </summary>
        public int SnapTime
        {
            get
            {
                return this.snapTime;
            }

            set
            {
                this.snapTime = value;
                this.UpdateSnapSettingOption();
            }
        }

        /// <summary>
        /// 値スナップ
        /// </summary>
        public int SnapValue
        {
            get
            {
                return this.snapValue;
            }

            set
            {
                this.snapValue = value;
                this.UpdateSnapSettingOption();
            }
        }

        /// <summary>
        /// 0固定モード
        /// </summary>
        public int ZeroPinMode { get; set; }

        /// <summary>
        /// Gets the interpolation mode.
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData AnimationTable
        {
            get
            {
                var table = this.GetDataModelValue(() => this.AnimationTable);
                table.Scale = this.innerScale;
                return table;
            }

            set
            {
                var table = value;
                table.Scale = this.innerScale;
                int count = this.AnimationTable.Count;
                var binder = new EffectMakerSetMemberBinder("AnimationTable", false, false);

                // キー数変化時のみデータ変更時のリロードを抑制して、変更後にリロードする処理を下記の関数に集約
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.TrySetMember(binder, table),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// スナップ設定のデフォルト値を設定します。
        /// </summary>
        /// <param name="settingName">設定名</param>
        /// <param name="isEnabled">スナップを有効にするかどうか</param>
        /// <param name="snapTime">時間スナップ</param>
        /// <param name="snapValue">値スナップ</param>
        public void SetDefaultSnapSetting(string settingName, bool enableSnap = true, int snapTime = 0, int snapValue = 2)
        {
            // スナップのデフォルト値を適用
            this.EnableOtherKeySnap = enableSnap;
            this.SnapTime = snapTime;
            this.SnapValue = snapValue;

            // オプションに保存されたデフォルト値を取得
            CurveEditorSnapSetting setting;
            OptionStore.RootOptions.Interface.CurveEditorSnapSettings.TryGetValue(settingName, out setting);

            // オプションに保存されたデフォルト値があればそれを適用する
            if (setting != null)
            {
                this.EnableOtherKeySnap = setting.EnableSnap;
                this.SnapTime = setting.SnapValueX;
                this.SnapValue = setting.SnapValueY;
            }

            // スナップ設定名に有効な値をセットすると、スナップ値の変更がオプションに反映されるようになるため
            // 全てのスナップ値の変更が終わってからスナップ設定名をセットする
            this.snapSettingName = settingName;
        }

        /// <summary>
        /// 現在のスナップ設定をオプションに反映させます。
        /// </summary>
        private void UpdateSnapSettingOption()
        {
            if (string.IsNullOrEmpty(this.snapSettingName) == false)
            {
                CurveEditorSnapSetting setting = new CurveEditorSnapSetting
                {
                    EnableSnap = this.EnableOtherKeySnap,
                    SnapValueX = this.SnapTime,
                    SnapValueY = this.SnapValue
                };

                OptionStore.RootOptions.Interface.CurveEditorSnapSettings[this.snapSettingName] = setting;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether reset settings.
        /// </summary>
        protected virtual void OnReset()
        {
            if (ExportableViewModel.IsPasting)
            {
                return;
            }

            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            {
                using (new CommandCombiner())
                {
                    var binder = new EffectMakerSetMemberBinder("LoopMode", false, false);
                    this.TrySetMember(binder, 0);
                    binder = new EffectMakerSetMemberBinder("InterpolationMode", false, false);
                    this.TrySetMember(binder, 0);
                    this.InitData();
                }
            }
        }

        /// <summary>
        /// 値を初期化
        /// </summary>
        protected void InitData()
        {
            var table = new AnimationTableData(0, this.DefaultValue)
            {
                Scale = this.innerScale
            };
            this.SetDataModelValue(table, () => this.AnimationTable);
        }

        /// <summary>
        /// 値以外の設定を初期化
        /// </summary>
        private void InitSettings()
        {
            this.EnableX = true;
            this.EnableY = true;
            this.EnableZ = true;
            this.EnableW = true;
            this.EnableOtherKeySnap = true;
            this.SnapTime = 0;
            this.SnapValue = 2;
            this.ZeroPinMode = -2;
        }
    }
}
