﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterData.
    /// </summary>
    public class EmitterTextureViewModel : PropertyPageViewModel<EmitterTextureData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterTextureViewModel(HierarchyViewModel parent, EmitterTextureData dataModel)
            : base(parent, dataModel)
        {
            this.EmitterTextureFileViewModel =
                new EmitterTextureFileViewModel(
                    this, dataModel.EmitterTextureFileData);
            this.Children.Add(this.EmitterTextureFileViewModel);

            this.EmitterTextureUvAnimationViewModel =
                new EmitterTextureUvAnimationViewModel(
                    this, dataModel.EmitterTextureUvAnimationData);
            this.Children.Add(this.EmitterTextureUvAnimationViewModel);

            this.EmitterTexturePatternAnimationViewModel =
                new EmitterTexturePatternAnimationViewModel(
                    this, dataModel.EmitterTexturePatternAnimationData);
            this.Children.Add(this.EmitterTexturePatternAnimationViewModel);

            this.EmitterTextureOptionViewModel =
                new EmitterTextureOptionViewModel(
                    this, dataModel.EmitterTextureOptionData);
            this.Children.Add(this.EmitterTextureOptionViewModel);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// EmitterTextureFileViewModel.
        /// </summary>
        public EmitterTextureFileViewModel EmitterTextureFileViewModel
        {
            get; private set;
        }

        /// <summary>
        /// EmitterTextureOptionViewModel.
        /// </summary>
        public EmitterTextureOptionViewModel EmitterTextureOptionViewModel
        {
            get;
            private set;
        }

        /// <summary>
        /// EmitterTextureUvAnimationViewModel.
        /// </summary>
        public EmitterTextureUvAnimationViewModel EmitterTextureUvAnimationViewModel
        {
            get; private set;
        }

        /// <summary>
        /// EmitterTexturePatternAnimationViewModel.
        /// </summary>
        public EmitterTexturePatternAnimationViewModel EmitterTexturePatternAnimationViewModel
        {
            get; private set;
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel
        {
            get; private set;
        }

        /// <summary>
        /// Sets the file path.
        /// </summary>
        public string FilePath
        {
            get { return this.EmitterTextureFileViewModel.FilePath; }
            set { this.EmitterTextureFileViewModel.FilePath = value; }
        }

        /// <summary>
        /// テクスチャビューモデルが同じスロットを指しているかをチェックします。
        /// </summary>
        /// <param name="other">もう1つのテクスチャビューモデル</param>
        /// <returns>一致していたらtrue,不一致だったらfalse.</returns>
        public bool IsSameTabNumber(EmitterTextureViewModel other)
        {
            var thisParent = this.Parent as EmitterTextureGroupViewModel;
            var otherParent = other.Parent as EmitterTextureGroupViewModel;
            Debug.Assert(thisParent != null && otherParent != null, "Invalid View Models!");

            int thisID = thisParent.GetTextureTabNumber(this);
            int otherID = otherParent.GetTextureTabNumber(other);
            return thisID != -1 && otherID != -1 && thisID == otherID;
        }
    }
}
