﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Net.Mime;
using System.Windows;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterData.
    /// </summary>
    public class EmitterRotationViewModel : PropertyPageViewModel<EmitterRotationData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "EnableRotationXGroup",
            "EnableRotationYGroup",
            "EnableRotationZGroup",
        };

        /// <summary>
        /// X軸グループの表示状態.
        /// </summary>
        private bool enableRotationXGroup;

        /// <summary>
        /// Y軸グループの表示状態.
        /// </summary>
        private bool enableRotationYGroup;

        /// <summary>
        /// Z軸グループの表示状態.
        /// </summary>
        private bool enableRotationZGroup;

        /// <summary>
        /// データモデル.
        /// </summary>
        private EmitterRotationData dataModel;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterRotationViewModel(HierarchyViewModel parent, EmitterRotationData dataModel)
            : base(parent, dataModel)
        {
            this.dataModel = dataModel;

            // 若干危うい修正だが、XYZの各要素VMを先に生成することで、グループ可視フラグを後から更新できるようにした
            this.EmitterRotationXViewModel = new EmitterRotationXViewModel(
                this, dataModel.EmitterRotationXData);
            this.EmitterRotationYViewModel = new EmitterRotationYViewModel(
                this, dataModel.EmitterRotationYData);
            this.EmitterRotationZViewModel = new EmitterRotationZViewModel(
                this, dataModel.EmitterRotationZData);
            this.EmitterRotationBasicViewModel = new EmitterRotationBasicViewModel(
                this, dataModel.EmitterRotationBasicData);

            // せめてChildrenへの追加はもとの順番通りに
            this.Children.Add(this.EmitterRotationBasicViewModel);
            this.Children.Add(this.EmitterRotationXViewModel);
            this.Children.Add(this.EmitterRotationYViewModel);
            this.Children.Add(this.EmitterRotationZViewModel);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// X軸グループの表示状態.
        /// </summary>
        public bool EnableRotationXGroup
        {
            get
            {
                return this.enableRotationXGroup;
            }

            set
            {
                this.SetValue(ref this.enableRotationXGroup, value);
                this.UpdateRotationVisibility();
            }
        }

        /// <summary>
        /// Y軸グループの表示状態.
        /// </summary>
        public bool EnableRotationYGroup
        {
            get
            {
                return this.enableRotationYGroup;
            }

            set
            {
                this.SetValue(ref this.enableRotationYGroup, value);
                this.UpdateRotationVisibility();
            }
        }

        /// <summary>
        /// Z軸グループの表示状態.
        /// </summary>
        public bool EnableRotationZGroup
        {
            get
            {
                return this.enableRotationZGroup;
            }

            set
            {
                this.SetValue(ref this.enableRotationZGroup, value);
                this.UpdateRotationVisibility();
            }
        }

        /// <summary>
        /// EmitterRotationBasicViewModel.
        /// </summary>
        public EmitterRotationBasicViewModel EmitterRotationBasicViewModel { get; private set; }

        /// <summary>
        /// EmitterRotationXViewModel.
        /// </summary>
        public EmitterRotationXViewModel EmitterRotationXViewModel { get; private set; }

        /// <summary>
        /// EmitterRotationYViewModel.
        /// </summary>
        public EmitterRotationYViewModel EmitterRotationYViewModel { get; private set; }

        /// <summary>
        /// EmitterRotationZViewModel.
        /// </summary>
        public EmitterRotationZViewModel EmitterRotationZViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 回転グループ表示状態の更新
        /// </summary>
        private void UpdateRotationVisibility()
        {
            if (this.EmitterRotationXViewModel == null ||
                this.EmitterRotationYViewModel == null ||
                this.EmitterRotationZViewModel == null)
            {
                return;
            }

            // 一旦全部非表示にしないとXYZの表示順がおかしくなる
            this.EmitterRotationXViewModel.GroupVisibility = false;
            this.EmitterRotationYViewModel.GroupVisibility = false;
            this.EmitterRotationZViewModel.GroupVisibility = false;

            this.EmitterRotationXViewModel.GroupVisibility = this.enableRotationXGroup;
            this.EmitterRotationYViewModel.GroupVisibility = this.enableRotationYGroup;
            this.EmitterRotationZViewModel.GroupVisibility = this.enableRotationZGroup;
        }
    }
}
