﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterParticleLifeViewModel.
    /// </summary>
    public class EmitterParticleLifeViewModel : PropertyGroupViewModel<EmitterParticleLifeData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterParticleLifeViewModel(
            HierarchyViewModel parent, EmitterParticleLifeData dataModel)
            : base(parent, dataModel)
        {
            this.EmitterLifeAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.LifeAnimation, InitialKeyValues.ParticleLife);
            this.Children.Add(this.EmitterLifeAnimationViewModel);

            this.OnShowCurveEditorDialog = new AnonymousExecutable(() =>
            {
                this.UpdateLifeAnimation();
                WorkspaceRootViewModel.Instance.ShowCurveEditor();
            });

            // 寿命エミッタ時間アニメの設定
            this.EmitterLifeAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsLifeCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterLifeAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableLifeAnimation);
                    this.UpdateLifeAnimation();
                }
            };
            this.EmitterLifeAnimationViewModel.SetDefaultSnapSetting("EmitterParticleLife", snapValue: 4);
            this.UpdateLifeAnimation();

            var mmfvm = new MultipleModificationFlagsViewModel(this);
            var emitterLifeParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Life),
                ViewModelBase.NameOf(() => this.IsLifeCurveModified));
            mmfvm.SetPropertyDictionary("LifeParams", emitterLifeParamsProperties);
            mmfvm.SetModificationDictionary("LifeParams", new IModificationPropertyOwner[] { this.EmitterLifeAnimationViewModel });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
        }

        /// <summary>
        /// 寿命のエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsLifeCurveModified
        {
            get { return this.EmitterLifeAnimationViewModel.IsModified; }
        }

        /// <summary>
        /// 寿命アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableLifeAnimation
        {
            get
            {
                return this.EmitterLifeAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterLifeAnimationViewModel.EnableAnimation = value;
                this.UpdateLifeAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// 寿命無限
        /// </summary>
        [UseDataModelOriginalValue]
        public bool EnableInfinityLife
        {
            get
            {
                return this.GetDataModelValue(() => this.EnableInfinityLife);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EnableInfinityLife);
                this.UpdateLifeAnimation();
            }
        }

        /// <summary>
        /// EmitterLifeAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterLifeAnimationViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// カーブエディタを表示するExecutableを取得します。
        /// </summary>
        public IExecutable OnShowCurveEditorDialog { get; private set; }

        /// <summary>
        /// プロパティ変更通知を強制的に発行します。
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();
            this.UpdateLifeAnimation();
        }

        /// <summary>
        /// 寿命アニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateLifeAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 100.0f,
                LabelDigit = 0,
                DefaultZeroPin = -1,
                MaxLimit = 1000000.0f,
                MinLimit = 1.0f,
                DefaultValue = InitialKeyValues.ParticleLife[0],
                IntegerOnly = true,
                Channels = new List<string>
                    {
                        "X",
                    },
                DataContext = this.EnableInfinityLife ? null : this.EmitterLifeAnimationViewModel,
                AnimationName = "EmitterParticleLife",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }
    }
}
