﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Net.Mime;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryData;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.FileMonitor;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterEmitterShapeViewModel.
    /// </summary>
    public class EmitterEmitterShapeViewModel : PropertyGroupViewModel<EmitterEmitterShapeData>, IModificationFlagOwner
    {
        /// <summary>
        /// コピペの対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreCopyPropertyNames = new string[]
        {
            "SamplePrimitiveFilePath",
        };

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreModifyPropertyNames = new string[]
        {
            "CircleSubGroupVisibility",
            "CircleSameDivideSubGroupVisibility",
            "FillCircleSubGroupVisibility",
            "SphereSubGroupVisibility",
            "SphereSameDivideSubGroupVisibility",
            "SphereSameDivide64SubGroupVisibility",
            "FillSphereSubGroupVisibility",
            "CylinderSubGroupVisibility",
            "FillCylinderSubGroupVisibility",
            "BoxSubGroupVisibility",
            "FillBoxSubGroupVisibility",
            "LineSubGroupVisibility",
            "LineSameDivideSubGroupVisibility",
            "RectangleSubGroupVisibility",
            "PrimitiveSubGroupVisibility",
            "ScaleSubGroupVisibility",
            "ArcTypeVisibility",
            "LongitudeVisibility",
            "LatitudeVisibility",
            "SameDivideRandomVisibility",
            "WarningMessages",
            "PrimitiveFilePathBgColor",
            "MessageOfDivisionAndEmission",
        };

        /// <summary>
        /// 親エミッタセット
        /// </summary>
        private EmitterSetViewModel emitterSetViewModel = null;

        /// <summary>
        /// ファイルモニタイベントをロックするフラグ
        /// </summary>
        private bool monitorEventLock = false;

        /// <summary>
        /// 警告ウィンドウを表示するかどうか
        /// </summary>
        private bool onWarningWindow = true;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterEmitterShapeViewModel(
            HierarchyViewModel parent, EmitterEmitterShapeData dataModel)
            : base(parent, dataModel)
        {
            this.EmitterTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypePoint, 0),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeCircle, 1),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeCircleSameDivide, 2),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeFillCircle, 3),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeSphere, 4),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeSphereSameDivide, 5),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeSphereSameDivide64, 6),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeFillSphere, 7),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeCylinder, 8),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeFillCylinder, 9),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeBox, 10),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeFillBox, 11),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeLine, 12),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeLineSameDivide, 13),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypeRectangle, 14),
                new KeyValuePair<string, object>(Resources.EmitterEmitterTypePrimitive, 15)
            };

            this.SphereAndDivisionDivisionTypeItems = new []
            {
                new KeyValuePair<string, object>("2", 0),
                new KeyValuePair<string, object>("3", 1),
                new KeyValuePair<string, object>("4", 2),
                new KeyValuePair<string, object>("6", 3),
                new KeyValuePair<string, object>("8", 4),
                new KeyValuePair<string, object>("12", 5),
                new KeyValuePair<string, object>("20", 6),
                new KeyValuePair<string, object>("32", 7),
            };

            this.EmissionTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterEmitterEmissionNormal, 0),
                new KeyValuePair<string, object>(Resources.EmitterEmitterEmissionRandom, 1),
                new KeyValuePair<string, object>(Resources.EmitterEmitterEmissionIndexOrder, 2),
            };

            this.ArcTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterEmitterLatitude, 0),
                new KeyValuePair<string, object>(Resources.EmitterEmitterLongitude, 1),
            };

            this.AxisTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterEmitterAxisTypePositiveX, 0),
                new KeyValuePair<string, object>(Resources.EmitterEmitterAxisTypeNegativeX, 1),
                new KeyValuePair<string, object>(Resources.EmitterEmitterAxisTypePositiveY, 2),
                new KeyValuePair<string, object>(Resources.EmitterEmitterAxisTypeNegativeY, 3),
                new KeyValuePair<string, object>(Resources.EmitterEmitterAxisTypePositiveZ, 4),
                new KeyValuePair<string, object>(Resources.EmitterEmitterAxisTypeNegativeZ, 5),
            };

            this.ShapeScaleAnimationViewModel = new EmitterAnimationViewModel(
                this,
                dataModel.ShapeScaleAnimation,
                InitialKeyValues.ShapeScale);
            this.Children.Add(this.ShapeScaleAnimationViewModel);

            // スケールのエミッタ時間アニメを設定する
            this.ShapeScaleAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsShapeScaleCurveModified);
                if (IsRaisedProperty(e, () => this.ShapeScaleAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableShapeScaleAnimation);
                    this.UpdateShapeScaleAnimation();
                }
            };
            this.ShapeScaleAnimationViewModel.SetDefaultSnapSetting("EmitterShapeScale", snapValue: 3);
            this.UpdateShapeScaleAnimation();

            // エミッタだけ削除されたときに親エミッタセットがnullになるので先に取得しておく
            // エミッタ削除後にプリミティブファイル変更イベントを処理するときに必要
            this.emitterSetViewModel = GetParent<EmitterSetViewModel>(this);
            Debug.Assert(this.emitterSetViewModel != null, "親エミッタセットがnull");

            // 警告メッセージを初期化
            this.WarningMessages = new List<Tuple<string, Color, Font>>();
            this.UpdateWarningMessages(false);
            this.UpdatePrimitiveInfo();

            this.OnShowCurveEditorDialog = new AnonymousExecutable(() =>
            {
                this.UpdateShapeScaleAnimation();
                WorkspaceRootViewModel.Instance.ShowCurveEditor();
            });

            this.OnReloadPrimitiveFileExecutable = new AnonymousExecutable(this.OnReloadPrimitiveFile);

            // コピペ時に含めないプロパティを登録する.
            this.AddIgnoreCopyProperties(this.ignoreCopyPropertyNames);

            var mmfvm = new MultipleModificationFlagsViewModel(this);
            var shapeScaleParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.EmitterScaleShapeScale),
                ViewModelBase.NameOf(() => this.IsShapeScaleCurveModified));
            mmfvm.SetPropertyDictionary("ShapeScaleParams", shapeScaleParamsProperties);
            mmfvm.SetModificationDictionary("ShapeScaleParams", new IModificationPropertyOwner[] { this.ShapeScaleAnimationViewModel });
            mmfvm.AddIgnoreProperties(this.ignoreModifyPropertyNames);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
        }

        /// <summary>
        /// スケールのエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsShapeScaleCurveModified
        {
            get { return this.ShapeScaleAnimationViewModel.IsModified; }
        }

        /// <summary>
        /// スケールエミッタ時間アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableShapeScaleAnimation
        {
            get
            {
                return this.ShapeScaleAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.ShapeScaleAnimationViewModel.EnableAnimation = value;
                this.UpdateShapeScaleAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>エミッタタイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> EmitterTypeItems { get; private set; }

        /// <summary>球(等分割)タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> SphereAndDivisionDivisionTypeItems { get; private set; }

        /// <summary>放出タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> EmissionTypeItems { get; private set; }

        /// <summary>弧の開き方の項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> ArcTypeItems { get; private set; }

        /// <summary>軸の項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> AxisTypeItems { get; private set; }

        /// <summary>円グループの表示状態を設定または取得します。</summary>
        public bool CircleSubGroupVisibility
        {
            get { return this.EmitterType == 1; }
        }

        /// <summary>円(等分割)グループの表示状態を取得します。</summary>
        public bool CircleSameDivideSubGroupVisibility
        {
            get { return this.EmitterType == 2; }
        }

        /// <summary>円(ボリューム)グループの表示状態を取得します。</summary>
        public bool FillCircleSubGroupVisibility
        {
            get { return this.EmitterType == 3; }
        }

        /// <summary>球グループの表示状態を取得します。</summary>
        public bool SphereSubGroupVisibility
        {
            get { return this.EmitterType == 4; }
        }

        /// <summary>球(等分割)グループの表示状態を取得します。</summary>
        public bool SphereSameDivideSubGroupVisibility
        {
            get { return this.EmitterType == 5; }
        }

        /// <summary>球(等分割64)グループの表示状態を取得します。</summary>
        public bool SphereSameDivide64SubGroupVisibility
        {
            get { return this.EmitterType == 6; }
        }

        /// <summary>球(ボリューム)グループの表示状態を取得します。</summary>
        public bool FillSphereSubGroupVisibility
        {
            get { return this.EmitterType == 7; }
        }

        /// <summary>円柱グループの表示状態を取得します。</summary>
        public bool CylinderSubGroupVisibility
        {
            get { return this.EmitterType == 8; }
        }

        /// <summary>円柱(ボリューム)グループの表示状態を取得します。</summary>
        public bool FillCylinderSubGroupVisibility
        {
            get { return this.EmitterType == 9; }
        }

        /// <summary>立方体グループの表示状態を取得します。</summary>
        public bool BoxSubGroupVisibility
        {
            get { return this.EmitterType == 10; }
        }

        /// <summary>立方体(ボリューム)グループの表示状態を取得します。</summary>
        public bool FillBoxSubGroupVisibility
        {
            get { return this.EmitterType == 11; }
        }

        /// <summary>ライングループの表示状態を取得します。</summary>
        public bool LineSubGroupVisibility
        {
            get { return this.EmitterType == 12; }
        }

        /// <summary>ライン(等分割)グループの表示状態を取得します。</summary>
        public bool LineSameDivideSubGroupVisibility
        {
            get { return this.EmitterType == 13; }
        }

        /// <summary>矩形グループの表示状態を取得します。</summary>
        public bool RectangleSubGroupVisibility
        {
            get { return this.EmitterType == 14; }
        }

        /// <summary>プリミティブグループの表示状態を取得します。</summary>
        public bool PrimitiveSubGroupVisibility
        {
            get { return this.EmitterType == 15; }
        }

        /// <summary>スケールグループの表示状態を取得します。</summary>
        public bool ScaleSubGroupVisibility
        {
            get { return this.EmitterType != 0; }
        }

        /// <summary>弧の開き方ラベルの表示状態を取得します。</summary>
        public bool ArcTypeVisibility
        {
            get { return this.EmitterType == 4 || this.EmitterType == 7; }
        }

        /// <summary>経度グループの表示状態を取得します。</summary>
        public bool LongitudeVisibility
        {
            get { return (this.EmitterType == 4 || this.EmitterType == 7) && this.EmitterShapeArcType == 0; }
        }

        /// <summary>緯度グループの表示状態を取得します。</summary>
        public bool LatitudeVisibility
        {
            get
            {
                bool visible = false;
                visible |= (this.EmitterType == 4 || this.EmitterType == 7) &&
                    this.EmitterShapeArcType == 1;
                visible |= this.EmitterType == 5;
                visible |= this.EmitterType == 6;
                return visible;
            }
        }

        /// <summary>
        /// 等分割のランダムUIの表示状態を取得します。
        /// </summary>
        public bool SameDivideRandomVisibility
        {
            get
            {
                var emissionType = this.GetDataModelValue(() => this.EmitterShapeEmissionType);
                return (this.EmitterType == 2 || this.EmitterType == 13) && emissionType == 0;
            }
        }

        /// <summary>
        /// エミッタのタイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int EmitterType
        {
            get
            {
                return this.GetDataModelValue(() => this.EmitterType);
            }

            set
            {
                using (new CommandCombiner())
                {
                    if (IsUsingSize(this.EmitterType))
                    {
                        if (IsUsingLength(value))
                        {
                            var size = this.GetDataModelValue(() => this.DataModel.EmitterTypeEmitterRadius);
                            this.SetDataModelValue(size.Z, () => this.DataModel.EmitterShapeLineLength);
                        }
                    }
                    else if (IsUsingLength(this.EmitterType))
                    {
                        if (IsUsingSize(value))
                        {
                            var length = this.GetDataModelValue(() => this.DataModel.EmitterShapeLineLength);
                            var size = this.GetDataModelValue(() => this.DataModel.EmitterTypeEmitterRadius);
                            this.SetDataModelValue(new Vector3f(size.X, size.Y, length), () => this.DataModel.EmitterTypeEmitterRadius);
                        }
                    }

                    this.SetDataModelValue(value, () => this.EmitterType);
                }

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.CircleSubGroupVisibility);
                this.OnPropertyChanged(() => this.CircleSameDivideSubGroupVisibility);
                this.OnPropertyChanged(() => this.FillCircleSubGroupVisibility);
                this.OnPropertyChanged(() => this.SphereSubGroupVisibility);
                this.OnPropertyChanged(() => this.SphereSameDivideSubGroupVisibility);
                this.OnPropertyChanged(() => this.SphereSameDivide64SubGroupVisibility);
                this.OnPropertyChanged(() => this.FillSphereSubGroupVisibility);
                this.OnPropertyChanged(() => this.CylinderSubGroupVisibility);
                this.OnPropertyChanged(() => this.FillCylinderSubGroupVisibility);
                this.OnPropertyChanged(() => this.BoxSubGroupVisibility);
                this.OnPropertyChanged(() => this.FillBoxSubGroupVisibility);
                this.OnPropertyChanged(() => this.LineSubGroupVisibility);
                this.OnPropertyChanged(() => this.LineSameDivideSubGroupVisibility);
                this.OnPropertyChanged(() => this.RectangleSubGroupVisibility);
                this.OnPropertyChanged(() => this.PrimitiveSubGroupVisibility);

                this.OnPropertyChanged(() => this.ScaleSubGroupVisibility);
                this.OnPropertyChanged(() => this.ArcTypeVisibility);
                this.OnPropertyChanged(() => this.LongitudeVisibility);
                this.OnPropertyChanged(() => this.LatitudeVisibility);
                this.OnPropertyChanged(() => this.SameDivideRandomVisibility);

                this.UpdateEmissionViewModel();
                this.UpdateShapeScaleAnimation();
            }
        }

        /// <summary>
        /// 放出タイプの値を設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int EmitterShapeEmissionType
        {
            get
            {
                return this.GetDataModelValue(() => this.EmitterShapeEmissionType);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EmitterShapeEmissionType);
                this.OnPropertyChanged(() => this.SameDivideRandomVisibility);
                this.OnPropertyChanged(() => this.MessageOfDivisionAndEmission);
                this.OnPropertyChanged(() => this.MessageOfPrimitiveAndEmission);
                this.UpdateEmissionViewModel();
            }
        }

        /// <summary>
        /// エミッタのタイプを設定または取得します。
        /// </summary>
        public int EmitterShapeArcType
        {
            get
            {
                return this.GetDataModelValue(() => this.EmitterShapeArcType);
            }

            set
            {
                this.SetDataModelValue(value, () => this.EmitterShapeArcType);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.LongitudeVisibility);
                this.OnPropertyChanged(() => this.LatitudeVisibility);
            }
        }

        /// <summary>
        /// プリミティブファイルパスを取得または設定します。
        /// </summary>
        public string PrimitiveFilePath
        {
            get
            {
                return this.GetDataModelValue<string>(() => this.PrimitiveFilePath);
            }

            set
            {
                if (!string.IsNullOrEmpty(this.PrimitiveFilePath) && Path.IsPathRooted(this.PrimitiveFilePath))
                {
                    this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.PrimitiveFilePath);
                    this.emitterSetViewModel.PrimitiveFileChanged -= this.OnPrimitiveFileChanged;
                }

                if (!string.IsNullOrEmpty(value) && Path.IsPathRooted(value))
                {
                    this.emitterSetViewModel.FileMonitor.RegisterMonitorPreCopyPath(value, null, this.OnMonitorEvent);
                    this.emitterSetViewModel.PrimitiveFileChanged += this.OnPrimitiveFileChanged;
                }

                // 同じパスが渡された時は処理をスキップ(無用なスターマークの原因)
                if (value == this.PrimitiveFilePath)
                {
                    return;
                }

                if (this.SetDataModelValue(value, () => this.PrimitiveFilePath))
                {
                    this.UpdateWarningMessages();
                    this.UpdatePrimitiveInfo();
                }

                if (this.onWarningWindow == true)
                {
                    this.CheckLocatable();
                }
            }
        }

        /// <summary>
        /// サンプルプリミティブがあるディレクトリを返す
        /// </summary>
        public string SamplePrimitiveFilePath
        {
            get
            {
                return Path.Combine(IOConstants.ExecutableFolderPath, "Primitives");
            }

            set
            {
                this.onWarningWindow = true;
                this.PrimitiveFilePath = value;
            }
        }

        /// <summary>
        /// プリミティブファイルパス入力テキストボックスの背景色です.
        /// </summary>
        public Color PrimitiveFilePathBgColor { get; set; }

        /// <summary>
        /// プリミティブファイルに関する警告メッセージです.
        /// </summary>
        public List<Tuple<string, Color, Font>> WarningMessages { get; set; }

        /// <summary>
        /// 分割数と放出レートの関係についてのメッセージです。
        /// </summary>
        public string MessageOfDivisionAndEmission
        {
            get
            {
                var emissionType = this.GetDataModelValue(() => this.EmitterShapeEmissionType);
                return emissionType == 0 ?
                    Resources.EmitterEmitterShapeCommentDivisionCount :
                    Resources.EmitterEmitterShapeCommentEmissiveDivision;
            }
        }

        /// <summary>
        /// プリミティブ使用時の放出タイプ選択に伴うメッセージです。
        /// </summary>
        public string MessageOfPrimitiveAndEmission
        {
            get
            {
                var emissionType = this.GetDataModelValue(() => this.EmitterShapeEmissionType);
                return emissionType == 0 ?
                    Resources.EmitterEmitterShapeCommentPrimitive :
                    Resources.EmitterEmitterShapeCommentPrimitiveEmission;
            }
        }

        /// <summary>
        /// プリミティブの情報を表示します。
        /// </summary>
        public string PrimitiveProperty { get; private set; }

        /// <summary>
        /// プリミティブファイルをリロードするためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnReloadPrimitiveFileExecutable { get; private set; }

        /// <summary>
        /// 形状のスケールアニメ
        /// </summary>
        public EmitterAnimationViewModel ShapeScaleAnimationViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// カーブエディタを表示するExecutableを取得します。
        /// </summary>
        public IExecutable OnShowCurveEditorDialog { get; private set; }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            base.UpdateChildViewModels();
            this.UpdateWarningMessages(true);
        }

        /// <summary>
        /// Disposer
        /// </summary>
        public override void Dispose()
        {
            if (!string.IsNullOrEmpty(this.PrimitiveFilePath))
            {
                this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.PrimitiveFilePath);
                this.emitterSetViewModel.PrimitiveFileChanged -= this.OnPrimitiveFileChanged;
            }

            base.Dispose();
        }

        /// <summary>
        /// Resend PropertyChanged notification for all properties.
        /// This is required when the data model changes independently from the view model.
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();

            this.UpdateShapeScaleAnimation();

            this.UpdateEmissionViewModel();

            this.UpdateWarningMessages();
            this.UpdatePrimitiveInfo();

            // タブのコピペ時だけ、ここで警告ウィンドウを出す.
            if (ExportableViewModel.IsPasting == true)
            {
                this.CheckLocatable();
            }
        }

        /// <summary>
        /// 形状スケールアニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateShapeScaleAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1.0f,
                LabelDigit = 2,
                MaxLimit = 1000000.0f,
                MinLimit = 0.0f,
                DefaultValue = InitialKeyValues.ShapeScale[0],
                DefaultZeroPin = -1,
                Channels = new List<string>
                    {
                        "X",
                        "Y",
                        "Z"
                    },
                DataContext = this.ScaleSubGroupVisibility ? this.ShapeScaleAnimationViewModel : null,
                AnimationName = "EmitterShapeScale",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }

        /// <summary>
        /// 警告ウィンドウを出すか指定して、PrimitiveFilePathを更新する.
        /// </summary>
        /// <param name="filePath">プリミティブファイルのパス</param>
        /// <param name="warning">trueなら警告ウィンドウを表示する.falseなら警告ウィンドウを表示しない.</param>
        public void SetPrimitiveFilePathWithWarningWindow(string filePath, bool warning)
        {
            // warningがtrueだと警告ウィンドウを表示する.
            // warningがfalseだと警告ウィンドウを出さない。
            this.onWarningWindow = warning;

            // FilePathを更新する.
            this.PrimitiveFilePath = filePath;

            // 警告ウィンドウがでる状態に更新しておく.
            this.onWarningWindow = true;
        }

        /// <summary>
        /// ラインかどうかを判定します。
        /// </summary>
        /// <param name="type">形状タイプを表す整数値</param>
        /// <returns>ライン形状だったらtrue.</returns>
        private static bool IsUsingLength(int type)
        {
            return type == 12 || type == 13;
        }

        /// <summary>
        /// 3次元のサイズを持つ形状かどうかを判定します。
        /// </summary>
        /// <param name="type">形状タイプを表す整数値</param>
        /// <returns>3次元のサイズを持つ形状ならtrue.</returns>
        private static bool IsUsingSize(int type)
        {
            return (type >= 1 && type <= 11) || type == 14;
        }

        /// <summary>
        /// ファイル監視システムのイベントがあった場合の処理
        /// </summary>
        /// <param name="path">監視パス</param>
        /// <param name="userData">ユーザーデータ(未使用)</param>
        /// <param name="watcherChangeTypes">変更タイプ</param>
        private void OnMonitorEvent(string path, object userData, WatcherChangeTypes watcherChangeTypes)
        {
            // プリミティブの自動リロードがOffになっていたら、何もしない
            if (OptionStore.RootOptions.Basic.Primitives.AutoReload == false)
            {
                return;
            }

            if (this.monitorEventLock)
            {
                return;
            }

            this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.PrimitiveFilePath);

            this.monitorEventLock = true;

            switch (watcherChangeTypes)
            {
                case WatcherChangeTypes.Changed:
                case WatcherChangeTypes.Created:
                    this.OnReloadPrimitiveFile("PrimitiveFilePath");
                    break;
                case WatcherChangeTypes.Renamed:
                    if (File.Exists(path))
                    {
                        this.OnReloadPrimitiveFile("PrimitiveFilePath");
                    }
                    else
                    {
                        this.UpdateWarningMessages(false);
                        this.UpdatePrimitiveInfo();
                    }

                    break;
                case WatcherChangeTypes.Deleted:
                    this.UpdateWarningMessages(false);
                    this.UpdatePrimitiveInfo();
                    break;
            }

            this.monitorEventLock = false;

            this.emitterSetViewModel.FileMonitor.RegisterMonitorPreCopyPath(this.PrimitiveFilePath, null, this.OnMonitorEvent);
            this.emitterSetViewModel.TriggerPrimitiveFileChanged(this, path);
        }

        /// <summary>
        /// プリミティブファイルに変更があったとき、リロード処理を行った後の処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="path">プリミティブファイルパス</param>
        private void OnPrimitiveFileChanged(object sender, string path)
        {
            if (sender == this || path != this.PrimitiveFilePath)
            {
                return;
            }

            this.UpdateWarningMessages(false);
            this.UpdatePrimitiveInfo();
        }

        /// <summary>
        /// プリミティブファイルのリロード処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnReloadPrimitiveFile(object parameter)
        {
            var binder = new EffectMaker.Foundation.Dynamic.EffectMakerGetMemberBinder((string)parameter);
            object path = null;
            this.TryGetMember(binder, out path);
            var pathString = path as string;

            // リロード要求.
            BusinessLogic.Manager.PrimitiveManager.Instance.Reload(pathString);

            // 警告メッセージを更新
            this.UpdateWarningMessages();
            this.UpdatePrimitiveInfo();
            this.CheckLocatable();
        }

        /// <summary>
        /// 警告メッセージを更新します.
        /// </summary>
        /// <param name="notice">親エミッタに通知するかどうかのフラグ</param>
        private void UpdateWarningMessages(bool notice = true)
        {
            this.WarningMessages.Clear();

            bool reachable, locatable;
            AssetsManager.GetAssetStatus(this.PrimitiveFilePath, this.emitterSetViewModel.FilePath, BusinessLogic.AssetResourceTypes.Primitive, out reachable, out locatable);

            // ファイルパスがないとき
            if (reachable == false)
            {
                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                    Resources.EmitterPrimitiveWarningNotFound,
                    Color.Red,
                    null));

                this.PrimitiveFilePathBgColor = Color.LightPink;
            }
            else
            {
                this.PrimitiveFilePathBgColor = Color.White;
            }

            // ロケートできないとき
            if (locatable == false)
            {
                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                    Resources.EmitterPrimitiveWarningUnreachable,
                    Color.SlateBlue,
                    null));

                // unreachableのフラグは立てない
            }

            // ファイルがあるとき
            if (reachable)
            {
                // モデルデータを取得
                Foundation.Model.ModelData modelData = null;
                var primitiveManager = EffectMaker.BusinessLogic.Manager.PrimitiveManager.Instance;
                LoadModelResults result = primitiveManager.LoadModelWithData(this.PrimitiveFilePath, true, out modelData);

                if (result == LoadModelResults.InvalidFileVersion)
                {
                    this.WarningMessages.Add(new Tuple<string, Color, Font>(
                        Resources.EmitterEmitterShapeWarningInvalidFileVersion,
                        Color.Red,
                        null));

                    this.PrimitiveFilePathBgColor = Color.LightPink;
                }
            }

            // ノードアイコンを更新するため、エミッタのアセットステータスをアップデート
            if (notice)
            {
                var emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>((ViewModelBase)this.Parent);
                emitterViewModel.UpdateAssetStatus(!reachable, false);
            }

            this.OnPropertyChanged(() => this.WarningMessages);
            this.OnPropertyChanged(() => this.PrimitiveFilePathBgColor);
        }

        /// <summary>
        /// プリミティブの情報を更新します。
        /// </summary>
        private void UpdatePrimitiveInfo()
        {
            // モデルデータを取得
            Foundation.Model.ModelData modelData = null;
            if (string.IsNullOrEmpty(this.PrimitiveFilePath) == false && File.Exists(this.PrimitiveFilePath))
            {
                EffectMaker.BusinessLogic.Manager.PrimitiveManager.Instance.LoadModelWithData(this.PrimitiveFilePath, true, out modelData);
            }

            if (modelData == null)
            {
                this.PrimitiveProperty = Properties.Resources.EmitterParticleShapeCommentPrimitiveNotLoaded;
            }
            else
            {
                // 頂点数
                this.PrimitiveProperty = Properties.Resources.EmitterParticleShapeCommentNumberOfVertex +
                    " " + modelData.Position.Count + Environment.NewLine;

                // 内包するアセットの一覧
                bool firstTime = true;
                this.PrimitiveProperty += Properties.Resources.EmitterParticleShapeCommentAttribute + " ";
                if (modelData.Position.Count > 0)
                {
                    this.PrimitiveProperty += "pos";
                    firstTime = false;
                }
                if (modelData.Normal.Count > 0)
                {
                    if (firstTime == false)
                    {
                        this.PrimitiveProperty += ", ";
                    }
                    this.PrimitiveProperty += "normal";
                    firstTime = false;
                }
                if (modelData.Tangent.Count > 0)
                {
                    if (firstTime == false)
                    {
                        this.PrimitiveProperty += ", ";
                    }
                    this.PrimitiveProperty += "tangent";
                    firstTime = false;
                }
                if (modelData.TexCoord0.Count > 0)
                {
                    if (firstTime == false)
                    {
                        this.PrimitiveProperty += ", ";
                    }
                    this.PrimitiveProperty += "uv0";
                    firstTime = false;
                }
                if (modelData.TexCoord1.Count > 0)
                {
                    if (firstTime == false)
                    {
                        this.PrimitiveProperty += ", ";
                    }
                    this.PrimitiveProperty += "uv1";
                    firstTime = false;
                }
                if (modelData.Color.Count > 0)
                {
                    if (firstTime == false)
                    {
                        this.PrimitiveProperty += ", ";
                    }
                    this.PrimitiveProperty += "color";
                }
                this.PrimitiveProperty += Environment.NewLine;

                // uvチャネルの名前
                string channelName0 = modelData.TexCoordName0;
                channelName0 = string.IsNullOrEmpty(channelName0) ? "-" : channelName0;

                string channelName1 = modelData.TexCoordName1;
                channelName1 = string.IsNullOrEmpty(channelName1) ? "-" : channelName1;

                this.PrimitiveProperty += string.Format("{0} ( {1} , {2} )", Properties.Resources.EmitterParticleShapeCommentUvChannelNames, channelName0, channelName1);
            }

            this.OnPropertyChanged(() => this.PrimitiveProperty);
        }

        /// <summary>
        /// プリミティブがエミッタセットから探索可能かどうかメッセージを出します.
        /// </summary>
        private void CheckLocatable()
        {
            // ファイルパスが指定されていないとき終了
            if (string.IsNullOrEmpty(this.PrimitiveFilePath))
            {
                return;
            }

            // ファイルパスが相対パスのとき終了
            if (Path.IsPathRooted(this.PrimitiveFilePath) == false)
            {
                return;
            }

            bool reachable, locatable;
            AssetsManager.GetAssetStatus(this.PrimitiveFilePath, this.emitterSetViewModel.FilePath, BusinessLogic.AssetResourceTypes.Primitive, out reachable, out locatable);

            // エミッタセットからプリミティブを探索できないとき
            if (locatable == false)
            {
                DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowWarningOnLoadPrimitive(string.Format(Resources.WarningPrimitiveLocatable, Path.GetFileName(this.PrimitiveFilePath)));
            }
        }

        /// <summary>
        /// 放出タブのUIの状態を更新します。
        /// </summary>
        private void UpdateEmissionViewModel()
        {
            var emitter = this.FindNearestParentOfType<EmitterViewModel>();
            emitter.EmitterEmissionViewModel.EmitterEmissionTimingViewModel.RaiseEmitRateTimingEnable();
        }
    }
}
