﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterEmissionGravityViewModel.
    /// </summary>
    public class EmitterEmissionGravityViewModel : PropertyGroupViewModel<EmitterEmissionGravityData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterEmissionGravityViewModel(
            HierarchyViewModel parent, EmitterEmissionGravityData dataModel)
            : base(parent, dataModel)
        {
            // FE1経由でコンバートしたデータ用に運動量ランダムをここでクランプする
            this.DataModel.MomentumRandom = MathUtility.Clamp(this.DataModel.MomentumRandom, 0.0f, 1.0f);

            // 重力はバイナライズ時に1/100するので、ビューモデルにスケールを通知しておく
            this.EmitterGravityAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.GravityAnimation, InitialKeyValues.EmissionGravity, 0.01f);
            this.Children.Add(this.EmitterGravityAnimationViewModel);

            this.OnShowCurveEditorDialog = new AnonymousExecutable(() =>
            {
                this.UpdateGravityAnimation();
                WorkspaceRootViewModel.Instance.ShowCurveEditor();
            });
            this.OnApplyRegularGravity = new AnonymousExecutable(this.ApplyRegularGravity);

            // 重力のエミッタ時間アニメの設定
            this.EmitterGravityAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsGravityCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterGravityAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableGravityAnimation);
                    this.UpdateGravityAnimation();
                }
            };
            this.EmitterGravityAnimationViewModel.SetDefaultSnapSetting("EmitterEmissionGravity", snapValue: 1);
            this.UpdateGravityAnimation();

            var mmfvm = new MultipleModificationFlagsViewModel(this);
            var emitterGravityParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.GravityVolume),
                ViewModelBase.NameOf(() => this.IsGravityCurveModified));
            mmfvm.SetPropertyDictionary("EmitterGravityVolumeParams", emitterGravityParamsProperties);
            mmfvm.SetModificationDictionary("EmitterGravityVolumeParams", new IModificationPropertyOwner[] { this.EmitterGravityAnimationViewModel });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
        }

        /// <summary>
        /// 重力のエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsGravityCurveModified
        {
            get { return this.EmitterGravityAnimationViewModel.IsModified;  }
        }

        /// <summary>
        /// 重力アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableGravityAnimation
        {
            get
            {
                return this.EmitterGravityAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterGravityAnimationViewModel.EnableAnimation = value;
                this.UpdateGravityAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// 標準重力が有効か否かを取得します。
        /// </summary>
        public bool EnableRegularGravity
        {
            get { return OptionStore.ProjectConfig.EnableRegularGravity; }
        }

        /// <summary>
        /// テキストボックスに表示する標準重力を取得します。
        /// </summary>
        public string RegularGravity
        {
            get
            {
                if (!this.EnableRegularGravity)
                {
                    return string.Empty;
                }

                return CalcRegularGravity().ToString();
            }
        }

        /// <summary>
        /// EmitterGravityAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterGravityAnimationViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// カーブエディタを表示するExecutableを取得します。
        /// </summary>
        public IExecutable OnShowCurveEditorDialog { get; private set; }

        /// <summary>
        /// 標準重力の適用を行うExecutableを取得します。
        /// </summary>
        public IExecutable OnApplyRegularGravity { get; private set; }

        /// <summary>
        /// プロパティ変更通知を強制的に発行します。
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();
            this.UpdateGravityAnimation();
        }

        /// <summary>
        /// 重力アニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateGravityAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1.0f,
                LabelDigit = 2,
                DefaultZeroPin = -1,
                MaxLimit = 1000000.0f,
                MinLimit = 0.0f,
                DefaultValue = InitialKeyValues.EmissionGravity[0],
                Channels = new List<string>
                    {
                        "X",
                    },
                DataContext = this.EmitterGravityAnimationViewModel,
                AnimationName = "EmitterEmissionGravity",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }

        /// <summary>
        /// オプションダイアログが閉じられた際に標準重力に関する項目を更新します。
        /// </summary>
        protected override void OnOptionChanged(object sender, EventArgs e)
        {
            this.OnPropertyChanged(() => this.EnableRegularGravity);
            this.OnPropertyChanged(() => this.RegularGravity);
            base.OnOptionChanged(sender, e);
        }

        /// <summary>
        /// 標準重力を適用します。
        /// </summary>
        /// <param name="parameter">使用しません。</param>
        private void ApplyRegularGravity(object parameter)
        {
            this.SetDataModelValue(CalcRegularGravity(), () => DataModel.GravityVolume);
        }

        /// <summary>
        /// 標準重力を計算します。
        /// </summary>
        /// <returns>プロジェクトコンフィグの設定値に基づいた標準重力</returns>
        private static float CalcRegularGravity()
        {
            // 単位を cm から m に変換する
            var m = (decimal)OptionStore.ProjectConfig.UnitLength / 100;
            var t = (decimal)OptionStore.ProjectConfig.FpsForRegularGravity;
            // UI 表示は 100 倍する
            var result = 9.8m / (t * t) / m * 100;
            var str = result.ToString("G3");
            return float.Parse(str);
        }
    }
}
