﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterCombinerShaderViewModel.
    /// </summary>
    public class EmitterCombinerShaderViewModel : PropertyGroupViewModel<EmitterCombinerShaderData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "EditingColorInfo",
            "EditorActive",
            "EditorTime",
            "EditingColorInfo",
            "EditorValue",
            "GroupVisibility",
        };

        /// <summary>
        /// 色編集情報
        /// </summary>
        private EditingColorInfo editingColorInfo = new EditingColorInfo();

        /// <summary>
        /// アクティブか？
        /// </summary>
        private bool colorPickerActive;

        /// <summary>
        /// 時間
        /// </summary>
        private int colorPickerTime;

        /// <summary>
        /// 値
        /// </summary>
        private float editorValue;

        /// <summary>
        /// グループの表示状態.
        /// </summary>
        private bool groupVisibility = true;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterCombinerShaderViewModel(
            HierarchyViewModel parent, EmitterCombinerShaderData dataModel)
            : base(parent, dataModel)
        {
            this.ShaderTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterCombinerShaderTypeNormal, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterCombinerShaderTypeRefraction, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterCombinerShaderTypeDistortion, 2),
            };

            this.OnGetParticleLifeExecutable = new AnonymousExecutable((o) =>
            {
                var param = o as float[];
                var emitterVm = this.FindNearestParentOfType<EmitterViewModel>();
                param[0] = emitterVm.EmitterParticleViewModel.EmitterParticleLifeViewModel.GetDataModelValue<int>("Life");
            });

            var mmfvm = new MultipleModificationFlagsViewModel(this);
            var coefficientAnimationParams = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.CoefficientAnimationTable));
            mmfvm.SetPropertyDictionary("CoefficientAnimationParams", coefficientAnimationParams);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;

            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// グループの表示状態.
        /// </summary>
        public bool GroupVisibility
        {
            get { return this.groupVisibility; }
            set { this.SetValue(ref this.groupVisibility, value); }
        }

        /// <summary>シェーダータイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> ShaderTypeItems { get; private set; }

        /// <summary>
        /// パーティクルの寿命を取得するExecutableを取得します。
        /// </summary>
        public IExecutable OnGetParticleLifeExecutable { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// シェーダータイプの切り換え時にテクスチャのチェックを行う
        /// </summary>
        [UseDataModelOriginalValue]
        public int ShaderType
        {
            get
            {
                // 値をスルーして返す
                return this.GetDataModelValue(() => this.ShaderType);
            }

            set
            {
                // 値をスルーしてセット
                var binder = new EffectMakerSetMemberBinder("ShaderType", false, false);
                this.TrySetMember(binder, value);

                // 歪みや屈折時にはリニア編集モードでも非リニアテクスチャを許容するのでステータスを更新する
                var emitterVm = this.FindNearestParentOfType<EmitterViewModel>();
                if (emitterVm != null)
                {
                    emitterVm.UpdateAssetStatus(false, false);
                    emitterVm.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel.UpdateWarningMessages(false);
                }
            }
        }

        /// <summary>
        /// 係数アニメ
        /// </summary>
        public AnimationTableData CoefficientAnimationTable
        {
            get
            {
                return this.DataModel.CoefficientAnimationTable;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.CoefficientAnimationTable);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);
                this.TrySetMember(binder, value);
            }
        }

        /// <summary>
        /// 色編集情報
        /// </summary>
        public EditingColorInfo EditingColorInfo
        {
            get
            {
                return this.editingColorInfo;
            }

            set
            {
                if (value == null)
                {
                    return;
                }

                if (ExportableViewModel.IsPasting)
                {
                    return;
                }

                this.SetValue(ref this.editingColorInfo, new EditingColorInfo(value));
                this.OnPropertyChanged(() => this.EditorActive);

                this.EditorValue = EditingColorInfo.Color.A;
                this.EditorTime = EditingColorInfo.Time;
                this.EditorActive = EditingColorInfo.Control != null;
            }
        }

        /// <summary>
        /// 編集中時間
        /// </summary>
        public int EditorEditingTime { get; set; }

        /// <summary>
        /// 編集中値
        /// </summary>
        public float EditorEditingValue { get; set; }

        /// <summary>
        /// 時間
        /// </summary>
        public int EditorTime
        {
            get
            {
                return this.colorPickerTime;
            }

            set
            {
                this.SetValue(ref this.colorPickerTime, value);

                this.EditingColorInfo.Time = value;
                this.OnPropertyChanged(() => this.EditingColorInfo);
            }
        }

        /// <summary>
        /// 値
        /// </summary>
        public float EditorValue
        {
            get
            {
                return this.editorValue;
            }

            set
            {
                this.SetValue(ref this.editorValue, value);

                this.EditingColorInfo.Color = new ColorRgba(1.0f, 1.0f, 1.0f, this.editorValue);
                this.OnPropertyChanged(() => this.EditingColorInfo);
            }
        }

        /// <summary>
        /// アクティブか？
        /// </summary>
        public bool EditorActive
        {
            get { return this.colorPickerActive;}
            set { this.SetValue(ref this.colorPickerActive, value); }
        }
    }
}
