﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Xml.Linq;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.ClipboardDataTypes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterColorColorViewModel.
    /// </summary>
    public class EmitterColorColorViewModel : ColorPickerOwnerViewModel<EmitterEmitterColorData>, IModificationFlagOwner
    {
        /// <summary>
        /// カラー挙動 - 固定
        /// </summary>
        private const int ColorBehaviorTypeFix = 0;

        /// <summary>
        /// カラー挙動 - 32キー
        /// </summary>
        private const int ColorBehaviorType32Key = 2;

        /// <summary>
        /// アルファ挙動 - 固定
        /// </summary>
        private const int AlphaBehaviorTypeFix = 0;

        /// <summary>
        /// アルファ挙動 - 32キー
        /// </summary>
        private const int AlphaBehaviorType32Key = 2;

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "Color0ValueFixColorEditorVisibility",
            "Color0Gradation32KeyEditorVisibility",
            "Color1ValueFixColorEditorVisibility",
            "Color1Gradation32KeyEditorVisibility",
            "Alpha0ValueFixColorEditorVisibility",
            "Alpha0Gradation32KeyEditorVisibility",
            "Alpha1ValueFixColorEditorVisibility",
            "Alpha1Gradation32KeyEditorVisibility",
        };

        /// <summary>
        /// カラー0アニメ折りたたみ状態
        /// </summary>
        private bool color0Fold = false;

        /// <summary>
        /// カラー1アニメ折りたたみ状態
        /// </summary>
        private bool color1Fold = false;

        /// <summary>
        /// アルファ0アニメ折りたたみ状態
        /// </summary>
        private bool alpha0Fold = false;

        /// <summary>
        /// アルファ1アニメ折りたたみ状態
        /// </summary>
        private bool alpha1Fold = false;

        /// <summary>
        /// 入れ替え作業中フラグ
        /// </summary>
        private bool isExchangingData = false;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterColorColorViewModel(
            HierarchyViewModel parent, EmitterEmitterColorData dataModel)
            : base(parent, dataModel)
        {
            this.BlockingFunc = () => this.isExchangingData;

            this.Color0SettingTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, ColorBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType32Key, ColorBehaviorType32Key),
            };
            this.Color1SettingTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, ColorBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType32Key, ColorBehaviorType32Key),
            };
            this.Alpha0SettingTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, AlphaBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType32Key, AlphaBehaviorType32Key),
            };
            this.Alpha1SettingTypeItems = new []
            {
                new KeyValuePair<string, object>(Resources.EmitterColorSettingTypeFix, AlphaBehaviorTypeFix),
                new KeyValuePair<string, object>(Resources.EmitterColorSettingType32Key, AlphaBehaviorType32Key),
            };

            this.Color0AnimationData = new EmitterColorAnimationViewModel(
                this,
                dataModel.Color0Animation,
                InitialKeyValues.Colors,
                new Action[]
                {
                    () => this.OnPropertyChanged(() => this.IsColor0AnimationModified),
                    () => this.OnPropertyChanged(() => this.ColorAnimation0),
                    () => this.OnPropertyChanged(() => this.Color0AnimationEnable),
                    () => this.OnPropertyChanged(() => this.Color0LoopMode),
                    () => this.OnPropertyChanged(() => this.Color0InterMode)
                });
            this.Children.Add(this.Color0AnimationData);

            this.Color1AnimationData = new EmitterColorAnimationViewModel(
                this,
                dataModel.Color1Animation,
                InitialKeyValues.Colors,
                new Action[]
                {
                    () => this.OnPropertyChanged(() => this.IsColor1AnimationModified),
                    () => this.OnPropertyChanged(() => this.ColorAnimation1),
                    () => this.OnPropertyChanged(() => this.Color1AnimationEnable),
                    () => this.OnPropertyChanged(() => this.Color1LoopMode),
                    () => this.OnPropertyChanged(() => this.Color1InterMode)
                });
            this.Children.Add(this.Color1AnimationData);

            this.Alpha0AnimationData = new EmitterColorAnimationViewModel(
                this,
                dataModel.Alpha0Animation,
                InitialKeyValues.Colors,
                new Action[]
                {
                    () => this.OnPropertyChanged(() => this.IsAlpha0AnimationModified),
                    () => this.OnPropertyChanged(() => this.AlphaAnimation0),
                    () => this.OnPropertyChanged(() => this.Alpha0AnimationEnable),
                    () => this.OnPropertyChanged(() => this.Alpha0LoopMode),
                    () => this.OnPropertyChanged(() => this.Alpha0InterMode)
                });
            this.Children.Add(this.Alpha0AnimationData);

            this.Alpha1AnimationData = new EmitterColorAnimationViewModel(
                this,
                dataModel.Alpha1Animation,
                InitialKeyValues.Colors,
                new Action[]
                {
                    () => this.OnPropertyChanged(() => this.IsAlpha1AnimationModified),
                    () => this.OnPropertyChanged(() => this.AlphaAnimation1),
                    () => this.OnPropertyChanged(() => this.Alpha1AnimationEnable),
                    () => this.OnPropertyChanged(() => this.Alpha1LoopMode),
                    () => this.OnPropertyChanged(() => this.Alpha1InterMode)
                });
            this.Children.Add(this.Alpha1AnimationData);

            this.OnPropertyChanged(() => this.Color0BehaviorType);
            this.OnPropertyChanged(() => this.Color1BehaviorType);
            this.OnPropertyChanged(() => this.Alpha0BehaviorType);
            this.OnPropertyChanged(() => this.Alpha1BehaviorType);

            this.CopyColorExecutable = new AnonymousExecutable(this.OnCopyColor);
            this.PasteColorExecutable = new AnonymousExecutable(this.OnPasteColor);

            this.OnResetExecutable = new AnonymousExecutable(this.ResetTable);

            this.DisableColorPicker();

            var mmfvm = new MultipleModificationFlagsViewModel(this);

            // Color 0
            var emitterColor0Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Color0BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Color0Value),
                ViewModelBase.NameOf(() => this.IsColor0AnimationModified));
            mmfvm.SetPropertyDictionary("EmitterColor0Params", emitterColor0Params);
            mmfvm.SetModificationDictionary("EmitterColor0Params", new IModificationPropertyOwner[] { this.Color0AnimationData });

            // Color 1
            var emitterColor1Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Color1BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Color1Value),
                ViewModelBase.NameOf(() => this.IsColor1AnimationModified));
            mmfvm.SetPropertyDictionary("EmitterColor1Params", emitterColor1Params);
            mmfvm.SetModificationDictionary("EmitterColor1Params", new IModificationPropertyOwner[] { this.Color1AnimationData });

            // Alpha 0
            var emitterAlpha0Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Alpha0BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Alpha0Value),
                ViewModelBase.NameOf(() => this.IsAlpha0AnimationModified));
            mmfvm.SetPropertyDictionary("EmitterAlpha0Params", emitterAlpha0Params);
            mmfvm.SetModificationDictionary("EmitterAlpha0Params", new IModificationPropertyOwner[] { this.Alpha0AnimationData });

            // Alpha 1
            var emitterAlpha1Params = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.Alpha1BehaviorType),
                ViewModelBase.NameOf(() => dataModel.Alpha1Value),
                ViewModelBase.NameOf(() => this.IsAlpha1AnimationModified));
            mmfvm.SetPropertyDictionary("EmitterAlpha1Params", emitterAlpha1Params);
            mmfvm.SetModificationDictionary("EmitterAlpha1Params", new IModificationPropertyOwner[] { this.Alpha1AnimationData });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
            this.ModificationFlagViewModel.AddIgnoreProperties(this.IgnoreColorPickerPropertyNames);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// Color0のAnimationDataが変更されたかどうか
        /// </summary>
        public bool IsColor0AnimationModified
        {
            get { return this.Color0AnimationData.IsModified;  }
        }

        /// <summary>
        /// Color1のAnimationDataが変更されたかどうか
        /// </summary>
        public bool IsColor1AnimationModified
        {
            get { return this.Color1AnimationData.IsModified; }
        }

        /// <summary>
        /// Alpha0のAnimationDataが変更されたかどうか
        /// </summary>
        public bool IsAlpha0AnimationModified
        {
            get { return this.Alpha0AnimationData.IsModified; }
        }

        /// <summary>
        /// Alpha1のAnimationDataが変更されたかどうか
        /// </summary>
        public bool IsAlpha1AnimationModified
        {
            get { return this.Alpha1AnimationData.IsModified; }
        }

        /// <summary>Color0挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Color0SettingTypeItems { get; private set; }

        /// <summary>Color1挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Color1SettingTypeItems { get; private set; }

        /// <summary>Alpha0挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Alpha0SettingTypeItems { get; private set; }

        /// <summary>Alpha1挙動タイプの項目を取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> Alpha1SettingTypeItems { get; private set; }

        /// <summary>
        /// カラー0アニメ有効無効フラグ
        /// </summary>
        public bool Color0AnimationEnable
        {
            get
            {
                return this.Color0BehaviorType == ColorBehaviorType32Key;
            }

            set
            {
                this.Color0BehaviorType = value ? ColorBehaviorType32Key : ColorBehaviorTypeFix;
            }
        }

        /// <summary>
        /// カラー1アニメ有効無効フラグ
        /// </summary>
        public bool Color1AnimationEnable
        {
            get
            {
                return this.Color1BehaviorType == ColorBehaviorType32Key;
            }

            set
            {
                this.Color1BehaviorType = value ? ColorBehaviorType32Key : ColorBehaviorTypeFix;
            }
        }

        /// <summary>
        /// アルファ0アニメ有効無効フラグ
        /// </summary>
        public bool Alpha0AnimationEnable
        {
            get
            {
                return this.Alpha0BehaviorType == AlphaBehaviorType32Key;
            }

            set
            {
                this.Alpha0BehaviorType = value ? AlphaBehaviorType32Key : AlphaBehaviorTypeFix;
            }
        }

        /// <summary>
        /// アルファ1アニメ有効無効フラグ
        /// </summary>
        public bool Alpha1AnimationEnable
        {
            get
            {
                return this.Alpha1BehaviorType == AlphaBehaviorType32Key;
            }

            set
            {
                this.Alpha1BehaviorType = value ? AlphaBehaviorType32Key : AlphaBehaviorTypeFix;
            }
        }

        #region カラー挙動タイプ

        /// <summary>
        /// カラー0挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Color0BehaviorType
        {
            get
            {
                return this.GetDataModelValue(() => this.Color0BehaviorType);
            }

            set
            {
                if (this.Color0BehaviorType == value)
                {
                    return;
                }

                using (new CommandCombiner())
                {
                    this.SetDataModelValue(value, () => this.Color0BehaviorType);

                    this.OnPropertyChanged(() => this.Color0ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Color0Gradation32KeyEditorVisibility);
                    if (this.Color0BehaviorType != 2)
                    {
                        this.DisableColorPicker();
                    }

                    this.OnPropertyChanged(() => this.Color0AnimationEnable);

                    this.Color0AnimationData.EnableAnimation = value == ColorBehaviorType32Key;
                }
            }
        }

        /// <summary>
        /// カラー0固定カラーコントロールを表示するか？
        /// </summary>
        public bool Color0ValueFixColorEditorVisibility
        {
            get { return this.Color0BehaviorType == EmitterColorColorViewModel.ColorBehaviorTypeFix; }
        }

        /// <summary>
        /// カラー0８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Color0Gradation32KeyEditorVisibility
        {
            get { return this.Color0BehaviorType == EmitterColorColorViewModel.ColorBehaviorType32Key; }
        }

        /// <summary>
        /// カラー1挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Color1BehaviorType
        {
            get
            {
                return this.GetDataModelValue(() => this.Color1BehaviorType);
            }

            set
            {
                if (this.Color1BehaviorType == value)
                {
                    return;
                }

                using (new CommandCombiner())
                {
                    this.SetDataModelValue(value, () => this.Color1BehaviorType);

                    this.OnPropertyChanged(() => this.Color1ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Color1Gradation32KeyEditorVisibility);
                    if (this.Color1BehaviorType != 2)
                    {
                        this.DisableColorPicker();
                    }

                    this.OnPropertyChanged(() => this.Color1AnimationEnable);

                    this.Color1AnimationData.EnableAnimation = value == ColorBehaviorType32Key;
                }
            }
        }

        /// <summary>
        /// カラー1固定カラーコントロールを表示するか？
        /// </summary>
        public bool Color1ValueFixColorEditorVisibility
        {
            get { return this.Color1BehaviorType == EmitterColorColorViewModel.ColorBehaviorTypeFix; }
        }

        /// <summary>
        /// カラー1８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Color1Gradation32KeyEditorVisibility
        {
            get { return this.Color1BehaviorType == EmitterColorColorViewModel.ColorBehaviorType32Key; }
        }

        #endregion

        #region アルファ挙動タイプ

        /// <summary>
        /// アルファ0挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Alpha0BehaviorType
        {
            get
            {
                return this.GetDataModelValue(() => this.Alpha0BehaviorType);
            }

            set
            {
                if (this.Alpha0BehaviorType == value)
                {
                    return;
                }

                using (new CommandCombiner())
                {
                    this.SetDataModelValue(value, () => this.Alpha0BehaviorType);

                    this.OnPropertyChanged(() => this.Alpha0ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Alpha0Gradation32KeyEditorVisibility);
                    if (this.Alpha0BehaviorType != 2)
                    {
                        this.DisableColorPicker();
                    }

                    this.OnPropertyChanged(() => this.Alpha0AnimationEnable);

                    this.Alpha0AnimationData.EnableAnimation = value == ColorBehaviorType32Key;
                }
            }
        }

        /// <summary>
        /// アルファ1固定カラーコントロールを表示するか？
        /// </summary>
        public bool Alpha0ValueFixColorEditorVisibility
        {
            get { return this.Alpha0BehaviorType == EmitterColorColorViewModel.AlphaBehaviorTypeFix; }
        }

        /// <summary>
        /// アルファ1８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Alpha0Gradation32KeyEditorVisibility
        {
            get { return this.Alpha0BehaviorType == EmitterColorColorViewModel.AlphaBehaviorType32Key; }
        }

        /// <summary>
        /// アルファ1挙動タイプを設定または取得します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int Alpha1BehaviorType
        {
            get
            {
                return this.GetDataModelValue(() => this.Alpha1BehaviorType);
            }

            set
            {
                if (this.Alpha1BehaviorType == value)
                {
                    return;
                }

                using (new CommandCombiner())
                {
                    this.SetDataModelValue(value, () => this.Alpha1BehaviorType);

                    this.OnPropertyChanged(() => this.Alpha1ValueFixColorEditorVisibility);
                    this.OnPropertyChanged(() => this.Alpha1Gradation32KeyEditorVisibility);
                    if (this.Alpha1BehaviorType != 2)
                    {
                        this.DisableColorPicker();
                    }

                    this.OnPropertyChanged(() => this.Alpha1AnimationEnable);
                    this.Alpha1AnimationData.EnableAnimation = value == ColorBehaviorType32Key;
                }
            }
        }

        /// <summary>
        /// アルファ1固定カラーコントロールを表示するか？
        /// </summary>
        public bool Alpha1ValueFixColorEditorVisibility
        {
            get { return this.Alpha1BehaviorType == EmitterColorColorViewModel.AlphaBehaviorTypeFix; }
        }

        /// <summary>
        /// アルファ1８キーアニメコントロールを表示するか？
        /// </summary>
        public bool Alpha1Gradation32KeyEditorVisibility
        {
            get { return this.Alpha1BehaviorType == EmitterColorColorViewModel.AlphaBehaviorType32Key; }
        }
        #endregion

        #region カラー値
        /// <summary>
        /// カラー0
        /// </summary>
        [UseDataModelOriginalValue]
        public ColorRgba Color0Value
        {
            get
            {
                return this.DataModel.Color0Value;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Color0Value);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);

                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// カラー1
        /// </summary>
        [UseDataModelOriginalValue]
        public ColorRgba Color1Value
        {
            get
            {
                return this.DataModel.Color1Value;
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Color1Value);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);

                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// カラーアニメーション0
        /// </summary>
        public AnimationTableData ColorAnimation0
        {
            get
            {
                return this.DataModel.Color0Animation.AnimationTable;
            }

            set
            {
                var name = "AnimationTable";
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);
                int count = this.ColorAnimation0.Count;
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.Color0AnimationData.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// カラーアニメーション1
        /// </summary>
        public AnimationTableData ColorAnimation1
        {
            get
            {
                return this.DataModel.Color1Animation.AnimationTable;
            }

            set
            {
                var name = "AnimationTable";
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);
                int count = this.ColorAnimation1.Count;
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.Color1AnimationData.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// カラーアニメーション0データ
        /// </summary>
        public EmitterColorAnimationViewModel Color0AnimationData { get; private set; }

        /// <summary>
        /// カラーアニメーション1データ
        /// </summary>
        public EmitterColorAnimationViewModel Color1AnimationData { get; private set; }

        #endregion

        #region アルファ値
        /// <summary>
        /// アルファ0
        /// </summary>
        [UseDataModelOriginalValue]
        public float Alpha0Value
        {
            get
            {
                return this.GetDataModelValue(() => this.Alpha0Value);
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Alpha0Value);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);
                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// アルファ1
        /// </summary>
        [UseDataModelOriginalValue]
        public float Alpha1Value
        {
            get
            {
                return this.GetDataModelValue(() => this.Alpha1Value);
            }

            set
            {
                var name = ViewModelBase.NameOf(() => this.Alpha1Value);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);
                this.TrySetMember(binder, value);
                this.RefreshColorPicker(name, value);
            }
        }

        /// <summary>
        /// アルファアニメーション0
        /// </summary>
        public AnimationTableData AlphaAnimation0
        {
            get
            {
                return this.DataModel.Alpha0Animation.AnimationTable;
            }

            set
            {
                var name = "AnimationTable";
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);
                int count = this.AlphaAnimation0.Count;
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.Alpha0AnimationData.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// アルファアニメーション1
        /// </summary>
        public AnimationTableData AlphaAnimation1
        {
            get
            {
                return this.DataModel.Alpha1Animation.AnimationTable;
            }

            set
            {
                var name = "AnimationTable";
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(name, false, false);
                int count = this.AlphaAnimation1.Count;
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.Alpha1AnimationData.TrySetMember(binder, value),
                    value.Count != count,
                    this.DataModel);
            }
        }

        /// <summary>
        /// アルファアニメーション0データ
        /// </summary>
        public EmitterColorAnimationViewModel Alpha0AnimationData { get; private set; }

        /// <summary>
        /// アルファアニメーション1データ
        /// </summary>
        public EmitterColorAnimationViewModel Alpha1AnimationData { get; private set; }

        #endregion

        /// <summary>
        /// カラー0のループモードです。
        /// </summary>
        public bool Color0LoopMode
        {
            get { return this.Color0AnimationData.LoopMode == 1; }
            set { this.Color0AnimationData.LoopMode = value ? 1 : 0; }
        }

        /// <summary>
        /// カラー1のループモードです。
        /// </summary>
        public bool Color1LoopMode
        {
            get { return this.Color1AnimationData.LoopMode == 1; }
            set { this.Color1AnimationData.LoopMode = value ? 1 : 0; }
        }

        /// <summary>
        /// アルファ0のループモードです。
        /// </summary>
        public bool Alpha0LoopMode
        {
            get { return this.Alpha0AnimationData.LoopMode == 1; }
            set { this.Alpha0AnimationData.LoopMode = value ? 1 : 0; }
        }

        /// <summary>
        /// アルファ1のループモードです。
        /// </summary>
        public bool Alpha1LoopMode
        {
            get { return this.Alpha1AnimationData.LoopMode == 1; }
            set { this.Alpha1AnimationData.LoopMode = value ? 1 : 0; }
        }

        /// <summary>
        /// カラー0の補間方式です。
        /// </summary>
        public int Color0InterMode
        {
            get { return this.Color0AnimationData.InterpolationMode; }
            set { this.Color0AnimationData.InterpolationMode = value; }
        }

        /// <summary>
        /// カラー1の補間方式です。
        /// </summary>
        public int Color1InterMode
        {
            get { return this.Color1AnimationData.InterpolationMode; }
            set { this.Color1AnimationData.InterpolationMode = value; }
        }

        /// <summary>
        /// アルファ0の補間方式です。
        /// </summary>
        public int Alpha0InterMode
        {
            get { return this.Alpha0AnimationData.InterpolationMode; }
            set { this.Alpha0AnimationData.InterpolationMode = value; }
        }

        /// <summary>
        /// アルファ1の補間方式です。
        /// </summary>
        public int Alpha1InterMode
        {
            get { return this.Alpha1AnimationData.InterpolationMode; }
            set { this.Alpha1AnimationData.InterpolationMode = value; }
        }

        /// <summary>
        /// カラー0アニメ折りたたみ状態
        /// </summary>
        public bool Color0Fold
        {
            get
            {
                return this.color0Fold;
            }

            set
            {
                this.color0Fold = value;
                this.OnPropertyChanged(() => this.Color0Expanded);
            }
        }

        /// <summary>
        /// カラー0アニメ展開状態
        /// </summary>
        public bool Color0Expanded
        {
            get { return !this.Color0Fold; }
        }

        /// <summary>
        /// カラー1アニメ折りたたみ状態
        /// </summary>
        public bool Color1Fold
        {
            get
            {
                return this.color1Fold;
            }

            set
            {
                this.color1Fold = value;
                this.OnPropertyChanged(() => this.Color1Expanded);
            }
        }

        /// <summary>
        /// カラー1アニメ展開状態
        /// </summary>
        public bool Color1Expanded
        {
            get { return !this.Color1Fold; }
        }

        /// <summary>
        /// アルファ0アニメ折りたたみ状態
        /// </summary>
        public bool Alpha0Fold
        {
            get
            {
                return this.alpha0Fold;
            }

            set
            {
                this.alpha0Fold = value;
                this.OnPropertyChanged(() => this.Alpha0Expanded);
            }
        }

        /// <summary>
        /// アルファ0アニメ展開状態
        /// </summary>
        public bool Alpha0Expanded
        {
            get { return !this.Alpha0Fold; }
        }

        /// <summary>
        /// アルファ1アニメ折りたたみ状態
        /// </summary>
        public bool Alpha1Fold
        {
            get
            {
                return this.alpha1Fold;
            }

            set
            {
                this.alpha1Fold = value;
                this.OnPropertyChanged(() => this.Alpha1Expanded);
            }
        }

        /// <summary>
        /// アルファ1アニメ展開状態
        /// </summary>
        public bool Alpha1Expanded
        {
            get { return !this.Alpha1Fold; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// Get an executable to run when reset.
        /// </summary>
        public IExecutable OnResetExecutable { get; private set; }

        /// <summary>
        /// Gets the Copy executable instance.
        /// </summary>
        public IExecutable CopyColorExecutable { get; private set; }

        /// <summary>
        /// Gets the Paste executable instance.
        /// </summary>
        public IExecutable PasteColorExecutable { get; private set; }

        /// <summary>
        /// データの０と１を交換する
        /// </summary>
        /// <param name="kind">カラー要素</param>
        public void ExchangeData(ColorElementType kind)
        {
            MultiNodeEditUtil.DisableMultiNodeEdit = true;
            this.isExchangingData = true;
            using (new AnonymousDisposable(() =>
            {
                this.isExchangingData = false;
                MultiNodeEditUtil.DisableMultiNodeEdit = false;
            }))
            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            using (new CommandCombiner())
            {
                if (kind == ColorElementType.Color)
                {
                    var tmpBehavior = this.Color0BehaviorType;
                    var tmpValue = this.Color0Value.Clone() as ColorRgba;
                    var tmpTable = this.Color0AnimationData.AnimationTable.Clone() as AnimationTableData;
                    var tmpLoop = this.Color0LoopMode;
                    var tmpInter = this.Color0InterMode;

                    this.Color0BehaviorType = this.Color1BehaviorType;
                    this.SetDataModelValue((ColorRgba)this.Color1Value.Clone(), () => this.Color0Value);
                    this.Color0AnimationData.SetDataModelValue(this.Color1AnimationData.AnimationTable.Clone(), "AnimationTable");
                    this.Color0AnimationData.LoopMode = this.Color1LoopMode ? 1 : 0;
                    this.Color0AnimationData.InterpolationMode = this.Color1InterMode;

                    this.Color1BehaviorType = tmpBehavior;
                    this.SetDataModelValue(tmpValue, () => this.Color1Value);
                    this.Color1AnimationData.SetDataModelValue(tmpTable, "AnimationTable");
                    this.Color1AnimationData.LoopMode = tmpLoop ? 1 : 0;
                    this.Color1AnimationData.InterpolationMode = tmpInter;
                }
                else if (kind == ColorElementType.Alpha)
                {
                    var tmpBehavior = this.Alpha0BehaviorType;
                    var tmpValue = this.Alpha0Value;
                    var tmpTable = this.Alpha0AnimationData.AnimationTable.Clone() as AnimationTableData;
                    var tmpLoop = this.Alpha0LoopMode;
                    var tmpInter = this.Alpha0InterMode;

                    this.Alpha0BehaviorType = this.Alpha1BehaviorType;
                    this.SetDataModelValue(this.Alpha1Value, () => this.Alpha0Value);
                    this.Alpha0AnimationData.SetDataModelValue(this.Alpha1AnimationData.AnimationTable.Clone(), "AnimationTable");
                    this.Alpha0AnimationData.LoopMode = this.Alpha1LoopMode ? 1 : 0;
                    this.Alpha0AnimationData.InterpolationMode = this.Alpha1InterMode;

                    this.Alpha1BehaviorType = tmpBehavior;
                    this.SetDataModelValue(tmpValue, () => this.Alpha1Value);
                    this.Alpha1AnimationData.SetDataModelValue(tmpTable, "AnimationTable");
                    this.Alpha1AnimationData.LoopMode = tmpLoop ? 1 : 0;
                    this.Alpha1AnimationData.InterpolationMode = tmpInter;
                }
            }
        }

        /// <summary>
        /// コピーデータにパッチを当てる
        /// </summary>
        /// <param name="dstTarget">変更対象</param>
        /// <param name="current">現在の状態</param>
        /// <param name="copyData">コピーデータ</param>
        /// <returns>パッチ済みデータ</returns>
        private static string PatchCopyData(ColorTargetType dstTarget, string current, ColorUnitClipboardData copyData)
        {
            //// ■処理内容
            ////   copyData.Body の copyData.Target 位置のパラメータを
            ////   current の target 位置に上書きする

            var srcXml = XElement.Parse(copyData.Body);
            var dstXml = XElement.Parse(current);
            var srcTarget = copyData.Target;

            Action<string> paste = parent =>
            {
                Action<string> pasteValues = (target) =>
                {
                    var src = srcXml.Element(parent)
                            .Elements()
                            .Where(x => x.Name == "lp")
                            .FirstOrDefault(
                                x => x.Attribute("name").Value == MakeCopyDataPostFixString(srcTarget, target));
                    var dst = dstXml.Element(parent)
                            .Elements()
                            .Where(x => x.Name == "lp")
                            .FirstOrDefault(
                                x => x.Attribute("name").Value == MakeCopyDataPostFixString(dstTarget, target));

                    if ((src != null) && (dst != null))
                    {
                        dst.Attribute("value").Value = src.Attribute("value").Value;
                    }
                };

                Action<Func<ColorTargetType, string>> pasteElements = makeString =>
                {
                    var src = srcXml.Element(parent)
                            .Elements()
                            .Where(x => x.Name == "IXmlSerializable")
                            .FirstOrDefault(x => x.Attribute("name").Value == makeString(srcTarget));
                    var dst = dstXml.Element(parent)
                            .Elements()
                            .Where(x => x.Name == "IXmlSerializable")
                            .FirstOrDefault(x => x.Attribute("name").Value == makeString(dstTarget));

                    if ((src != null) && (dst != null))
                    {
                        dst.Elements().Remove();
                        dst.Add(src.Elements());
                    }
                };

                // BehaviorType
                pasteValues("BehaviorType");

                // Value
                pasteElements(t => MakeCopyDataPostFixString(t, "Value"));

                // Animation(DM)
                pasteElements(t => MakeCopyDataPostFixString(t, "Animation"));

                // Animation(VM)
                pasteElements(MakeCopyDataAnimationString);

                // Loop
                pasteValues("LoopMode");

                // Inter
                pasteValues("InterMode");
            };

            paste("data");
            paste("vm");

            return dstXml.ToString();
        }

        /// <summary>
        /// 色名にポストフィックスを付加した文字列の生成
        /// </summary>
        /// <param name="target">対象</param>
        /// <param name="postFix">The post Fix.</param>
        /// <returns>文字列</returns>
        private static string MakeCopyDataPostFixString(ColorTargetType target, string postFix)
        {
            switch (target)
            {
                case ColorTargetType.GlobalColor0:
                    return "Color0" + postFix;
                case ColorTargetType.GlobalColor1:
                    return "Color1" + postFix;
                case ColorTargetType.GlobalAlpha0:
                    return "Alpha0" + postFix;
                case ColorTargetType.GlobalAlpha1:
                    return "Alpha1" + postFix;
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// Animation文字列の生成
        /// </summary>
        /// <param name="target">対象</param>
        /// <returns>文字列</returns>
        private static string MakeCopyDataAnimationString(ColorTargetType target)
        {
            switch (target)
            {
                case ColorTargetType.GlobalColor0:
                    return "ColorAnimation0";
                case ColorTargetType.GlobalColor1:
                    return "ColorAnimation1";
                case ColorTargetType.GlobalAlpha0:
                    return "AlphaAnimation0";
                case ColorTargetType.GlobalAlpha1:
                    return "AlphaAnimation1";
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// アニメーションテーブルのリセット処理
        /// </summary>
        /// <param name="parameter">リセット対象を文字列で指定(Color0,Color1,Alpha0,Alpha1)</param>
        private void ResetTable(object parameter)
        {
            var initialValue = new AnimationTableData(0, InitialKeyValues.Colors);
            var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                "AnimationTable",
                false,
                true);
            using (new MessageBlockerWithSendBinaryOnce(this.DataModel))
            using (new CommandCombiner())
            {
                var target = (string)parameter;
                switch (target)
                {
                    case "Color0":
                        this.Color0LoopMode = false;
                        this.Color0InterMode = 0;
                        this.Color0AnimationData.TrySetMember(binder, initialValue);
                        this.OnPropertyChanged(() => this.ColorAnimation0);
                        break;
                    case "Color1":
                        this.Color1LoopMode = false;
                        this.Color1InterMode = 0;
                        this.Color1AnimationData.TrySetMember(binder, initialValue);
                        this.OnPropertyChanged(() => this.ColorAnimation1);
                        break;
                    case "Alpha0":
                        this.Alpha0LoopMode = false;
                        this.Alpha0InterMode = 0;
                        this.Alpha0AnimationData.TrySetMember(binder, initialValue);
                        this.OnPropertyChanged(() => this.AlphaAnimation0);
                        break;
                    case "Alpha1":
                        this.Alpha1LoopMode = false;
                        this.Alpha1InterMode = 0;
                        this.Alpha1AnimationData.TrySetMember(binder, initialValue);
                        this.OnPropertyChanged(() => this.AlphaAnimation1);
                        break;
                }
            }
        }

        /// <summary>
        /// Called when the Copy executable is run.
        /// </summary>
        /// <param name="param">パラメータ</param>
        private void OnCopyColor(object param)
        {
            var copyData = new ColorUnitClipboardData
            {
                Target = (ColorTargetType)param,
                Body = this.ExportValuesAsXml()
            };

            Clipboard.SetData(ColorUnitClipboardData.ClipboardFormat, copyData);
        }

        /// <summary>
        /// Called when the Paste executable is run.
        /// </summary>
        /// <param name="param">パラメータ</param>
        private void OnPasteColor(object param)
        {
            if (Clipboard.ContainsData(ColorUnitClipboardData.ClipboardFormat) == false)
            {
                return;
            }

            var copyData = Clipboard.GetData(ColorUnitClipboardData.ClipboardFormat) as ColorUnitClipboardData;

            System.Diagnostics.Debug.Assert(
                ColorUtility.MakeColorElementType(copyData.Target) == ColorUtility.MakeColorElementType((ColorTargetType)param),
                "色要素が食い違っている");

            var current = this.ExportValuesAsXml();
            string newValue = PatchCopyData((ColorTargetType)param, current, copyData);

            CommandManager.Execute(new PasteGroupCommand(this, current, newValue));
        }
    }
}
