﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterBasicInheritanceViewModel.
    /// </summary>
    public class EmitterBasicInheritanceViewModel : PropertyGroupViewModel<EmitterBasicInheritanceData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "IsChild",
        };

        /// <summary>
        /// 親エミッタ時に変更を無視するプロパティ名のリストです。
        /// </summary>
        private readonly string[] ignoresOnParentEmitter = new string[]
        {
            "EnableDrawPath", "EnableColor0", "EnableColor1", "EnableAlpha0",　"EnableAlpha1", "EnableScale",　"EnableRotation"
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterBasicInheritanceViewModel(
            HierarchyViewModel parent, EmitterBasicInheritanceData dataModel)
            : base(parent, dataModel)
        {
            this.OverrideParentEmitterSetting();

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var mmfvm = new MultipleModificationFlagsViewModel(this);
            mmfvm.SetPropertyDictionary("EnableColor0or1", new [] { "EnableColor0", "EnableColor1" });
            mmfvm.SetPropertyDictionary("EnableAlpha0Set", new [] { "EnableAlpha0", "EnableAlpha0EachFrame" });
            mmfvm.SetPropertyDictionary("EnableAlpha1Set", new [] { "EnableAlpha1", "EnableAlpha1EachFrame" });
            this.ModificationFlagViewModel = mmfvm;
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);

            var emitterViewModel = this.Parent.Parent as EmitterViewModel;

            // 階層化の解除を行ったときに表示を更新する
            if (emitterViewModel != null)
            {
                emitterViewModel.PropertyChanged += (s, e) =>
                {
                    if (e.PropertyName == "Parent")
                    {
                        base.FirePropertyChanges();
                    }
                };
            }
        }

        /// <summary>
        /// このエミッタがチャイルドならtrue、そうじゃなかったらfalse.
        /// </summary>
        public bool IsChild
        {
            get
            {
                var evm = this.Parent.Parent.Parent as EmitterViewModel;
                if (evm != null)
                {
                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 親エミッタでこっそり上書きする処理をコピペの際に走るよう引っ掛けます
        /// </summary>
        public override void FirePropertyChanges()
        {
            this.OverrideParentEmitterSetting();
            base.FirePropertyChanges();
        }

        /// <summary>
        /// 親エミッタで上書きしている継承設定を無視するか否かを返します。
        /// </summary>
        /// <param name="propertyName">プロパティ名</param>
        /// <returns>無視するならtrue,そうでなければfalse.</returns>
        public bool IsIgnoreMemberOnParent(string propertyName)
        {
            return !this.IsChild && this.ignoresOnParentEmitter.Contains(propertyName);
        }

        /// <summary>
        /// 親エミッタでこっそり上書きされるべきデータを上書きします。
        /// </summary>
        private void OverrideParentEmitterSetting()
        {
            if (!this.IsChild)
            {
                this.DataModel.EnableDrawPath = false;
                this.DataModel.EnableColor0 = false;
                this.DataModel.EnableColor1 = false;
                this.DataModel.EnableAlpha0 = false;
                this.DataModel.EnableAlpha1 = false;
                this.DataModel.EnableScale = false;
                this.DataModel.EnableRotation = false;
            }
        }
    }
}
