﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Properties;
using NameValueItem = System.Collections.Generic.KeyValuePair<string, object>;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterBasicBasicViewModel.
    /// </summary>
    public class EmitterBasicBasicViewModel : PropertyGroupViewModel<EmitterBasicBasicData>, IModificationFlagOwner
    {
        /// <summary>The name value items of particle follow types for UI.</summary>
        private static readonly NameValueItem[] ConstParticleFollowTypeItems =
        {
            new NameValueItem(Resources.EmitterBasicBehaviorFollowTypeAll, 0),
            new NameValueItem(Resources.EmitterBasicBehaviorFollowTypeNone, 1),
            new NameValueItem(Resources.EmitterBasicBehaviorFollowTypePos, 2),
        };

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "CanUsingGpu",
            "ProcessType",
        };

        /// <summary>
        /// 親のビューモデル
        /// </summary>
        private readonly EmitterBasicViewModel parentViewModel = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterBasicBasicViewModel(
            HierarchyViewModel parent, EmitterBasicBasicData dataModel)
            : base(parent, dataModel)
        {
            this.parentViewModel = (EmitterBasicViewModel)this.Parent;

            SpecManager.CurrentSpecChanged += this.SpecManagerOnCurrentSpecChanged;

            this.OnPropertyChanged(() => this.CanUsingGpu);

            // Always create the modification flag view model IN THE END of the constructor,
            // in case any initialization triggers the modification events.
            var applyParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.EnableEndProcess),
                ViewModelBase.NameOf(() => dataModel.EnableAlphaFade));
            var mmfvm = new MultipleModificationFlagsViewModel(this);
            mmfvm.SetPropertyDictionary("EnableEndAndFade", applyParamsProperties);
            this.ModificationFlagViewModel = mmfvm;
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// オブジェクトを破棄する際にイベントハンドラを解除します。
        /// </summary>
        public override void Dispose()
        {
            SpecManager.CurrentSpecChanged -= this.SpecManagerOnCurrentSpecChanged;
            base.Dispose();
        }

        /// <summary>パーティクル追従タイプを取得します.</summary>
        public IEnumerable<KeyValuePair<string, object>> ParticleFollowTypeItems
        {
            get { return ConstParticleFollowTypeItems; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// CPUで動作させるかどうかを表すフラグ。
        /// OFFの場合は状況に応じてGPUかSOかを選択します。
        /// </summary>
        public bool IsUsingCpu
        {
            get
            {
                var value = this.GetDataModelValue(() => this.DataModel.ProcessType);
                return value == 0;
            }

            set
            {
                var canUsingGpu = this.CanUsingGpu;
                if (value || !canUsingGpu)
                {
                    // CPUの使用を明示された、またはGPU利用不可状態の場合は自動的にCPUモードへ
                    var binder = new EffectMakerSetMemberBinder("ProcessType", false, false);
                    this.TrySetMember(binder, 0);
                }
                else
                {
                    var emitterVm = GetParent<EmitterViewModel>(this);
                    var binder = new EffectMakerSetMemberBinder("ProcessType", false, false);
                    bool hasField = emitterVm.Children.OfType<FieldViewModel>().Any(vm => vm.EnableConvert);
                    var pluginChild = emitterVm.Children.OfType<ReservedShaderNodeViewModel>()
                        .Where(vm => vm.EnableConvert);

                    // フィールドか、SO限定のプラグインがいる場合はSO、そうでなければGPUモードへ
                    this.TrySetMember(
                        binder, hasField || pluginChild.Any(n => !n.EnableGpu && n.EnableSo) ? 2 : 1);
                }

                // EffectBasicViewModelのIsUsingCpuプロパティも更新する
                this.parentViewModel.EmitterBasicRenderViewModel.UpdateZSortCommentVisibility();
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// GPUが使える状態か否かを更新して取得します。
        /// CPU強制やSOへの振り分けも行います。
        /// </summary>
        public bool CanUsingGpu
        {
            get
            {
                if (SpecManager.CurrentSpec.ForceCpuCalc)
                {
                    return false;
                }

                var emitterVm = GetParent<EmitterViewModel>(this);
                if (emitterVm != null)
                {
                    var pluginChild = emitterVm.Children.OfType<ReservedShaderNodeViewModel>()
                        .Where(vm => vm.EnableConvert).ToArray();

                    bool forceCpu = emitterVm.Children.OfType<EmitterViewModel>().Any(vm => vm.EnableConvert);
                    forceCpu |= pluginChild.Any(n => !n.EnableGpu && !n.EnableSo);
                    forceCpu |= emitterVm.Children.OfType<CustomActionViewModel>().Any(vm => vm.EnableConvert);

                    bool hasFieldChild = emitterVm.Children.OfType<FieldViewModel>().Any(vm => vm.EnableConvert);

                    if (!this.IsUsingCpu)
                    {
                        var binder = new EffectMakerSetMemberBinder("ProcessType", false, false);
                        if (forceCpu)
                        {
                            // 前の判定でCPU強制になるようならCPUモードで確定
                            this.TrySetMember(binder, 0);
                        }
                        else if (hasFieldChild || pluginChild.Any(n => !n.EnableGpu && n.EnableSo))
                        {
                            // フィールドか、SO限定のプラグインがいるかでいったん振り分け
                            if (pluginChild.Any(n => !n.EnableSo))
                            {
                                // この分岐でSO不可のプラグインが混在していたらCPUモードにする
                                this.TrySetMember(binder, 0);
                                forceCpu = true;
                            }
                            else
                            {
                                // 混在が無ければSOモードで確定
                                this.TrySetMember(binder, 2);
                            }
                        }
                        else
                        {
                            // プラグインがあり、GPUが不可なものだったらCPUに戻す
                            if (pluginChild.Any() && pluginChild.Any(n => !n.EnableGpu))
                            {
                                this.TrySetMember(binder, 0);
                                forceCpu = true;
                            }
                            else
                            {
                                this.TrySetMember(binder, 1);
                            }
                        }

                        this.OnPropertyChanged(() => this.IsUsingCpu);
                    }

                    return !forceCpu;
                }

                return true;
            }
        }

        /// <summary>
        /// GPUを使える状態か否かを更新します。
        /// </summary>
        /// <param name="param">更新後にGPUモードを切り替えたい(Undo用)時に値を渡します。</param>
        /// <returns>状態更新前のGPUモードを返します。</returns>
        public int RaiseCanUsingGpu(int param = -1)
        {
            var result = this.GetDataModelValue(() => this.DataModel.ProcessType);
            this.OnPropertyChanged(() => this.CanUsingGpu);

            if (param != -1)
            {
                var binder = new EffectMakerSetMemberBinder("ProcessType", false, false);
                this.TrySetMember(binder, param);
            }

            this.OnPropertyChanged(() => this.IsUsingCpu);

            // CanUsingGpuのgetterを呼んでProcessTypeの更新を促す
            bool canUsingGpu = this.CanUsingGpu;

            return result;
        }

        /// <summary>
        /// スペック切り換え時にGPU利用の可否を更新します。
        /// </summary>
        /// <param name="sender">使用しません。</param>
        /// <param name="eventArgs">使用しません。</param>
        private void SpecManagerOnCurrentSpecChanged(object sender, EventArgs eventArgs)
        {
            this.RaiseCanUsingGpu(SpecManager.CurrentSpec.ForceCpuCalc ? 0 : -1);
        }
    }
}
