﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// UDD2.0から生成されるカスタムアクション設定のサブグループビューモデルです。
    /// </summary>
    public class CustomActionGroupViewModel :
        PropertyGroupViewModel<CustomActionSettingData>, IModificationFlagOwner, INamedViewModel
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="name">バインディング名</param>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public CustomActionGroupViewModel(string name, HierarchyViewModel parent, CustomActionSettingData dataModel)
            : base(parent, dataModel)
        {
            this.BindingName = name;

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// バインディング名を取得します。
        /// </summary>
        public string BindingName { get; private set; }

        /// <summary>
        /// アセットファイルの基準フォルダを取得します。
        /// </summary>
        public string UserAssetsBaseFolderPath
        {
            get
            {
                string path = OptionStore.ProjectConfig.UserAssetsBaseFolderPath;

                if (path.EndsWith("\\") == false)
                {
                    path = path + "\\";
                }

                return path;
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }
    }

    /// <summary>
    /// UDD2.0から生成されるカスタムアクション設定のカラーサブグループビューモデルです。
    /// </summary>
    public class CustomActionColorGroupViewModel :
        ColorPickerOwnerViewModel<CustomActionSettingData>, IModificationFlagOwner, INamedViewModel
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="name">バインディング名</param>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public CustomActionColorGroupViewModel(string name, HierarchyViewModel parent, CustomActionSettingData dataModel)
            : base(parent, dataModel)
        {
            this.BindingName = name;
            this.OnGetParticleLifeExecutable = new AnonymousExecutable((o) =>
            {
                var param = o as float[];
                var emitterVm = this.FindNearestParentOfType<EmitterViewModel>();
                param[0] = emitterVm.EmitterParticleViewModel.EmitterParticleLifeViewModel.GetDataModelValue<int>("Life");
            });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.IgnoreColorPickerPropertyNames);
        }

        /// <summary>
        /// バインディング名を取得します。
        /// </summary>
        public string BindingName { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// パーティクルの寿命を取得するExecutableを取得します。
        /// </summary>
        public IExecutable OnGetParticleLifeExecutable { get; private set; }
    }

    /// <summary>
    /// カスタムアクションセッティングビューモデル
    /// </summary>
    public class CustomActionUserSettingDataViewModel :
        PropertyGroupViewModel<CustomActionSettingData>, IModificationFlagOwner, IUserDataInfoExportViewModel
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "UserDataInfo",
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        /// <param name="settingIndex">セッティングインデックス</param>
        public CustomActionUserSettingDataViewModel(
            HierarchyViewModel parent,
            CustomActionSettingData dataModel,
            int settingIndex)
            : base(parent, dataModel)
        {
            this.SettingIndex = settingIndex;

            if (dataModel != null)
            {
                // サブグループのセットアップ
                UserDataSubGroupUtil.SetupSubGroups(
                    this,
                    dataModel,
                    s => this.AddChildViewModel(new CustomActionGroupViewModel(s, this, dataModel)),
                    s => this.AddChildViewModel(new CustomActionColorGroupViewModel(s, this, dataModel)));

                // 子ビューモデルのコレクションから名前で解決するカスタムバインディングを設定します。
                this.CustomBindingResolver = s =>
                {
                    return this.Children.OfType<INamedViewModel>().FirstOrDefault(c => c.BindingName == s);
                };
            }

            // Always create the modification flag view model IN THE END of the constructor,
            // in case any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// Get the index of the custom shader setting.
        /// </summary>
        public int SettingIndex { get; private set; }

        /// <summary>
        /// ユーザーデータ情報を取得します。
        /// </summary>
        public UserDataInfo UserDataInfo
        {
            get
            {
                return CustomActionUserDataManager.FindCustomActionUserDataInfo(this.SettingIndex);
            }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }
    }
}
